﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "tmipc_defines.h"
#include "tmipc_result.h"
#include <string>

#if defined( TMIPC_TARGET_WIN )
#elif defined( TMIPC_TARGET_HORIZON )
    enum
    {
        TMIPC_STACK_SIZE = 16384
    };
    #include <nn/os.h>
    using nn::os::MessageQueueType;
#else
    #error Target Platform Undefined.
#endif

//==============================================================================
namespace tmipc {
//==============================================================================

class MultiEventWait;

class Mutex
{
protected:
#if defined( TMIPC_TARGET_WIN )
        CRITICAL_SECTION    m_CriticalSection;
#elif defined( TMIPC_TARGET_HORIZON )
        nn::os::MutexType   m_Mutex;
#else
    #error Target Platform Undefined.
#endif

        bool                m_Created;

public:
                            Mutex               ();
                           ~Mutex               ();

        void                Create              ();
        void                Destroy             ();
        void                Lock                ();
        void                Unlock              ();
};

//==============================================================================

class Event
{
    friend MultiEventWait;
protected:
#if defined( TMIPC_TARGET_WIN )
        HANDLE              m_hEvent;
#elif defined( TMIPC_TARGET_HORIZON )
        nn::os::EventType   m_Event;
#else
    #error Target Platform Undefined.
#endif

        bool                m_Created;

public:
                            Event               ();
                           ~Event               ();

        void                Create              ();
        void                Destroy             ();

        void                Set                 ();
        void                Reset               ();
        s32                 Wait                ( s32       timeout );
};

//==============================================================================

class MultiEventWait
{
protected:
#if defined( TMIPC_TARGET_WIN )
        HANDLE*                         m_phEvents;
#elif defined( TMIPC_TARGET_HORIZON )
        // The following data fields were allocated based on the Nintendo file:
        // .\nintendo\Siglo\sdk\Tests\Os\Sources\Tests\MultipleWait\test_MultipleWait.cpp
        nn::os::MultiWaitType           m_MultiWait;
        nn::os::MultiWaitHolderType*    m_pWaitHolders;
#else
    #error Target Platform Undefined.
#endif
        Event**                         m_pEvents;
        s32                             m_NumberOfEvents;

private:
        void                Deallocate          ();

public:
                            MultiEventWait      ();
                            ~MultiEventWait     ();

        // A [very] common case: exactly two Events to wait for.
        s32                 SetTwoEvents        ( Event*    pEvent1,    Event*  pEvent2 );

        // Allow the user to add an arbitrary number of Events to wait for.
        s32                 AddEvent            ( Event*    pEvent );

        // Wait for all Events in the array.  Waits for "infinite" time.
        // Out: SignalledIndex: upon success, this will be the element index that was signalled.
        // returns: TMIPC_RESULT_OK upon success, otherwise will return an error code.
        s32                 Wait                ( s32&    SignalledIndex );

        // Wait for all Events in the array.
        //  In: TimeoutMS:      the amount of time to wait for (milliseconds).  Must be >= 0.
        // Out: SignalledIndex: upon success, this will be the element index that was signalled.
        // returns: TMIPC_RESULT_OK upon success, otherwise will return an error code.
        s32                 Wait                ( s32       TimeoutMS,  s32&    SignalledIndex );
};

//==============================================================================

typedef s32 ThreadFunction( void* pArg );

class Thread
{
protected:
#if defined( TMIPC_TARGET_WIN )
        HANDLE              m_hThread;
        DWORD               m_ThreadId;
#elif defined( TMIPC_TARGET_HORIZON )
        nn::os::ThreadType  m_Thread;
#else
    #error Target Platform Undefined.
#endif

public:
                            Thread              ();
                           ~Thread              ();
#if defined( TMIPC_TARGET_WIN )
        tmipc::Result       Start               ( ThreadFunction*   pFn,
                                                  void*             pArg,
                                                  s32               Priority );
#elif defined( TMIPC_TARGET_HORIZON )
        tmipc::Result       Start               ( ThreadFunction*   pFn,
                                                  void*             pArg,
                                                  void*             pStack,
                                                  size_t            StackSize,
                                                  s32               Priority,
                                                  const char*       pName );
        const nn::os::ThreadType& GetThreadType () const { return m_Thread; }
#else
    #error Target Platform Undefined.
#endif
        tmipc::Result       Join                ( s32               Timeout = TMIPC_INFINITE );

static  void                SleepMs             ( s32               Ms );
};

//==============================================================================

class ScopedLock
{
protected:
        Mutex*              m_pMutex;

public:
    explicit                ScopedLock          ( Mutex&    mutex );
                           ~ScopedLock          ();
};

//==============================================================================

class MsgQueue
{
protected:
#if defined( TMIPC_TARGET_WIN )
            HANDLE              m_hSemaphore;   // Semaphore to sleep waiting threads.
            Mutex               m_Mutex;        // Mutex to control queue access.
#elif defined( TMIPC_TARGET_HORIZON )
            Mutex               m_Mutex;        // Mutex to control m_Count.
            MessageQueueType    m_q;            // OS message queue.
#else
    #error Target Platform Undefined.
#endif

            bool            m_Created;      // true when created.
            void**          m_pQueue;       // Pointer to storage for the queue.
            bool            m_Allocated;    // true if storage was allocated.
            s32             m_Capacity;     // Capacity of the queue (same as semaphore max count).
volatile    s32             m_Count;        // Number of items currently in queue.
volatile    s32             m_iRead;        // Index to read from.
volatile    s32             m_iWrite;       // Index to write to.

public:
                            MsgQueue    ();                         // Constructor.
                           ~MsgQueue    ();                         // Destructor.

            void            Create      ( s32       Capacity );     // Create the message queue.

            void            Create      ( s32       Capacity,
                                          void**    pStorage );     // Create the message queue with the provided storage.

            void            Destroy     ();                         // Destroy the message queue.

            bool            IsValid     ();                         // Is the message queue valid for use.

            bool            Send        ( void*     pMessage,
                                          bool      Block=true );   // Send a message to the queue, can block or return success code.
            bool            Receive     ( void**    ppMessage,
                                          bool      Block=true );   // Recieve a message from the queue.
            void*           Receive     ();                         // Receive a message from the queue, always blocks until success.

            s32             GetCount    () const;                   // Get the count of messages in the queue.
};

//==============================================================================
} // namespace tmipc
//==============================================================================
