﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "tmipc_task.h"
#include "tmipc_tasklist.h"
#include "tmipc_packet.h"
#include "../DejaInsight.h"
#include "../performance_monitor/performance_monitor_service.h"

//==============================================================================
namespace tmipc {
//==============================================================================

TaskList::TaskList()
{
}

//==============================================================================

TaskList::~TaskList()
{
}

//==============================================================================

void TaskList::Init()
{
    DEJA_TRACE( "TaskList::Init", "Init TaskList" );

    m_Mutex.Create();
}

//==============================================================================

void TaskList::Kill()
{
    DEJA_TRACE( "TaskList::Kill", "Kill TaskList" );

    m_Mutex.Destroy();
}


//==============================================================================

void TaskList::AddTask( Task* pTask )
{
    DEJA_TRACE( "TaskList::AddTask", "AddTask" );

    ScopedLock lock( m_Mutex );

    ASSERT( FindTask( pTask->GetTaskId() ) == nullptr );

    s32 Priority = pTask->GetPriority();
    ASSERT( (Priority >= 0) && (Priority < TMIPC_NUM_PRIORITIES) );

    m_Tasks[Priority].AddTail( pTask );
}

//==============================================================================

void TaskList::RemoveTask( Task* pTask )
{
    DEJA_TRACE( "TaskList::RemoveTask", "RemoveTask" );

    ScopedLock lock( m_Mutex );

    for( s32 i=0; i<TMIPC_NUM_PRIORITIES; i++ )
    {
        List<Task>& Tasks = m_Tasks[i];
        Task* pScan = Tasks.GetHead();
        while( pScan )
        {
            if( pScan == pTask )
            {
                Tasks.Remove( pTask );
                return;
            }
            pScan = pScan->m_pNext;
        }
    }

    // Task not found.
    ASSERT( 0 );
}

//==============================================================================

void TaskList::CancelTask( u32 TaskId )
{
    DEJA_TRACE( "TaskList::CancelTask", "CancelTask" );

    ScopedLock lock( m_Mutex );

    for( s32 i=0; i<TMIPC_NUM_PRIORITIES; i++ )
    {
        List<Task>& Tasks = m_Tasks[i];
        Task* pScan = Tasks.GetHead();
        while( pScan )
        {
            if( pScan->GetTaskId() == TaskId )
            {
                pScan->Cancel();
                return;
            }
            pScan = pScan->m_pNext;
        }
    }
}

//==============================================================================

void TaskList::CleanupFinishedTasks()
{
    ScopedLock lock( m_Mutex );

    // Loop over the priorities.
    for( s32 i=0; i<TMIPC_NUM_PRIORITIES; i++ )
    {
        // Loop over each task at this priority.
        List<Task>& Tasks = m_Tasks[i];
        Task* pScan = Tasks.GetHead();
        while( pScan )
        {
            Task* pNext = pScan->m_pNext;

            // Is it complete or canceled?
            s32 Status = pScan->GetStatus();
            switch( Status )
            {
            case Task::StatusComplete:
            case Task::StatusCanceled:
                {
                    // Remove this task.
                    RemoveTask( pScan );

                    // Free the task?
                    if( pScan->m_bExternallyOwned )
                    {
                        //TMA_TRACE( ">>>", "TaskList set event" );
                        pScan->m_Event.Set();
                    }
                    else
                    {
                        //TMA_TRACE( ">>>", "TaskList Delete Task" );
                        pScan->~Task();
                        tma::s_Deallocate( pScan, sizeof( Task ) );
                    }

#if ENABLE_PERFORMANCE_MONITOR_SERVICE
                    // Record how this task was marked as "finished".
                    if( pScan->GetStatus() == Task::StatusComplete )
                    {
                        // +1 completed task.
                        ADD_TASK_COUNT_COMPLETED_INCREMENT( 1 );
                    }
                    else
                    {
                        // +1 cancelled task.
                        ADD_TASK_COUNT_CANCELLED_INCREMENT( 1 );
                    }
#endif // ENABLE_PERFORMANCE_MONITOR_SERVICE
                }
                break;

            default:
                {
                    // Stop processing if any other status.
                    pNext = nullptr;
                }
                break;
            }

            pScan = pNext;
        }
    }
}

//==============================================================================

bool TaskList::ProcessSendPacket( bool IsConnected, Packet* pPacket)
{
    Task* pTask{ nullptr };

    ScopedLock lock( m_Mutex );

    // Loop over the priorities.
    for( s32 i=0; i<TMIPC_NUM_PRIORITIES; i++ )
    {
        // Loop over each task at this priority.
        List<Task>& Tasks = m_Tasks[i];
        Task* pScan = Tasks.GetHead();
        while( pScan )
        {
            //TMA_TRACE( ">>>", "Tick %p", pScan->GetTaskId() );

            Task* pNext = pScan->m_pNext;

            // Is it complete or canceled?
            s32 Status = pScan->GetStatus();
            switch( Status )
            {
            case Task::StatusComplete:
            case Task::StatusCanceled:
                {
                    // Remove this task.
                    RemoveTask( pScan );

                    // Free the task?
                    if( pScan->m_bExternallyOwned )
                    {
                        //TMA_TRACE( ">>>", "TaskList set event" );
                        pScan->m_Event.Set();
                    }
                    else
                    {
                        //TMA_TRACE( ">>>", "TaskList Delete Task" );
                        pScan->~Task();
                        tma::s_Deallocate( pScan, sizeof( Task ) );
                    }

#if ENABLE_PERFORMANCE_MONITOR_SERVICE
                    // Record how this task was marked as "finished".
                    if( pScan->GetStatus() == Task::StatusComplete )
                    {
                        // +1 completed task.
                        ADD_TASK_COUNT_COMPLETED_INCREMENT( 1 );
                    }
                    else
                    {
                        // +1 cancelled task.
                        ADD_TASK_COUNT_CANCELLED_INCREMENT( 1 );
                    }
#endif // ENABLE_PERFORMANCE_MONITOR_SERVICE
                }
                break;
            default:
                {
                    // Does it need packets?
                    if( !pTask && pScan->GetNeedPackets() )
                    {
                        // Found the task.
                        const bool bServiceAlwaysSendsPacket =
                        (
                            pScan->GetServiceId() == ServiceId_NodeTICS_BeaconQuery
                         || pScan->GetServiceId() == ServiceId_NodeTICS_TMSInfo
                         || pScan->GetServiceId() == ServiceId_NodeUSB_BeaconQuery
                         || pScan->GetServiceId() == ServiceId_NodeUSB_TMSInfo
                         || pScan->GetServiceId() == ServiceId_NodeUSB_ConnectHandshake
                         || pScan->GetServiceId() == ServiceId_NodeUSB_Disconnect
                         || pScan->GetServiceId() == ServiceId_DebugService
                        );
                        bool bCanSendIt{ bServiceAlwaysSendsPacket || IsConnected };
                        if( bCanSendIt )
                        {
                            pTask = pScan;
                        }
//else
//{
//TMA_POWER_TEST_PRINT( "[%s]!!! Task: 0x%p, Service: 0x%08X, Id: %d - Did *not* send it.\n", _BestFunctionName_, pScan, pScan->GetServiceId(), pScan->GetTaskId() );
//}
                    }
                }
                break;
            }

            pScan = pNext;
        }
    }

    // Allow the task to send data, if one was found.
    if( pTask )
    {
        // Initialize the packet and ask the task to fill it.
        pPacket->SetInitiate( 0 );
        pPacket->SetServiceId( pTask->GetServiceId() );
        pPacket->SetTaskId( pTask->GetTaskId() );
        pPacket->SetTaskType( pTask->GetTaskType() );
        pPacket->ResetCursor();
        pTask->OnSendPacket( pPacket );
    }

    return pTask != nullptr;
}

//==============================================================================

Task* TaskList::FindTask( u32 TaskId ) const
{
    DEJA_TRACE( "Task::FindTask", "FindTask %p", TaskId );

    for( s32 i=0; i<TMIPC_NUM_PRIORITIES; i++ )
    {
        const List<Task>& Tasks = m_Tasks[i];
        const Task* pScan = Tasks.GetHead();
        while( pScan )
        {
            if( pScan->GetTaskId() == TaskId )
            {
                return( const_cast<Task*>(pScan) );
            }
            pScan = pScan->m_pNext;
        }
    }

    // Task not found.
    return( NULL );
}

//==============================================================================

void TaskList::ProcessRecvPacket( Packet*   pPacket )
{
    DEJA_TRACE( "TaskList::ProcessRecvPacket", "FindTask %p", TaskId );

    ScopedLock lock( m_Mutex );

    // Find the Task for this packet and deliver it.
    u32 TaskId = pPacket->GetTaskId();
    Task* pTask = FindTask( TaskId );
    if( pTask )
    {
        pTask->OnRecvPacket( pPacket );
    }
    else
    {
        DEJA_ERROR( "TaskList::ProcessRecvPacket", "ProcessRecvPacket - Unable to find TaskID = %p", pPacket->GetTaskId() );
    }
}

//==============================================================================
// Moves tasks from the "active/live list" to the "sleeping list".
// Exception: Tasks that are tagged as service_task::CanPutToSleep are left in the main list.
void TaskList::GoToSleep()
{
TMA_POWER_TEST_PRINT( "[%s]!!! - Start\n", _BestFunctionName_ );
    ScopedLock Lock( m_Mutex );

    // Loop over the priorities.
    for( s32 i = 0; i < TMIPC_NUM_PRIORITIES; i++ )
    {
        // Loop over each task at this priority.
        List<Task>& SourceList      = m_Tasks[i];
        List<Task>& DestinationList = m_SleepingTasks[i];
        Task* pTask = SourceList.GetHead();
        while( pTask != nullptr )
        {
            Task* pNext{ pTask->m_pNext };

            // Remove the current Task from the "source TaskList".
            if( pTask->CanBePutToSleep() )
            {
TMA_POWER_TEST_PRINT( "[%s]!!! - Going to sleep: %p; Service Id: 0x%08X; Task Id: %d\n", _BestFunctionName_, pTask, pTask->GetServiceId(), pTask->GetTaskId() );
//pTask->OnDumpDebugInfo();
                SourceList.Remove( pTask );

                // Add the current task from the "source TaskList" to this TaskList.
                DestinationList.AddTail( pTask );
            }

            // Retrieve the next item from the source list.
            pTask = pNext;
        }
    }

TMA_POWER_TEST_PRINT( "[%s]!!! - Finish\n", _BestFunctionName_ );
}

//==============================================================================
// Moves tasks from the "sleeping list" to the "active/live list".
void TaskList::WakeUp()
{
TMA_POWER_TEST_PRINT( "[%s]!!! - Start\n", _BestFunctionName_ );
    ScopedLock Lock( m_Mutex );

    // Loop over the priorities.
    for( s32 i = 0; i < TMIPC_NUM_PRIORITIES; i++ )
    {
        // Loop over each task at this priority.
        List<Task>& SourceList      = m_SleepingTasks[i];
        List<Task>& DestinationList = m_Tasks[i];
        Task* pTask = SourceList.GetHead();
        while( pTask != nullptr )
        {
            // Preserve Task ordering.
            TMA_POWER_TEST_PRINT( "[%s]!!! - Task is waking up: %p; Service Id: 0x%08X; Task Id: %d\n", _BestFunctionName_, pTask, pTask->GetServiceId(), pTask->GetTaskId() );
//pTask->OnDumpDebugInfo();

            // Remove the current Task from the "source TaskList".
            SourceList.Remove( pTask );

            // Add the current task from the "source TaskList" to this TaskList.
            DestinationList.AddTail( pTask );

            // Retrieve the next item from the source list.
            pTask = SourceList.GetHead();
        }
    }

TMA_POWER_TEST_PRINT( "[%s]!!! - Finish\n", _BestFunctionName_ );
}

//==============================================================================

s32 TaskList::GetTaskCount() const
{
    s32 TaskCount{ 0 };
    ScopedLock ThisMutex{ const_cast<Mutex &>(m_Mutex) };

    // Loop over the priorities.
    for( s32 i=0; i<TMIPC_NUM_PRIORITIES; i++ )
    {
        // Loop over each task at this priority.
        TaskCount += m_Tasks[i].GetSize();
    }

    return TaskCount;
}

//==============================================================================

s32 TaskList::GetSleepingTaskCount() const
{
    s32 TaskCount{ 0 };
    ScopedLock ThisMutex{ const_cast<Mutex &>(m_Mutex) };

    // Loop over the priorities.
    for( s32 i=0; i<TMIPC_NUM_PRIORITIES; i++ )
    {
        // Loop over each task at this priority.
        TaskCount += m_SleepingTasks[i].GetSize();
    }

    return TaskCount;
}

//==============================================================================

void TaskList::CancelAll()
{
    DEJA_TRACE( "Task::CancelAll", "CancelAll" );
TMA_POWER_TEST_PRINT( "[%s]!!! - Start\n", _BestFunctionName_ );

    ScopedLock lock( m_Mutex );

    // Loop over the priorities.
    for( s32 i=0; i<TMIPC_NUM_PRIORITIES; i++ )
    {
        // Loop over each task at this priority.
        List<Task>& Tasks = m_Tasks[i];
        Task* pScan = Tasks.GetHead();
        while( pScan )
        {
            Task* pNext = pScan->m_pNext;
TMA_POWER_TEST_PRINT( "[%s]!!! - Cancel task: %p; Id: %d\n", _BestFunctionName_, pScan, pScan->GetTaskId() );
            pScan->Cancel();
            pScan = pNext;
        }
    }
TMA_POWER_TEST_PRINT( "[%s]!!! - Finish\n", _BestFunctionName_ );
}

//==============================================================================

bool TaskList::IsTaskIdFree( u32 TaskId )
{
    DEJA_TRACE( "Task::IsTaskIdFree", "IsTaskIdFree" );

    ScopedLock lock( m_Mutex );

    // Non-sleeping tasks.
    {
        // Loop over the priorities.
        for( s32 i=0; i<TMIPC_NUM_PRIORITIES; i++ )
        {
            // Loop over each task at this priority.
            List<Task>& Tasks = m_Tasks[i];
            Task* pTask = Tasks.GetHead();
            while( pTask )
            {
                Task* pNext = pTask->m_pNext;
                if( pTask->GetTaskId() == TaskId )
                {
                    return( false );
                }

                pTask = pNext;
            }
        }
    }

    // Sleeping tasks.
    {
        // Loop over the priorities.
        for( s32 i=0; i<TMIPC_NUM_PRIORITIES; i++ )
        {
            // Loop over each task at this priority.
            List<Task>& Tasks = m_SleepingTasks[i];
            Task* pTask = Tasks.GetHead();
            while( pTask )
            {
                Task* pNext = pTask->m_pNext;
                if( pTask->GetTaskId() == TaskId )
                {
                    return( false );
                }

                pTask = pNext;
            }
        }
    }

    // Must be free.
    return( true );
}

//==============================================================================

void TaskList::PrintTaskInfo()
{
    NN_SDK_LOG( "[tma][tasklist] - Start\n" );
    ScopedLock Lock( m_Mutex );

    // Non-sleeping tasks.
    {
        NN_SDK_LOG( "[tma][tasklist] - Active task count: %d\n", GetTaskCount() );
        // Loop over the priorities.
        for( s32 i = 0; i < TMIPC_NUM_PRIORITIES; i++ )
        {
            // Loop over each task at this priority.
            List<Task>& SourceList = m_Tasks[i];
            Task* pTask = SourceList.GetHead();
            while( pTask != nullptr )
            {
                Task* pNext{ pTask->m_pNext };

                NN_SDK_LOG( "[tma][tasklist]   Task: %p; Service Id: 0x%08X; Task Id: %d\n", pTask, pTask->GetServiceId(), pTask->GetTaskId() );
                pTask->OnDumpDebugInfo();

                // Retrieve the next item from the source list.
                pTask = pNext;
            }
        }
    }

    // Sleeping tasks.
    {
        NN_SDK_LOG( "[tma][tasklist] - Sleeping task count: %d\n", GetSleepingTaskCount() );
        // Loop over the priorities.
        for( s32 i = 0; i < TMIPC_NUM_PRIORITIES; i++ )
        {
            // Loop over each task at this priority.
            List<Task>& SourceList = m_SleepingTasks[i];
            Task* pTask = SourceList.GetHead();
            while( pTask != nullptr )
            {
                Task* pNext{ pTask->m_pNext };

                NN_SDK_LOG( "[tma][tasklist] Task: %p; Service Id: 0x%08X; Task Id: %d\n", pTask, pTask->GetServiceId(), pTask->GetTaskId() );
                pTask->OnDumpDebugInfo();

                // Retrieve the next item from the source list.
                pTask = pNext;
            }
        }
    }

    NN_SDK_LOG( "[tma][tasklist] - Finish\n" );
}

//==============================================================================
} // namespace tmipc
//==============================================================================
