﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include "tmipc_services_manager.h"
#include "tmipc_service.h"
#include "tmipc_result.h"
#include "tmipc_thread.h"
#include "tmipc_packet.h"
#include "tmipc_node.h"
#include "tmipc_task.h"
#include "../DejaInsight.h"
#include "../Version.h"
#include "../thread_tracker.h"
#include "../performance_monitor/performance_monitor_service.h"

// --------------------------------------------------------------------------
namespace tmipc {
// --------------------------------------------------------------------------

struct WorkItem
{
    enum opcode
    {
        OpNull,
        OpNewTask,
        OpFreeTask,
        OpRecvPacket,
        OpTick,
        OpDisconnected,
        OpDeleteServices,
        OpSleep,
    };

    s32             m_OpCode;
    Task*           m_pTask;
    Packet*         m_pPacket;
};

// --------------------------------------------------------------------------

ServicesManager::ServicesManager()
:   m_Initialized       ( false )
,   m_pFirstService     ( nullptr )
,   m_NextTaskId        ( 0 )
,   m_pWorkThreadStack  ( nullptr )
{
    DEJA_TRACE( "ServicesManager::ServicesManager", "ServicesManager::ServicesManager" );

    m_Lock.Create();

    // Create packet queues.
    m_FreeSendPacketQ.Create( TMIPC_NUM_SEND_PACKETS );
    m_FreeRecvPacketQ.Create( TMIPC_NUM_RECV_PACKETS );

    // Setup the send packets.
    for( s32 i=0; i<TMIPC_NUM_SEND_PACKETS; i++ )
    {
        void* pMem = tma::s_Allocate( sizeof( Packet ) );
        Packet* pPacket = new (pMem) Packet;
        pPacket->m_pFreeQ = &m_FreeSendPacketQ;
        m_FreeSendPacketQ.Send( pPacket );
    }

    // Setup the receive packets.
    for( s32 i=0; i<TMIPC_NUM_RECV_PACKETS; i++ )
    {
        void* pMem = tma::s_Allocate( sizeof( Packet ) );
        Packet* pPacket = new (pMem) Packet;
        pPacket->m_pFreeQ = &m_FreeRecvPacketQ;
        m_FreeRecvPacketQ.Send( pPacket );
    }

    // Initialize the Thread's Priority.
    m_ThreadPriority = nn::os::InvalidThreadPriority;

    // Create work queues.
    m_WorkFreeQ.Create( TMIPC_WORK_QUEUE_DEPTH, m_WorkFreeQueueStorage );
    m_WorkQ    .Create( TMIPC_WORK_QUEUE_DEPTH, m_WorkQueueStorage     );

    // Used for proper Task ID generation.
    m_NodeId = TMIPC_SERVER_NODE_ID;

    m_IsSleeping    = false;
    m_bPreserveStateThroughSleepWake = false;
    m_pNode         = nullptr;

    // Setup the work items.
    for( s32 i=0; i<TMIPC_WORK_QUEUE_DEPTH; i++ )
    {
        void* pMem = tma::s_Allocate( sizeof( WorkItem ) );
        WorkItem* pItem = new (pMem) WorkItem;
        m_WorkFreeQ.Send( pItem );
    }

    // Create the disconnect event.
    m_DisconnectEvent.Create();
    // Create the Wake/Sleep events.
    m_WakeUpEvent.Create();
    m_SleepEvent.Create();
}

// --------------------------------------------------------------------------

ServicesManager::~ServicesManager()
{
    ASSERT( !m_Initialized );

    // Destroy all the packets.
    Packet* pPacket = nullptr;
    while( m_FreeSendPacketQ.Receive( (void**)&pPacket, false ) )
    {
        pPacket->~Packet();
        tma::s_Deallocate( pPacket, sizeof( Packet ) );
        pPacket = nullptr;
    }
    while( m_FreeRecvPacketQ.Receive( (void**)&pPacket, false ) )
    {
        pPacket->~Packet();
        tma::s_Deallocate( pPacket, sizeof( Packet ) );
        pPacket = nullptr;
    }

    // Destory packet queues.
    m_FreeSendPacketQ.Destroy();
    m_FreeRecvPacketQ.Destroy();

    // Destroy all the work items.
    WorkItem* pItem = nullptr;
    while( m_WorkQ.Receive( (void**)&pItem, false ) )
    {
        pItem->~WorkItem();
        tma::s_Deallocate( pItem, sizeof( WorkItem ) );
        pItem = nullptr;
    }
    while( m_WorkFreeQ.Receive( (void**)&pItem, false ) )
    {
        pItem->~WorkItem();
        tma::s_Deallocate( pItem, sizeof( WorkItem ) );
        pItem = nullptr;
    }

    // Destroy work queues.
    m_WorkFreeQ.Destroy();
    m_WorkQ    .Destroy();

    m_Lock.Destroy();

    if( m_pWorkThreadStack )
    {
        tma::s_Deallocate( m_pWorkThreadStack, m_WorkThreadStackSize + (nn::os::StackRegionAlignment - 1) );
        m_pWorkThreadStack = nullptr;
#if defined( TMIPC_TARGET_HORIZON )
        tma::ThreadTracker::UnregisterThread( tma::ThreadTracker::ThreadId::ServicesManager );
#endif
    }

    m_DisconnectEvent.Destroy();
    m_WakeUpEvent.Destroy();
    m_SleepEvent.Destroy();
}

// --------------------------------------------------------------------------

void ServicesManager::Init()
{
    ASSERT( !m_Initialized );

    // Setup the TaskList.
    m_TaskList.Init();

    // Initialized.  This needs to occur before creating the WorkerThread, or
    // the WorkerThread will ASSERT( pThis->m_Initialized ).
    m_Initialized = true;

    // Start the worker thread.
#if defined( TMIPC_TARGET_WIN )
    m_WorkThread.Start( WorkThread, this, THREAD_PRIORITY_NORMAL );
#elif defined( TMIPC_TARGET_HORIZON )
    if( !m_pWorkThreadStack )
    {
        m_WorkThreadStackSize = ((TMIPC_STACK_SIZE + (nn::os::StackRegionAlignment - 1)) & ~(nn::os::StackRegionAlignment - 1));
        m_pWorkThreadStack = tma::s_Allocate( m_WorkThreadStackSize + (nn::os::StackRegionAlignment - 1) );
        ASSERT( m_pWorkThreadStack );
    }
    NN_SDK_ASSERT(m_ThreadPriority == NN_SYSTEM_THREAD_PRIORITY(tma, ServicesMgr));
    m_WorkThread.Start( WorkThread, this,
        (void*)(((u64)m_pWorkThreadStack - 1 + nn::os::StackRegionAlignment) & ~(nn::os::StackRegionAlignment - 1)),
        m_WorkThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(tma, ServicesMgr), NN_SYSTEM_THREAD_NAME(tma, ServicesMgr) );
    tma::ThreadTracker::RegisterThread( m_WorkThread.GetThreadType(), tma::ThreadTracker::ThreadId::ServicesManager );
#else
    #error Target Platform Undefined.
#endif
}

// --------------------------------------------------------------------------

void ServicesManager::Kill()
{
    if( m_Initialized )
    {
        if( m_pNode != nullptr )
        {
            if( m_pNode->IsConnected() )
            {
                m_pNode->Disconnect();
            }
        }

        // Send a terminate message to the work queue.
        WorkItem* pItem = nullptr;
        m_WorkQ.Send( pItem );

        // Wait for termination to complete.
        m_WorkThread.Join();

        // Teardown the TaskList.
        m_TaskList.Kill();

        // This seems counter intuitive, but this is not going to leak
        // memory, because the actual pointers are stored external to
        // ServicesManager (in tmagent.cpp).
        Service *pNext = nullptr;
        for( Service* pService = m_pFirstService; pService != nullptr; pService = pNext )
        {
            pNext = m_pFirstService->m_pNext;
            if (pNext != nullptr)
            {
                pNext->SetServicesManager( nullptr );
            }
            m_pFirstService->m_pNext = nullptr;
        }
        m_pFirstService = nullptr;

        // No longer initialized.
        m_Initialized = false;
    }
}

// --------------------------------------------------------------------------

void ServicesManager::DiagnosticReport()
{
    // TODO:  Write diagnostic messages to the console.
    NN_SDK_LOG( "[tma] - Diagnostic report start -\n");
    m_TaskList.PrintTaskInfo();
    NN_SDK_LOG( "[tma] - Diagnostic report end -\n");
}

// --------------------------------------------------------------------------

void ServicesManager::WakeUp( Node* pNode )
{
    ASSERT( pNode != nullptr );
    ASSERT( m_pNode == nullptr );
    if( IsSleeping() )
    {
        m_pNode = pNode;
        m_pNode->SetWasWokenUp( PreserveStateThroughSleepWake() );
        m_pNode->SetServicesManager( this );
        // Signal the "blocking call", so the WorkThread can resume.
        m_WakeUpEvent.Set();
    }
}

// --------------------------------------------------------------------------
// This is a blocking call, until the WorkThread processes the "Sleep request".
void ServicesManager::GoToSleep()
{
    ASSERT( m_Initialized );

    if( !IsSleeping() )
    {
        // Reset the Sleep and WakeUp events, so they can be waited on.
        m_SleepEvent.Reset();
        m_WakeUpEvent.Reset();

        // Cancel all tasks in the list.
        WorkItem* pItem = (WorkItem*)m_WorkFreeQ.Receive();
        pItem->m_OpCode = WorkItem::OpSleep;
        m_WorkQ.Send( pItem );
        m_SleepEvent.Wait( TMIPC_INFINITE );
        m_IsSleeping = true;
    }
}

// --------------------------------------------------------------------------

bool ServicesManager::IsSleeping() const
{
    return m_IsSleeping;
}

// --------------------------------------------------------------------------

bool ServicesManager::IsConnected() const
{
    const bool bIsConnected{ (m_pNode != nullptr) && m_pNode->IsConnected() };
    return bIsConnected;
}

// --------------------------------------------------------------------------
// Returns true if the node was connected, and the connected Target Manager supports state preservation.
bool ServicesManager::PreserveStateThroughSleepWake() const
{
    return m_bPreserveStateThroughSleepWake;
}

// --------------------------------------------------------------------------

void ServicesManager::ResetPreserveStateThroughSleepWake()
{
    m_bPreserveStateThroughSleepWake = false;
}

// --------------------------------------------------------------------------

tmipc::Result ServicesManager::AddService( Service* pService )
{
    DEJA_TRACE( "ServicesManager::AddService", "AddService %d", pService->GetId() );
    ScopedLock lock( m_Lock );

    // Check if this service already exists?
    if( GetService( pService->GetId() ) )
    {
        // Service is already in the list.
        ASSERT( 0 );
        return TMIPC_RESULT_DUPLICATE_SERVICE;
    }

    // Add to service list.
    pService->m_pNext   = m_pFirstService;
    m_pFirstService     = pService;

    // Set the ServicesManager pointer in the service.
    pService->SetServicesManager( this );

    // Added.
    return TMIPC_RESULT_OK;
}

// --------------------------------------------------------------------------

Service* ServicesManager::GetService( u32 ServiceId )
{
    DEJA_TRACE( "ServicesManager::GetService", "GetService %d", ServiceId );
    ScopedLock lock( m_Lock );

    // Loop over the services.
    Service* pTest = m_pFirstService;
    while( pTest )
    {
        // Found it?
        if( pTest->GetId() == ServiceId )
        {
            return pTest;
        }

        // Try next service.
        pTest = pTest->m_pNext;
    }

    // Not found.
    return nullptr;
}

// --------------------------------------------------------------------------

u32 ServicesManager::AllocTaskId()
{
// Do *not* lock this out here, because it can lead to a nested mutex lock hang.
//    ScopedLock lock( m_Lock );

    u32 Id;
    for( ;; )
    {
        // Generate a new Id.
        // (Keep the m_Lock locked for the briefest amount of time.)
        {
            ScopedLock lock( m_Lock );

            Id = (m_NodeId << 16) | (m_NextTaskId & 0xffff);
            ++m_NextTaskId;
        }

        // Check if the Id is unused.
        if( m_TaskList.IsTaskIdFree( Id ) )
        {
            break;
        }
    }

    return Id;
}

// --------------------------------------------------------------------------

Packet* ServicesManager::AllocSendPacket()
{
    DEJA_TRACE( "ServicesManager::AllocSendPacket", "AllocSendPacket %d", m_FreeSendPacketQ.GetCount() );
    DEJA_CONTEXT( "ServicesManager::AllocSendPacket" );

    ASSERT( m_Initialized );

    Packet* pPacket = (Packet*)m_FreeSendPacketQ.Receive();
    pPacket->ResetCursor();
    pPacket->m_pHeader->m_DataLen = 0;

    return pPacket;
}

// --------------------------------------------------------------------------

Packet* ServicesManager::AllocRecvPacket()
{
    DEJA_TRACE( "ServicesManager::AllocRecvPacket", "AllocRecvPacket %d", m_FreeRecvPacketQ.GetCount() );

    ASSERT( m_Initialized );

    Packet* pPacket = (Packet*)m_FreeRecvPacketQ.Receive();
    pPacket->ResetCursor();

    return pPacket;
}

// --------------------------------------------------------------------------

void ServicesManager::FreePacket( Packet* pPacket )
{
    DEJA_TRACE( "ServicesManager::FreePacket", "FreePacket" );

    ASSERT( m_Initialized );
    if( pPacket != nullptr )
    {
        pPacket->m_pFreeQ->Send( pPacket );
    }
}

// --------------------------------------------------------------------------

Result ServicesManager::SubmitTask( Task* pTask, Packet* pPacket )
{
    //TMA_TRACE( "ServicesManager", "SubmitTask %p", pTask->GetTaskId() );

    ASSERT( m_Initialized );

    WorkItem* pItem = (WorkItem*)m_WorkFreeQ.Receive();
    pItem->m_OpCode  = WorkItem::OpNewTask;
    pItem->m_pTask   = pTask;
    pItem->m_pPacket = pPacket;
    m_WorkQ.Send( pItem );

    // +1: "This" Work Item is a least one, but the m_WorkQ can be consumed before the code gets here.
    ADD_WORK_QUEUE_SUBMIT_INCREMENT( m_WorkQ.GetCount() + 1 );

    return TMIPC_RESULT_OK;
}

// --------------------------------------------------------------------------

Result ServicesManager::FreeTask( Task* pTask )
{
    ASSERT( m_Initialized );

    WorkItem* pItem = (WorkItem*)m_WorkFreeQ.Receive();
    pItem->m_OpCode  = WorkItem::OpFreeTask;
    pItem->m_pTask   = pTask;
    m_WorkQ.Send( pItem );

    return TMIPC_RESULT_OK;
}

// --------------------------------------------------------------------------

void ServicesManager::CancelTask( u32 TaskId )
{
    if( m_Initialized )
    {
        m_TaskList.CancelTask( TaskId );
    }
}

// --------------------------------------------------------------------------

void ServicesManager::CancelAllTasks()
{
    ScopedLock Lock( m_Lock );
    m_TaskList.CancelAll();
}

// --------------------------------------------------------------------------

void ServicesManager::Tick()
{
    ASSERT( m_Initialized );

    if( m_WorkQ.GetCount() == 0 )
    {
        WorkItem* pItem = nullptr;
        if( m_WorkFreeQ.Receive( (void**)&pItem, false ) )
        {
            pItem->m_OpCode = WorkItem::OpTick;
            m_WorkQ.Send( pItem );
        }
    }
}

// --------------------------------------------------------------------------

void ServicesManager::Disconnected()
{
    ASSERT( m_Initialized );
//TODO: This hack can be prevented, if ServicesManager owned the m_pNode...
    if( !IsSleeping() )
    {
        m_DisconnectEvent.Reset();

        // Cancel all tasks in the list.
        WorkItem* pItem = (WorkItem*)m_WorkFreeQ.Receive();
        pItem->m_OpCode = WorkItem::OpDisconnected;
        m_WorkQ.Send( pItem );

        m_DisconnectEvent.Wait( 0 );
    }
}

// --------------------------------------------------------------------------

void ServicesManager::RemoveServices()
{
    ASSERT( m_Initialized );

    // Cancel all tasks in the list.
    WorkItem* pItem = (WorkItem*)m_WorkFreeQ.Receive();
    pItem->m_OpCode = WorkItem::OpDeleteServices;
    m_WorkQ.Send( pItem );
}

// --------------------------------------------------------------------------

void ServicesManager::SetThreadPriority( s32 Priority )
{
    m_ThreadPriority = Priority;
}

// --------------------------------------------------------------------------

s32 ServicesManager::GetThreadPriority()
{
    return m_ThreadPriority;
}

// --------------------------------------------------------------------------

void ServicesManager::SetNode( Node* pNode )
{
    m_pNode = pNode;
}

// --------------------------------------------------------------------------

Result ServicesManager::Send( Packet* pPacket )
{
    Result Result = TMIPC_RESULT_FAILED;
    ASSERT( m_pNode != nullptr );

    if( m_pNode != nullptr )
    {
        Result = m_pNode->Send( pPacket );
    }

    return Result;
}

// --------------------------------------------------------------------------

void ServicesManager::ProcessReceived( Packet* pPacket )
{
    // Dispatch the packet as a Work Item.
    WorkItem* pItem = (WorkItem*)m_WorkFreeQ.Receive();
    pItem->m_OpCode = WorkItem::OpRecvPacket;
    pItem->m_pPacket = pPacket;
    m_WorkQ.Send( pItem );
}

// --------------------------------------------------------------------------
// Returns the number of Packets that have been allocated for: Sending and Receiving.
void ServicesManager::GetAllocatedPacketCounts( s32& Sending, s32& Receiving ) const
{
    Sending   = TMIPC_NUM_SEND_PACKETS - m_FreeSendPacketQ.GetCount();
    Receiving = TMIPC_NUM_RECV_PACKETS - m_FreeRecvPacketQ.GetCount();
}

// --------------------------------------------------------------------------
// Returns the number of outstanding Tasks (not completed and not cancelled).
void ServicesManager::GetCurrentTaskCount( s32& Count ) const
{
    Count = m_TaskList.GetTaskCount();
}

// --------------------------------------------------------------------------
// Returns the number of outstanding m_WorkQ items.
void ServicesManager::GetCurrentWorkQueueCount( s32& Count ) const
{
    Count = m_WorkQ.GetCount();
}

// --------------------------------------------------------------------------

s32 ServicesManager::WorkThread( void* pArg )
{
    DEJA_TRACE( "ServicesManager::WorkThread", "WorkThread Started" );

    ServicesManager* pThis = (ServicesManager*)pArg;
    ASSERT( pThis->m_Initialized );

    bool bExiting = false;

    for( ;; )
    {
        // Read the next work item.
        WorkItem* pItem = nullptr;
        pThis->m_WorkQ.Receive( (void**)&pItem, !bExiting );

        // nullptr means exit.
        if( pItem == nullptr )
        {
            // TODO: Stop any tasks being added to the system.

            // Cancel all existing tasks.
            pThis->m_TaskList.CancelAll();

            // Set flag for exiting.
            bExiting = true;

            break;
        }
        else
        {
            // Switch on the work item opcode.
            switch( pItem->m_OpCode )
            {
            case WorkItem::OpNewTask:
                pThis->ProcessNewTask( pItem );
                break;

            case WorkItem::OpFreeTask:
                pThis->ProcessFreeTask( pItem );
                break;

            case WorkItem::OpRecvPacket:
                pThis->ProcessRecvPacket( pItem );
                break;

            case WorkItem::OpTick:
                // This is just a kick from another thread to call TickTasks() below.
                break;

            case WorkItem::OpDisconnected:
                pThis->ProcessDisconnected();
                break;

            case WorkItem::OpDeleteServices:
                pThis->ProcessDeleteServices();
                break;

            case WorkItem::OpSleep:
                // This will not return until woken up.
                pThis->ProcessSleep();
                break;

            default:
                // Unknown OpCode.
                ASSERT( 0 );
                break;
            }

            // Put it back on the work item free queue.
            pThis->m_WorkFreeQ.Send( pItem );
        }

        // Tick the Tasks to fill any free packets.
        pThis->ProcessTick();
    }

    // Exit the thread.
    return 0;
}

// --------------------------------------------------------------------------

void ServicesManager::ProcessNewTask( WorkItem* pItem )
{
    //DEJA_TRACE( "ServicesManager::ProcessNewTask", "ProcessNewTask" );
    DEJA_CONTEXT( "ServicesManager::ProcessNewTask" );

    // Add the task to the task list.
    Task* pTask = pItem->m_pTask;
    ASSERT( pTask );
    const u32 TaskId{ pTask->GetTaskId() };
    (void)TaskId;
    m_TaskList.AddTask( pTask );

    //TMA_TRACE( "ServicesManager", "ProcessNewTask TaskId = %p", pTask->GetTaskId() );

    if( (m_pNode != nullptr) && m_pNode->IsConnected() )
    {
        DEJA_TRACE( "ServicesManager::ProcessNewTask", "ProcessNewTask - Added %p", TaskId );

        // Send the first packet.
        Packet* pPacket = pItem->m_pPacket;
        if( pPacket )
        {
            Send( pPacket );
        }
    }
    else
    {
        DEJA_TRACE( "ServicesManager::ProcessNewTask", "ProcessNewTask - Canceled %p", TaskId );

        pItem->m_pTask->Cancel();
        FreePacket( pItem->m_pPacket );
    }
}

// --------------------------------------------------------------------------

void ServicesManager::ProcessFreeTask( WorkItem* pItem )
{
    DEJA_TRACE( "ServicesManager::ProcessFreeTask", "ProcessFreeTask %p", pItem->m_pTask->GetTaskId() );
    //TMA_TRACE( "ServicesManager", "ProcessFreeTask" );
    DEJA_CONTEXT( "ServicesManager::ProcessFreeTask" );

    Task* pTask = pItem->m_pTask;
    ASSERT( pTask );
    pTask->~Task();
    tma::s_Deallocate( pTask, sizeof( Task ) );
}

// --------------------------------------------------------------------------

void ServicesManager::ProcessRecvPacket( WorkItem* pItem )
{
    DEJA_CONTEXT( "ServicesManager::ProcessRecvPacket" );

    Packet* pPacket = pItem->m_pPacket;
    ASSERT( pPacket );
    DEJA_TRACE( "ServicesManager::ProcessRecvPacket", "ProcessRecvPacket:  Initiate = %d, TaskID = %p", pPacket->m_pHeader->m_Initiate, pPacket->GetTaskId() );
    //TMA_TRACE( "ServicesManager", "ProcessRecvPacket TaskId = %p", pPacket->GetTaskId() );

    // Is this an initiation packet?
    if( pPacket->m_pHeader->m_Initiate )
    {
        // Get the service that this packet targets.
        Service* pService = GetService( pPacket->m_pHeader->m_ServiceId );
        if( pService )
        {
            // Call the service to create a corresponding task.
            Task* pTask = pService->OnNewTask( pPacket );
            if( pTask )
            {
                pTask->m_bExternallyOwned = false;
                m_TaskList.AddTask( pTask );
            }
        }
    }
    else
    {
        // Find the Task for this packet and deliver it.
        m_TaskList.ProcessRecvPacket( pPacket );
    }

    // Free the packet.
    FreePacket( pPacket );
}

// --------------------------------------------------------------------------

void ServicesManager::ProcessTick()
{
    DEJA_TRACE( "ServicesManager::ProcessTick", "ProcessTick" );
    //TMA_TRACE( "ServicesManager", "ProcessTick" );
    DEJA_CONTEXT( "ServicesManager::ProcessTick" );

    bool bProcessSendCalled = false;

    ASSERT( m_pNode != nullptr );
    while ((m_pNode != nullptr) && (m_FreeSendPacketQ.GetCount() > 1))
    {
        // Get the highest priority task that wants a packet.
        const bool IsConnected = m_pNode->IsConnected();

        // Get a packet to send.
        Packet* pPacket = (Packet*)m_FreeSendPacketQ.Receive();

        bProcessSendCalled = true;
        const bool bTaskFound{ m_TaskList.ProcessSendPacket( IsConnected, pPacket ) };
        if( bTaskFound )
        {
            const tmipc::Result Result{ Send( pPacket ) };

            // Exit Tick if the Send failed.
            if( Result != TMIPC_RESULT_OK )
            {
                break;
            }
        }
        else
        {
            // Task was not found, discard the Packet.
            FreePacket( pPacket );
            break;
        }
    }

    // If ProcessSendPacket was not called above then we need to call
    // CleanupFinishedTasks manually.
    if( !bProcessSendCalled )
    {
        m_TaskList.CleanupFinishedTasks();
    }

    DEJA_TRACE( "ServicesManager::ProcessTick", "ProcessTick ended" );
}

// --------------------------------------------------------------------------

void ServicesManager::ProcessDisconnected()
{
    DEJA_TRACE( "ServicesManager::ProcessDisconnected", "ProcessDisconnected" );
    //TMA_TRACE( "ServicesManager", "ProcessDisconnected" );

    // Cancel all tasks in the list.
    ScopedLock Lock( m_Lock );
    m_TaskList.CancelAll();

    m_DisconnectEvent.Set();
}

// --------------------------------------------------------------------------

void ServicesManager::ProcessDeleteServices()
{
    DEJA_TRACE( "ServicesManager::ProcessDeleteServices", "ProcessDeleteServices" );

    // Delete all our services.
    ScopedLock lock( m_Lock );
    Service* pList = m_pFirstService;
    while( pList )
    {
        Service* pNext = pList->m_pNext;

        pList->Destroy();
        pList->~Service();
        tma::s_Deallocate( pList, sizeof( Service ) );

        pList = pNext;
    }
    m_pFirstService = nullptr;
}

// --------------------------------------------------------------------------

void ServicesManager::ProcessSleep()
{
//    ScopedLock lock( m_Lock );
TMA_POWER_TEST_PRINT( "[%s]!!! - A:. m_TaskList: %d; Sleeping TaskList: %d.\n", _BestFunctionName_, m_TaskList.GetTaskCount(), m_TaskList.GetSleepingTaskCount() );

    // Store all remaining tasks in the SleepingTaskList, until woken up.
TMA_POWER_TEST_PRINT( "[%s]!!! - B:. m_TaskList: %d; Sleeping TaskList: %d.\n", _BestFunctionName_, m_TaskList.GetTaskCount(), m_TaskList.GetSleepingTaskCount() );
    if( m_pNode != nullptr )
    {
//TMA_POWER_TEST_PRINT( "[%s]!!! - C:. m_TaskList: %d; Sleeping TaskList: %d.\n", _BestFunctionName_, m_TaskList.GetTaskCount(), m_TaskList.GetSleepingTaskCount() );
        m_bPreserveStateThroughSleepWake = m_pNode->PreserveStateThroughSleepWake();
    }

TMA_POWER_TEST_PRINT( "[%s]!!! - D:. m_TaskList: %d; Sleeping TaskList: %d.\n", _BestFunctionName_, m_TaskList.GetTaskCount(), m_TaskList.GetSleepingTaskCount() );
//    m_TaskList.PrintTaskInfo();

    if( PreserveStateThroughSleepWake() )
    {
TMA_POWER_TEST_PRINT( "[%s]!!! - E:. m_TaskList: %d; Sleeping TaskList: %d.\n", _BestFunctionName_, m_TaskList.GetTaskCount(), m_TaskList.GetSleepingTaskCount() );
        m_TaskList.GoToSleep();
        // Let the services know they are going to sleep.
        for( Service* pList = m_pFirstService; pList != nullptr; pList = pList->m_pNext )
        {
            pList->OnSleep();
        }
    }
TMA_POWER_TEST_PRINT( "[%s]!!! - F:. m_TaskList: %d; Sleeping TaskList: %d.\n", _BestFunctionName_, m_TaskList.GetTaskCount(), m_TaskList.GetSleepingTaskCount() );

    // Release the GoToSleep() call.
    m_SleepEvent.Set();
    // Wait for the "WakeUp()" call.
TMA_POWER_TEST_PRINT( "[%s]!!! - G:. m_TaskList: %d; Sleeping TaskList: %d.\n", _BestFunctionName_, m_TaskList.GetTaskCount(), m_TaskList.GetSleepingTaskCount() );
    m_WakeUpEvent.Wait( TMIPC_INFINITE );
    m_IsSleeping = false;
TMA_POWER_TEST_PRINT( "[%s]!!! - H:. m_TaskList: %d; Sleeping TaskList: %d.\n", _BestFunctionName_, m_TaskList.GetTaskCount(), m_TaskList.GetSleepingTaskCount() );

//    m_TaskList.PrintTaskInfo();

    // A WakeUp was issued: Restore all tasks to the m_TaskList.
    if( PreserveStateThroughSleepWake() )
    {
TMA_POWER_TEST_PRINT( "[%s]!!! - I:. m_TaskList: %d; Sleeping TaskList: %d.\n", _BestFunctionName_, m_TaskList.GetTaskCount(), m_TaskList.GetSleepingTaskCount() );
        // Let the services know they are waking up.
        for( Service* pList = m_pFirstService; pList != nullptr; pList = pList->m_pNext )
        {
            pList->OnWakeUp();
        }

        m_TaskList.WakeUp();
//        m_TaskList.PrintTaskInfo();
TMA_POWER_TEST_PRINT( "[%s]!!! - J:. m_TaskList: %d; Sleeping TaskList: %d.\n", _BestFunctionName_, m_TaskList.GetTaskCount(), m_TaskList.GetSleepingTaskCount() );
//        // New packets have come back "online", so make sure a Tick is issued to kickstart the Tasks again (in the WorkThread).
//        Tick();
//        m_TaskList.PrintTaskInfo();
    }
TMA_POWER_TEST_PRINT( "[%s]!!! - K:. m_TaskList: %d; Sleeping TaskList: %d.\n", _BestFunctionName_, m_TaskList.GetTaskCount(), m_TaskList.GetSleepingTaskCount() );
}


// --------------------------------------------------------------------------
} // namespace tmipc
// --------------------------------------------------------------------------
