﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "tmipc_defines.h"
#include <string>

#ifdef NN_SDK_LOG
#define TMIPC_ERROR_LOG(x) NN_SDK_LOG("TMIPC UNEXPECTED BEHAVIOR: %s", x)
#else
#define TMIPC_ERROR_LOG(x)
#endif

//==============================================================================
namespace tmipc {
//==============================================================================

class MsgQueue;

enum PacketConsts
{
    // This size *must* match the TargetManager's equivalent in file:
    // .\Oasis\Source\TargetManager\tmipc\src\tmipc_packet.h.
    MAX_PACKET_DATA = (56 * 1024)
};

//==============================================================================

class Packet
{
    friend class ServicesManager;

public:

    struct Header
    {
        u32             m_ServiceId;        // Service Id.
        u32             m_TaskId;           // Task Id.
        s16             m_TaskType;         // Task type for initiation of tasks.
        u8              m_Initiate;         // 1 = Initiation, 0 = Continuation.
        u8              m_Version;          // Header version number.
        u32             m_DataLen;          // Length of data section to follow.
        u32             m_Reserved1;        // Reserved for future use...
        u32             m_Reserved2;        // Reserved for future use...
        u32             m_Reserved3;        // Reserved for future use...
        u32             m_Reserved4;        // Reserved for future use...
    };

private:
        u8*             m_pBuffer;          // Allocated buffer.
        u32             m_BufferLen;        // Length of the allocated buffer.
        Header*         m_pHeader;          // Pointer to Header in the allocated buffer.
        u8*             m_pDataStart;       // Pointer to Start of Data section in buffer.
        u8*             m_pDataEnd;         // Pointer to End of Data section in buffer.
        u8*             m_pCursor;          // Cursor into the buffer.
        MsgQueue*       m_pFreeQ;           // Which Queue to free this packet to.

public:
                        Packet          ();
virtual                ~Packet          ();

        void            SetServiceId    ( u32                   ServiceId );
        u32             GetServiceId    () const;
        void            SetTaskId       ( u32                   TaskId );
        u32             GetTaskId       () const;
        void            SetTaskType     ( s32                   TaskType );
        s32             GetTaskType     () const;
        void            SetInitiate     ( u8                    Initiate );
        u32             GetInitiate     () const;
        void            SetVersion      ( u8                    Version );
        u32             GetVersion      () const;
        s32             GetDataLen      ();
        s32             GetDataLen      () const;
        s32             GetSendLen      () const;
        s32             GetRecvLen      () const;
        u8*             GetBuffer       ();
        const u8*       GetBuffer       () const;
        u8*             GetCursor       ();

        s32             ResetCursor     ();

        s32             WriteData       ( const void*           pData,
                                          s32                   nBytes );
        s32             WriteU8         ( const u8&             V );
        s32             WriteS8         ( const s8&             V );
        s32             WriteS16        ( const s16&            V );
        s32             WriteU16        ( const u16&            V );
        s32             WriteS32        ( const s32&            V );
        s32             WriteU32        ( const u32&            V );
        s32             WriteS64        ( const s64&            V );
        s32             WriteU64        ( const u64&            V );
        s32             WriteString     ( const char*           pString );
        s32             WriteString     ( const std::string&    String );
        s32             WriteFormat     ( const char*           pFormat, ... );

        s32             ReadData        ( void*                 pData,
                                          s32                   nBytes );
        s32             ReadU8          ( u8&                   V );
        s32             ReadS8          ( s8&                   V );
        s32             ReadS16         ( s16&                  V );
        s32             ReadU16         ( u16&                  V );
        s32             ReadS32         ( s32&                  V );
        s32             ReadU32         ( u32&                  V );
        s32             ReadS64         ( s64&                  V );
        s32             ReadU64         ( u64&                  V );
        s32             ReadString      ( char*                 pBuffer,
                                          s32                   bufferLen,
                                          s32*                  pStringLen = NULL );
        s32             ReadString      ( std::string&          String );
};

//==============================================================================
} // namespace tmipc
//==============================================================================
