﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "tmipc_node.h"
#if defined( TMIPC_TARGET_HORIZON )
#include "tmipc_node_usb.h"
#include <nn/usb/usb_Device.h>
#endif // TMIPC_TARGET_HORIZON

//==============================================================================
namespace tmipc {
//==============================================================================

#if defined( TMIPC_TARGET_HORIZON )

class USBInterface
{
private:

    // "Magic Numbers".
    enum
    {
        USB_WRITE_ENDPOINT_INDEX = 0,     // This is TargetManager's "Read".
        USB_READ_ENDPOINT_INDEX  = 1,     // This is TargetManager's "Write".
    };

#if ENABLE_BENCHMARK_TESTING
    typedef void (*OnPacketReceivedCallback)( const tmipc::Packet& Packet );
    typedef void (*OnPacketSendCallback)( const tmipc::Packet& Packet );
#endif // ENABLE_BENCHMARK_TESTING

private:

    // Send/Receive buffers: These sizes *must be* at least as big as a packet.
static  const int                           m_ReceiveBufferSize   = { 1024 * 96 };
static  const int                           m_SendBufferSize      = { 1024 * 96 };
static  NN_USB_DMA_ALIGN unsigned char      m_ReceiveDataBuffer[m_ReceiveBufferSize];
static  NN_USB_DMA_ALIGN unsigned char      m_SendDataBuffer[m_SendBufferSize];

static  const s32                           m_HeaderBufferSize    = { 4096 };
static  NN_USB_DMA_ALIGN unsigned char      m_ReceiveHeaderBuffer[m_HeaderBufferSize];

    // Related to the USB Indication polling.
static  const size_t                        m_IndicationPollingThreadStackSize = TMIPC_STACK_SIZE;
static  NN_OS_ALIGNAS_THREAD_STACK char     m_IndicationPollingThreadStack[m_IndicationPollingThreadStackSize];
static  tmipc::Thread                       m_IndicationPollingThread;  // The Indication polling thread.
static  nn::os::Event                       m_BreakEvent;  // The event to exit indication polling thread

    // The send function needs to be thread safe.
static  Mutex                               m_SendMutex;

    // Callback function for USB events.
    typedef void (*EventCallback)( tmipc::NodeUSB::USBEvent Event, void* lpArgs );
static  void*                               m_pEventArgument;
static  EventCallback                       m_pfEventCallback;

    // Nintendo's USB interface.
static  nn::usb::DsClient                   m_DsClient;
static  nn::usb::DsInterface                m_DsInterface;
static  nn::usb::DsEndpoint                 m_DsEndpoints[2];

static  nn::usb::UsbStringDescriptor        m_LanguageStringDescriptor;
static  nn::usb::UsbStringDescriptor        m_ManufacturerStringDescriptor;
static  nn::usb::UsbStringDescriptor        m_ProductStringFullSpeedDescriptor;
static  nn::usb::UsbStringDescriptor        m_SerialNumberStringDescriptor;
static  nn::usb::UsbStringDescriptor        m_InterfaceStringDescriptor;

static  nn::usb::UsbDeviceDescriptor        m_UsbDeviceDescriptorFs;
static  nn::usb::UsbDeviceDescriptor        m_UsbDeviceDescriptorHs;
static  nn::usb::UsbDeviceDescriptor        m_UsbDeviceDescriptorSs;
static  uint8_t                             m_BinaryObjectStore[];

static  nn::usb::UsbInterfaceDescriptor     m_UsbInterfaceDescriptor;           // need to be initialized once...
static  nn::usb::UsbEndpointDescriptor      m_UsbEndpointDescriptorsFs[2];      //
static  nn::usb::UsbEndpointDescriptor      m_UsbEndpointDescriptorsHs[2];      //
static  nn::usb::UsbEndpointDescriptor      m_UsbEndpointDescriptorsSs[2];      //
static nn::usb::UsbEndpointCompanionDescriptor m_UsbEndpointCompanionDescriptor;

static  bool                                m_bIsInitialized;

static  s32                                 m_ThreadPriority;

#if ENABLE_BENCHMARK_TESTING
    static OnPacketReceivedCallback m_pfOnPacketReceivedCallback;     // Called when a packet is received.
    static OnPacketSendCallback     m_pfOnPacketSendCallback;         // Called right before a packet is sent.
#endif // ENABLE_BENCHMARK_TESTING

private:

    // Indication polling thread.
static  s32     IndicationPollingThread( void* lpThis );

    // Sets the thread priority.
static  void    SetThreadPriority( s32 ThreadPriority);

    // Gets the thread priority.
static  s32     GetThreadPriority();

public:

    // Initialize the USB interface.
    static tmipc::Result Initialize( s32 ThreadPriority );

    // Finalize the USB interface.
    static tmipc::Result Finalize();

    // Sets the callback information.
    static void SetCallbackInformation( EventCallback pfEventCallback, void* Arguments );

    // Aborts blocking calls.
    static void AbortBlockingCalls();

    // Reads data from the USB connection.
    // Returns: TMIPC_RESULT_OK: success; non-TMIPC_RESULT_OK: an error occurred.
    static tmipc::Result Read( tmipc::Packet* pPacket );

    // Writes data to the USB connection.
    // Returns: TMIPC_RESULT_OK: success; non-TMIPC_RESULT_OK: an error occurred.
    static tmipc::Result Write( tmipc::Packet* pPacket );

    // Sends a Beacon request back to the remote connection.
    static tmipc::Result SendBeaconPacket( tmipc::Packet* pPacket );

#if ENABLE_BENCHMARK_TESTING
    // Sets/clears the callback invoked when any packet is received.
    static void SetOnPacketReceivedCallback( OnPacketReceivedCallback pfOnPacketReceivedCallback );
    // Sets/clears the callback invoked when right before a packet is sent.
    static void SetOnPacketSendCallback( OnPacketSendCallback pfOnPacketSendCallback );
#endif // ENABLE_BENCHMARK_TESTING
};

#endif // TMIPC_TARGET_HORIZON

//==============================================================================
} // namespace tmipc
//==============================================================================
