﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "tmipc_node_tics_plug.h"
#include "tmipc_packet.h"
#include "../DejaInsight.h"

//==============================================================================
namespace tmipc {
//==============================================================================

#if ENABLE_BENCHMARK_TESTING
TICSPlug::OnPacketReceivedCallback TICSPlug::m_pfOnPacketReceivedCallback = { nullptr };
TICSPlug::OnPacketSendCallback     TICSPlug::m_pfOnPacketSendCallback     = { nullptr };
#endif // ENABLE_BENCHMARK_TESTING

// --------------------------------------------------------------------------
// Constructor.
TICSPlug::TICSPlug()
{
    DEJA_TRACE( "TICSPlug::TICSPlug", "TICSPlug( %p )\n", this );
    Initialize();
}

// --------------------------------------------------------------------------
// Constructor.
TICSPlug::TICSPlug( const char* pName )
{
    DEJA_TRACE( "TICSPlug::TICSPlug", "TICSPlug( %p, '%s' )\n", this, ((pName != nullptr) && (*pName != '\0')) ? pName : "<null/empty>" );
    Initialize();
    strncpy( m_Name, pName, sizeof(m_Name) - 1 );
    m_Name[sizeof(m_Name) - 1] = '\0';
}

// --------------------------------------------------------------------------
// Destructor.
TICSPlug::~TICSPlug()
{
    DEJA_TRACE( "TICSPlug::~TICSPlug", "~TICSPlug( %p )\n", this );
    m_bSendWasCancelled             = false;
    m_bReceiveWasCancelled          = false;
    m_pfOnSessionStartedCallback    = nullptr;
    m_pfOnSessionDetachedCallback   = nullptr;
    m_pOnSessionStartedContext      = nullptr;
    m_Name[0]                       = '\0';
    m_SendEvent.Destroy();
    m_ReceiveEvent.Destroy();
}

// --------------------------------------------------------------------------
// Initializes member data.
void TICSPlug::Initialize()
{
    DEJA_TRACE( "TICSPlug::Initialize", "Initialize( %p )\n", this );
    memset( m_Name, 0, sizeof(m_Name) );
    m_ReceiveEvent.Create();
    m_SendEvent.Create();
    m_pfOnSessionStartedCallback    = nullptr;
    m_pfOnSessionDetachedCallback   = nullptr;
    m_pOnSessionStartedContext      = nullptr;
    m_bReceiveWasCancelled          = false;
    m_bSendWasCancelled             = false;
}

// --------------------------------------------------------------------------
// Callback for Receive complete.
int TICSPlug::OnReceiveComplete( tics::Buffer* pBuffer, long Offset, long Length )
{
    DEJA_TRACE( "TICSPlug::OnReceiveComplete", "OnReceiveComplete( %p, %d, %d )\n", pBuffer, Offset, Length );
    int Result = { 0 };

    // Allow the base class to do "additional work".
    Result = BaseClass::OnReceiveComplete( pBuffer, Offset, Length );

    // Set the event, so the Receive can become unblocked.
    m_ReceiveEvent.Set();

    return Result;
}

// --------------------------------------------------------------------------
// Callback for send complete.
int TICSPlug::OnSendComplete( tics::Buffer* pBuffer, long Offset, long Length )
{
    DEJA_TRACE( "TICSPlug::OnSendComplete", "OnSendComplete( %p, %d, %d ).\n", pBuffer, Offset, Length );
    int Result = { 0 };

    // Allow the base class to do "additional work".
    BaseClass::OnSendComplete( pBuffer, Offset, Length );

    // Set the event, to unblock Send( const tmipc::Packet * ).
    m_SendEvent.Set();

    return Result;
}

// --------------------------------------------------------------------------
// Received when a session starts.
void TICSPlug::OnSessionStarted( tics::portability::stl::string Type, Endpoint* pConnectedEndPoint )
{
    DEJA_TRACE( "TICSPlug::OnSessionStarted", "OnSessionStarted( Type: %s; ConnectedEndPoint: %p ).\n", Type.c_str(), pConnectedEndPoint );

    // Allow the base class to do "additional work".
    BaseClass::OnSessionStarted( Type, pConnectedEndPoint );
    SetType( Type );
    SetEndpoint( pConnectedEndPoint );

    m_IsConnected = true;

    // Let the listener know that a session has started.
    if( m_pfOnSessionStartedCallback != nullptr )
    {
        m_pfOnSessionStartedCallback( m_pOnSessionStartedContext );
    }
}

// --------------------------------------------------------------------------
// Received when a Plug detaches.
int TICSPlug::OnDetach()
{
    DEJA_TRACE( "TICSPlug::OnDetach", "OnDetach( %p )\n", this );

    // This plug has been detached...
    int Result = { tics::BRIDGE_ERROR_SUCCESS };
    if( m_pfOnSessionDetachedCallback != nullptr )
    {
        Result = m_pfOnSessionDetachedCallback( m_pOnSessionDetachedContext );
    }
    m_IsConnected = false;
    CancelBlockingOperations();

    return Result;
}

// --------------------------------------------------------------------------
// Received when a remote plug detaches.
int TICSPlug::OnRemoteDetach( tics::DisconnectReason reason )
{
    DEJA_TRACE( "TICSPlug::OnDetach", "OnDetach( %p, %d )\n", this, reason );
    m_IsConnected = false;
    CancelBlockingOperations();
    return tics::BRIDGE_ERROR_SUCCESS;
}

// --------------------------------------------------------------------------
// Allow the user to register a callback when a Session has started.
void TICSPlug::RegisterOnSessionStartedCallback( OnSessionStartedCallback pfOnSessionStartedCallback, void* pContext )
{
    m_pfOnSessionStartedCallback = pfOnSessionStartedCallback;
    m_pOnSessionStartedContext   = pContext;
}

// --------------------------------------------------------------------------
// Allow the user to register a callback when a Session has detached.
void TICSPlug::RegisterOnSessionDetachedCallback( OnSessionDetachedCallback pfOnSessionDetachedCallback, void* pContext )
{
    m_pfOnSessionDetachedCallback = pfOnSessionDetachedCallback;
    m_pOnSessionDetachedContext = pContext;
}

// --------------------------------------------------------------------------
// Receives data.  Blocking call. This function is not thread safe. It is
// assumed that the caller prevents Threading issues (Mutex, whatever).
tmipc::Result TICSPlug::Receive( tmipc::Packet& Packet )
{
    DEJA_TRACE( "TICSPlug::Receive", "Receive( %p, %p )\n", this, &Packet );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    if( IsConnected() )
    {
        // Set this event, so that the Wait() below will block until
        // OnReceiveComplete(...) Resets it.
        m_ReceiveEvent.Reset();

        // Tell the Plug to receive the Packet header...
        tmipc::Packet::Header* pHeader = { reinterpret_cast<tmipc::Packet::Header*>(Packet.GetBuffer()) };
        m_ReceiveBuffer.Reset( pHeader, sizeof(*pHeader) );
        BeginReceive( sizeof(*pHeader), sizeof(*pHeader), &m_ReceiveBuffer, 0 );

        // Wait for data to be returned or cancelled.  The packet gets filled
        // in OnReceiveComplete(...).
        m_ReceiveEvent.Wait( TMIPC_INFINITE );

        // Do *not* attempt to read the rest of the Packet, if an abort was
        // issued.
        if( !m_bReceiveWasCancelled )
        {
            // Do *not* attempt to read data, if the packet has no data.
            // i.e.: it's just a header.
            const s32 DataLength = { Packet.GetDataLen() };
            if( pHeader->m_DataLen > 0 )
            {
                m_ReceiveEvent.Reset();

                // Tell the Plug to receive the Data...
                u8*  pBuffer    = { Packet.GetBuffer() };
                long DataOffset = static_cast<long>(sizeof(tmipc::Packet::Header));
                m_ReceiveBuffer.Reset( pBuffer, DataLength );
                BeginReceive( DataLength, DataLength, &m_ReceiveBuffer, DataOffset );

                // Wait for data to be returned or cancelled.  The packet gets filled
                // in OnReceiveComplete(...).
                m_ReceiveEvent.Wait( TMIPC_INFINITE );
            }
        }

#if ENABLE_BENCHMARK_TESTING
        // Do *not* attempt to use the Packet, if an abort was issued.
        if( !m_bReceiveWasCancelled )
        {
            // Call the "receive callback".
            if( m_pfOnPacketReceivedCallback != nullptr )
            {
                m_pfOnPacketReceivedCallback( Packet );
            }
        }
#endif // ENABLE_BENCHMARK_TESTING
    }
    else
    {
        Result = tmipc::TMIPC_RESULT_DISCONNECTED;
    }

    // If the receive was cancelled, then make sure an error is returned.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        if( m_bReceiveWasCancelled )
        {
            Result = tmipc::TMIPC_RESULT_CANCELED;
        }
    }

    // Clear the cancelled receive flag.
    m_bReceiveWasCancelled = false;

    return Result;
}

// --------------------------------------------------------------------------
// Sends data.  Blocking call. This function is not thread safe. It is
// assumed that the caller prevents Threading issues (Mutex, whatever).
tmipc::Result TICSPlug::Send( const tmipc::Packet& Packet )
{
    DEJA_TRACE( "TICSPlug::Send", "Receive( %p, %p )\n", this, &Packet );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    if( IsConnected() )
    {
        m_SendEvent.Reset();

        // Fill the send buffer with data and start the transfer...
        const int   SendLength = Packet.GetSendLen();
        const void* pData      = Packet.GetBuffer();
        m_SendBuffer.Reset( const_cast<void*>(pData), SendLength );

#if ENABLE_BENCHMARK_TESTING
        // Call the "send callback".
        if( m_pfOnPacketSendCallback != nullptr )
        {
            m_pfOnPacketSendCallback( Packet );
        }
#endif // ENABLE_BENCHMARK_TESTING

        int result = BeginSend( &m_SendBuffer, 0, m_SendBuffer.m_DataLen, true );
        if (result == tics::BRIDGE_ERROR_SUCCESS)
        {
            // Wait for data to be sent (or cancelled).
            m_SendEvent.Wait( TMIPC_INFINITE );
        }
        else
        {
            return tmipc::TMIPC_RESULT_NOT_INITIALIZED;
        }

    }
    else
    {
        Result = tmipc::TMIPC_RESULT_DISCONNECTED;
    }

    // If the send was cancelled, then make sure an error is returned.
    if( m_bSendWasCancelled )
    {
        Result = tmipc::TMIPC_RESULT_CANCELED;
    }

    // Clear the cancelled send flag.
    m_bSendWasCancelled = false;

    return Result;
}

// --------------------------------------------------------------------------
// Cancels pending Send/Receive operations.
void TICSPlug::CancelBlockingOperations()
{
    DEJA_TRACE( "TICSPlug::CancelBlockingOperations", "CancelBlockingOperations( %p ).\n", this );
    // These get Reset in the Send/Receive functions.
//TODO: this could pose a problem if called before a connection has been made...
    m_bSendWasCancelled    = true;
    m_bReceiveWasCancelled = true;
    m_SendEvent.Set();
    m_ReceiveEvent.Set();
}

//==============================================================================
#if ENABLE_BENCHMARK_TESTING

//==============================================================================
// Sets/clears the callback invoked when any packet is received.
void TICSPlug::SetOnPacketReceivedCallback( OnPacketReceivedCallback pfOnPacketReceivedCallback )
{
    m_pfOnPacketReceivedCallback = pfOnPacketReceivedCallback;
}

//==============================================================================
// Sets/clears the callback invoked when right before a packet is sent.
void TICSPlug::SetOnPacketSendCallback( OnPacketSendCallback pfOnPacketSendCallback )
{
    m_pfOnPacketSendCallback = pfOnPacketSendCallback;
}

#endif // ENABLE_BENCHMARK_TESTING
//==============================================================================

}
//==============================================================================
