﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "tmipc_node.h"
#include "tmipc_service.h"
#include "tmipc_result.h"
#include "tmipc_thread.h"
#include "tmipc_packet.h"
#include "tmipc_services_manager.h"
#include "tmipc_task.h"
#include "..\DejaInsight.h"
#include "..\Version.h"

//==============================================================================
namespace tmipc {
//==============================================================================

Node::Node()
:   m_pEventCallback    ( nullptr )
,   m_pEventCallbackData( nullptr )
,   m_Initialized       ( false )
,   m_pServicesManager  ( nullptr )
,   m_PreserveStateThroughSleepWake( false )
,   m_WasWokenUp        ( false )
{
    DEJA_TRACE( "Node::Node", "Node::Node" );

    m_Lock.Create();
    m_ThreadPriority = nn::os::InvalidThreadPriority;
    m_NodeId = TMIPC_SERVER_NODE_ID;
}

//==============================================================================

Node::~Node()
{
    ASSERT( !m_Initialized );

    m_Lock.Destroy();
}

//==============================================================================
// Process the TMSInfo string.
void Node::ProcessTMSInfoString( const char* pTMSInfoString )
{
    tmipc::Result Result{ TMIPC_RESULT_OK };
TMA_POWER_TEST_PRINT( "[Node::ProcessTMSInfoString(...)]:%s%s - Start.\n", (pTMSInfoString != nullptr) ? " \n" : "", (pTMSInfoString != nullptr) ? pTMSInfoString : " <nullptr>"  );
    if( Result == TMIPC_RESULT_OK )
    {
TMA_POWER_TEST_PRINT( "[Node::ProcessTMSInfoString(...)]: - A.\n" );
        if( pTMSInfoString != nullptr )
        {
            tma::TMSInfo TMSInfo{};
            tma::DecodeTMSInfoString( TMSInfo, pTMSInfoString );

            m_PreserveStateThroughSleepWake = TMSInfo.BCID >= tmipc::TaskType_TMSInfoSupport;

            // Determine if this Target was sleeping, or if Target Manager did *not* know this Target was sleeping.
            if( !WasWokenUp() || !TMSInfo.WasSleeping )
            {
TMA_POWER_TEST_PRINT( "[Node::ProcessTMSInfoString(...)]: - B.  Tasks were cancelled.\n" );
                m_pServicesManager->CancelAllTasks();
            }

TMA_POWER_TEST_PRINT( "[Node::ProcessTMSInfoString(...)]: - C. BCID: %d >= %d.  Was woken up: '%s', PS: '%s'.\n", TMSInfo.BCID, tmipc::TaskType_TMSInfoSupport, WasWokenUp() ? "yes" : "no", TMSInfo.WasSleeping ? "True" : "False" );
        }
    }
TMA_POWER_TEST_PRINT( "[Node::ProcessTMSInfoString(...)]: Support State Preservation through Sleep/Wake? '%s'.\n", PreserveStateThroughSleepWake() ? "Yes" : "No" );
}

//==============================================================================

void Node::Init()
{
    ASSERT( !m_Initialized );

    // Initialized.
    m_Initialized = true;
}

//==============================================================================

void Node::Kill()
{
    if( m_Initialized )
    {
        // Call this *before* calling Disconnect, because a connection may not have
        // finished before tearing down the threads.
        StopListening();

        if( IsConnected() )
        {
            Disconnect();
        }

        // No longer initialized.
        m_Initialized = false;
    }
}

//==============================================================================

Packet* Node::AllocSendPacket()
{
    DEJA_TRACE( "Node::AllocSendPacket", "AllocSendPacket" );
    DEJA_CONTEXT( "Node::AllocSendPacket" );

    ASSERT( m_Initialized );
    ASSERT( m_pServicesManager );

    Packet* pPacket = m_pServicesManager->AllocSendPacket();
    return pPacket;
}

//==============================================================================

Packet* Node::AllocRecvPacket()
{
    DEJA_TRACE( "Node::AllocRecvPacket", "AllocRecvPacket" );

    ASSERT( m_Initialized );
    ASSERT( m_pServicesManager );

    Packet* pPacket = m_pServicesManager->AllocRecvPacket();
    return pPacket;
}

//==============================================================================

void Node::FreePacket( Packet* pPacket )
{
    DEJA_TRACE( "Node::FreePacket", "FreePacket" );

    ASSERT( m_Initialized );
    ASSERT( m_pServicesManager );
    m_pServicesManager->FreePacket( pPacket );
}

//==============================================================================
// Listens for connections.
Result Node::Listen()
{
    Result Result = Result::TMIPC_RESULT_OK;
    return Result;
}

//==============================================================================
// Stops listening for connections.  This does *not* cause a disconnect.
void Node::StopListening()
{
}

//==============================================================================

// Needed to allocate/deallocate Receive packets.
// TODO: Revisit this to see if there is a cleaner way to do this.
void Node::SetServicesManager( ServicesManager* pServicesManager )
{
    m_pServicesManager = pServicesManager;
}

//==============================================================================

void Node::SetEventCallback( EventCallback pCallback, void* pUserData )
{
    m_pEventCallback        = pCallback;
    m_pEventCallbackData    = pUserData;
}

//==============================================================================

void Node::SetThreadPriority( s32 Priority )
{
    m_ThreadPriority = Priority;
}

//==============================================================================

s32 Node::GetThreadPriority()
{
    return m_ThreadPriority;
}

//==============================================================================

void Node::ProcessReceived( Packet* pPacket )
{
    // Give the packet to the ServicesManager.  It will take ownership of the packet.
    m_pServicesManager->ProcessReceived( pPacket );
}

//==============================================================================

void Node::Tick()
{
    m_pServicesManager->Tick();
}

//==============================================================================

void Node::CancelAllPendingTasks()
{
    m_pServicesManager->CancelAllTasks();
}

//==============================================================================

void Node::OnEvent( eNodeEventCode EventType )
{
    if( m_pEventCallback != nullptr )
    {
        NodeEvent NotifyEvent(EventType);
        m_pEventCallback( m_pEventCallbackData, &NotifyEvent );
    }
}

//==============================================================================

void Node::Disconnected()
{
    ASSERT( m_Initialized );

    ASSERT( m_pServicesManager );
    if( m_pServicesManager != nullptr )
    {
        m_pServicesManager->Disconnected();
    }

    m_WasWokenUp = false;

    //Now send the event on.
    OnEvent( NodeEvent_Disconnected );
}

//==============================================================================
} // namespace tmipc
//==============================================================================
