﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\tmipc\tmipc_result.h"
#include "tio_Service.h"
#include "tio_Task.h"
#include <nn/dmnt/dmnt_Api.h>
#include "tio_types.h"
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>

#if 0
#define TARGETIO_TRACE( channel, ... ) NN_SDK_LOG( "(%lld)[tma][", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() ); \
    NN_SDK_LOG( channel ); \
    NN_SDK_LOG( "] - " ); \
    NN_SDK_LOG( __VA_ARGS__ ); \
    NN_SDK_LOG( "\n" )
#define TARGETIO_LOG( ... ) NN_SDK_LOG( __VA_ARGS__ )
#else
#define TARGETIO_TRACE( channel, ... )
#define TARGETIO_LOG( ... )
#endif

//==============================================================================
namespace tma { namespace target_io {
//==============================================================================

TIOTask::TIOTask()
{
#if PROFILE_TIO_TASKS
    m_StartingTick = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
#endif
}

//==============================================================================

TIOTask::~TIOTask()
{
#if PROFILE_TIO_TASKS
    int64_t EndTick = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
    TARGETIO_LOG( "TIOTask duration = %lld ticks\n", EndTick - m_StartingTick );
#endif
}

//==============================================================================

void TIOTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    (void)pPacket;
    // Done.
    Complete();
}

//==============================================================================

void TIOTask::OnSendPacket( tmipc::Packet* pPacket )
{
    // TODO - Handle Packet
    (void)pPacket;
    Complete();
}

//==============================================================================

void TIOTask::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;
    Complete();
}

//==============================================================================
// Generic response, used by most of our tasks
void TIOTask::OnComplete( tmapi::result Result )
{
    TARGETIO_TRACE( "TIOTask::OnComplete", "OnComplete - Result = %d", Result );
    tmipc::Packet* pPacket = AllocSendPacket();
    pPacket->WriteS32( Result );
    m_pServicesManager->Send( pPacket );
    Complete();
}

//==============================================================================

static bool IsMemoryCardInsertedError( nn::Result result )
{
    bool Result = false;
    if(result.IsSuccess() == false)
    {
        if( nn::fs::ResultPortSdCardNoDevice::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceRemoved::Includes( result ) ||
            nn::fs::ResultSdCardFileSystemInvalidatedByRemoved::Includes( result ) ||
            nn::fs::ResultPortSdCardNotActivated::Includes( result ) ||
            nn::fs::ResultSdCardAccessFailed::Includes( result )  ||
            nn::fs::ResultPortSdCardNotAwakened::Includes( result ) ||
            nn::fs::ResultPortSdCardNotSupported::Includes( result ) ||
            nn::fs::ResultPortSdCardNotImplemented::Includes( result )
            )
        {
            Result = true;
        }
    }

    return Result;
}

//==============================================================================

static bool IsMemoryCardIOError( nn::Result result )
{
    bool Result = false;
    if(result.IsSuccess() == false)
    {
        // All the possible errors:
        if (nn::fs::ResultPortSdCardResponseIndexError::Includes( result ) ||
            nn::fs::ResultPortSdCardResponseEndBitError::Includes( result ) ||
            nn::fs::ResultPortSdCardResponseCrcError::Includes( result ) ||
            nn::fs::ResultPortSdCardResponseTimeoutError::Includes( result ) ||
            nn::fs::ResultPortSdCardDataEndBitError::Includes( result ) ||
            nn::fs::ResultPortSdCardDataCrcError::Includes( result ) ||
            nn::fs::ResultPortSdCardDataTimeoutError::Includes( result ) ||
            nn::fs::ResultPortSdCardAutoCommandResponseIndexError::Includes( result ) ||
            nn::fs::ResultPortSdCardAutoCommandResponseEndBitError::Includes( result ) ||
            nn::fs::ResultPortSdCardAutoCommandResponseCrcError::Includes( result ) ||
            nn::fs::ResultPortSdCardAutoCommandResponseTimeoutError::Includes( result ) ||
            nn::fs::ResultPortSdCardCommandCompleteSwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardTransferCompleteSwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusAddressOutOfRange::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusAddressMisalign::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusBlockLenError::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusEraseSeqError::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusEraseParam::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusWpViolation::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusLockUnlockFailed::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusComCrcError::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusIllegalCommand::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusDeviceEccFailed::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusCcError::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusError::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusCidCsdOverwrite::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusWpEraseSkip::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusEraseReset::Includes( result ) ||
            nn::fs::ResultPortSdCardDeviceStatusSwitchError::Includes( result ) ||
            nn::fs::ResultPortSdCardUnexpectedDeviceState::Includes( result ) ||
            nn::fs::ResultPortSdCardUnexpectedDeviceCsdValue::Includes( result ) ||
            nn::fs::ResultPortSdCardAbortTransactionSwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardCommandInhibitCmdSwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardCommandInhibitDatSwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardBusySwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardIssueTuningCommandSwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardTuningFailed::Includes( result ) ||
            nn::fs::ResultPortSdCardMmcInitializationSwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardMmcNotSupportExtendedCsd::Includes( result ) ||
            nn::fs::ResultPortSdCardUnexpectedMmcExtendedCsdValue::Includes( result ) ||
            nn::fs::ResultPortSdCardMmcEraseSwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardSdCardValidationError::Includes( result ) ||
            nn::fs::ResultPortSdCardSdCardInitializationSwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardSdCardGetValidRcaSwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardUnexpectedSdCardAcmdDisabled::Includes( result ) ||
            nn::fs::ResultPortSdCardSdCardNotSupportSwitchFunctionStatus::Includes( result ) ||
            nn::fs::ResultPortSdCardUnexpectedSdCardSwitchFunctionStatus::Includes( result ) ||
            nn::fs::ResultPortSdCardSdCardNotSupportAccessMode::Includes( result ) ||
            nn::fs::ResultPortSdCardSdCardNot4BitBusWidthAtUhsIMode::Includes( result ) ||
            nn::fs::ResultPortSdCardSdCardNotSupportSdr104AndSdr50::Includes( result ) ||
            nn::fs::ResultPortSdCardSdCardCannotSwitchedAccessMode::Includes( result ) ||
            nn::fs::ResultPortSdCardSdCardFailedSwitchedAccessMode::Includes( result ) ||
            nn::fs::ResultPortSdCardSdCardUnacceptableCurrentConsumption::Includes( result ) ||
            nn::fs::ResultPortSdCardSdCardNotReadyToVoltageSwitch::Includes( result ) ||
            nn::fs::ResultPortSdCardSdCardNotCompleteVoltageSwitch::Includes( result ) ||
            nn::fs::ResultPortSdCardInternalClockStableSwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardSdHostStandardUnknownAutoCmdError::Includes( result ) ||
            nn::fs::ResultPortSdCardSdHostStandardUnknownError::Includes( result ) ||
            nn::fs::ResultPortSdCardSdmmcDllCalibrationSwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardSdmmcDllApplicationSwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardSdHostStandardFailSwitchTo18V::Includes( result ) ||
            nn::fs::ResultPortSdCardNoWaitedInterrupt::Includes( result ) ||
            nn::fs::ResultPortSdCardWaitInterruptSwTimeout::Includes( result ) ||
            nn::fs::ResultPortSdCardAbortCommandIssued::Includes( result ) ||
            nn::fs::ResultPortSdCardNotSupported::Includes( result ) ||
            nn::fs::ResultPortSdCardNotImplemented::Includes( result )
            )
        {
            Result = true;
        }
    }

    return Result;
}

//==============================================================================

tmapi::result TIOTask::ConvertToTMAPI( nn::Result Res )
{
    tmapi::result Ret = tmapi::result::RESULT_OK;

    if( Res.IsFailure() )
    {
        // Do we even have a memory card inserted?
        if( IsMemoryCardInsertedError( Res ) )
        {
            Ret = tmapi::result::RESULT_TARGET_IO_NO_MEMORY_CARD;
        }
        else if( nn::fs::ResultInvalidHostHandle::Includes(Res) )
        {
            Ret = tmapi::result::RESULT_INVALID_PARAMETER;
        }
        else if( nn::fs::ResultPathNotFound::Includes(Res) )
        {
            Ret = tmapi::result::RESULT_INVALID_PARAMETER;
        }
        else if(nn::fs::ResultTargetLocked::Includes(Res) )
        {
            Ret = tmapi::result::RESULT_TARGET_UNAVAILABLE;
        }
        else if(nn::fs::ResultUsableSpaceNotEnough::Includes(Res) )
        {
            Ret = tmapi::result::RESULT_TARGET_IO_INSUFFICIENT_SPACE;
        }
        else if(nn::fs::ResultPathAlreadyExists::Includes(Res) )
        {
            Ret = tmapi::result::RESULT_TARGET_IO_PATH_ALREADY_EXISTS;
        }
        else if(nn::fs::ResultOutOfRange::Includes(Res) )
        {
            Ret = tmapi::result::RESULT_TARGET_IO_SEARCH_COMPLETE;
        }
        else if( IsMemoryCardIOError( Res )  )
        {
            Ret = tmapi::result::RESULT_TARGET_IO_MEMORY_CARD_ERROR;
        }
        else
        {
            Ret = tmapi::result::RESULT_UNKNOWN;
        }
    }
    TARGETIO_TRACE( "TIOTask::ConvertToTMAPI", "Converted %x to %d", Res.GetInnerValueForDebug(), Ret );
    return Ret;
}

//==============================================================================
// Drive tasks
//==============================================================================

void TIOTaskGetFreeSpace::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;
    int64_t FreeBytesAvailable;
    int64_t TotalNumberOfBytes;
    int64_t TotalNumberOfFreeBytes;
    tmapi::result Result = tmapi::result::RESULT_OK;

    nn::Result res = nn::dmnt::TargetIO_GetFreeSpace( &FreeBytesAvailable, &TotalNumberOfBytes, &TotalNumberOfFreeBytes );

    if( res.IsFailure() )
    {
        Result = ConvertToTMAPI( res );
    }

    tmipc::Packet* pReplyPacket = AllocSendPacket();
    pReplyPacket->WriteS32( Result );
    pReplyPacket->WriteS64( FreeBytesAvailable );
    pReplyPacket->WriteS64( TotalNumberOfBytes );
    pReplyPacket->WriteS64( TotalNumberOfFreeBytes );

    m_pServicesManager->Send( pReplyPacket );

    Complete();
}

//==============================================================================
// Get volume information

void TIOTaskGetVolumeInformation::OnInitiate( tmipc::Packet* pPacket )
{
    (void)pPacket;

    char pVolumeNameBuffer[1024];
    int32_t VolumeSerialNumber;
    int32_t MaximumComponentLength;
    int32_t FileSystemFlags;
    char pFileSystemNameBuffer[1024];

    tmipc::Packet* pReplyPacket = AllocSendPacket();

    nn::Result res = nn::dmnt::TargetIO_GetVolumeInformation( pVolumeNameBuffer, sizeof(pVolumeNameBuffer), pFileSystemNameBuffer, sizeof(pFileSystemNameBuffer),
            &VolumeSerialNumber, &MaximumComponentLength, &FileSystemFlags );
    if( res.IsFailure() )
    {
        s32 Result = ConvertToTMAPI( res );
        pReplyPacket->WriteS32( Result );
    }
    else
    {
        pReplyPacket->WriteS32( tmapi::result::RESULT_OK );
        pReplyPacket->WriteS32( VolumeSerialNumber );
        pReplyPacket->WriteS32( MaximumComponentLength );
        pReplyPacket->WriteS32( FileSystemFlags );
        pReplyPacket->WriteString( pVolumeNameBuffer );
        pReplyPacket->WriteString( pFileSystemNameBuffer );
    }
    m_pServicesManager->Send( pReplyPacket );

    Complete();
}

//==============================================================================

//==============================================================================
// File tasks
//==============================================================================

TIOTaskFileRead::TIOTaskFileRead() : TIOTask()
{
    memset( &m_FileHandle, 0, sizeof(m_FileHandle) );
    m_AmountToRead = 0;
    m_AmountRead = 0;
    m_ReadOffset = 0;
}

//==============================================================================

TIOTaskFileRead::~TIOTaskFileRead()
{
}

//==============================================================================

void TIOTaskFileRead::Finish( tmapi::result Result )
{
    TARGETIO_TRACE( "TIOTaskFileRead::Finish", "Finish - Result = %d", Result );
    nn::dmnt::TargetIO_FileClose( &m_FileHandle );
    SetNeedPackets( false );
    if( Result != tmapi::result::RESULT_OK )
    {
        tmipc::Packet* pPacket = AllocSendPacket();
        pPacket->WriteS32( Result );
        m_pServicesManager->Send( pPacket );
    }
    Complete();
}

//==============================================================================
//Here's the size of the data we send at the start of each file read packet.
enum
{
    SIZE_OF_READ_BUFFER_HEADER = ( sizeof(s32) + sizeof(s32) ),
    SIZE_OF_READ_DATA_BUFFER  = ( (tmipc::MAX_PACKET_DATA) - (SIZE_OF_READ_BUFFER_HEADER) )
};

//==============================================================================

tmapi::result TIOTaskFileRead::HandlePacket( tmipc::Packet* pPacket )
{
    tmapi::result Result = tmapi::result::RESULT_INTERNAL_ERROR;

    s32 ReadThisPass = (s32)(m_AmountToRead > SIZE_OF_READ_DATA_BUFFER ? SIZE_OF_READ_DATA_BUFFER : m_AmountToRead );
    // Get our data
    void* pMem = s_Allocate( ReadThisPass );
    char* pBuffer = new (pMem) char[ReadThisPass];
    if( pBuffer != NULL )
    {
        // Read it
        int32_t NumberOfBytesRead;
        nn::Result res = nn::dmnt::TargetIO_FileRead( &m_FileHandle, pBuffer, ReadThisPass, &NumberOfBytesRead, m_ReadOffset );

        //How'd we do?
        if( res.IsSuccess() )
        {
            pPacket->WriteS32( tmapi::result::RESULT_OK );
            pPacket->WriteS32( NumberOfBytesRead );
            pPacket->WriteData( pBuffer, NumberOfBytesRead );
            m_ReadOffset += NumberOfBytesRead;
            m_AmountToRead -= NumberOfBytesRead;

            Result = tmapi::result::RESULT_OK;
        }

        //Clean up
        s_Deallocate( pBuffer, ReadThisPass );
    }

    return Result;
}

//==============================================================================

void TIOTaskFileRead::OnInitiate( tmipc::Packet* pPacket )
{
    char fileName[nn::tma::target_io::MAX_TARGET_IO_PATH];

    // Read in our particulars
    pPacket->ReadString( fileName, sizeof(fileName), NULL );
    pPacket->ReadS64( m_AmountToRead );
    pPacket->ReadS64( m_ReadOffset );

    TARGETIO_TRACE( "TIOTaskFileRead::OnInitiate", "OnInitiate:  Read %lld from %s", m_AmountToRead, fileName );

    if( strlen(fileName) <= 0 || m_AmountToRead < 0 || m_ReadOffset < 0 )
    {
        OnComplete( tmapi::result::RESULT_INVALID_PARAMETER );
    }
    else
    {
        nn::Result res = nn::dmnt::TargetIO_FileOpen( fileName, nn::fs::OpenMode_Read, (int32_t)nn::tma::target_io::TARGET_IO_OPEN_EXISTING, &m_FileHandle );

        if( res.IsSuccess() )
        {
            tmipc::Packet* p = AllocSendPacket();
            tmapi::result handlePacketRes = HandlePacket( p );
            if( handlePacketRes ==  tmapi::result::RESULT_OK )
            {
                m_pServicesManager->Send( p );
                if( m_AmountToRead > 0 )
                {
                    SetNeedPackets( true );
                }
                else
                {
                    Finish( tmapi::result::RESULT_OK );
                }
            }
            else
            {
                Finish( handlePacketRes );
            }
        }
        else  //Error opening the file
        {
            OnComplete( tmapi::result::RESULT_INVALID_PARAMETER );
        }
    }
}

//==============================================================================

void TIOTaskFileRead::OnSendPacket( tmipc::Packet* pPacket )
{
    TARGETIO_TRACE( "TIOTaskFileRead::OnSendPacket", "OnSendPacket" );
    tmapi::result handlePacketRes = HandlePacket( pPacket );

    if( ( m_AmountToRead <= 0 ) || ( handlePacketRes != tmapi::result::RESULT_OK) )
    {
        Finish( handlePacketRes );
    }
}

//==============================================================================
// File write

TIOTaskFileWrite::TIOTaskFileWrite() : TIOTask()
{
    memset( &m_FileHandle, 0, sizeof(m_FileHandle) );
    m_AmountToWrite = 0;
    m_AmountWritten = 0;
    m_WriteOffset = 0;
#if PROFILE_TIO_TASKS
    m_WorkingTicks = 0;
#endif
}

//==============================================================================

TIOTaskFileWrite::~TIOTaskFileWrite()
{
#if PROFILE_TIO_TASKS
    TARGETIO_LOG( "TIOTaskFileWrite working duration = %lld ticks\n", m_WorkingTicks );
#endif
}

//==============================================================================

tmapi::result TIOTaskFileWrite::HandlePacket( tmipc::Packet* pPacket )
{
    int64_t StartingTick = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
    tmapi::result Result = tmapi::result::RESULT_INTERNAL_ERROR;

    // Get our data
    int32_t WriteThisPass = 0;
    pPacket->ReadS32( WriteThisPass );

    // Write it
    int32_t NumberOfBytesWritten;
    nn::Result res = nn::dmnt::TargetIO_FileWrite( &m_FileHandle, (char*)pPacket->GetCursor(), WriteThisPass, &NumberOfBytesWritten, m_WriteOffset );

    //How'd we do?
    if( res.IsSuccess() )
    {
        m_AmountWritten += NumberOfBytesWritten;
        m_WriteOffset += NumberOfBytesWritten;
        m_AmountToWrite -= NumberOfBytesWritten;
        Result = tmapi::result::RESULT_OK;
    }
    else //    if( res.IsFailure() )
    {
        Result = ConvertToTMAPI( res );
    }

#if PROFILE_TIO_TASKS
    m_WorkingTicks += ( nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - StartingTick );
#endif
    if( Result != tmapi::result::RESULT_OK )
    {
        //Diagnose this error
        nn::dmnt::TargetIO_FileClose( &m_FileHandle );
        OnComplete( Result );
    }
    else if( m_AmountToWrite <= 0 )
    {
        tmipc::Packet* pPacket = AllocSendPacket();
        pPacket->WriteS32( tmapi::result::RESULT_OK );
        pPacket->WriteS32( m_AmountWritten );
        m_pServicesManager->Send( pPacket );
        nn::dmnt::TargetIO_FileClose( &m_FileHandle );
        Complete();
    }

    return Result;
}

//==============================================================================

void TIOTaskFileWrite::OnInitiate( tmipc::Packet* pPacket )
{
    char fileName[nn::tma::target_io::MAX_TARGET_IO_PATH];

    // Read in our particulars
    pPacket->ReadString( fileName, sizeof(fileName), NULL );
    pPacket->ReadS64( m_AmountToWrite );
    pPacket->ReadS64( m_WriteOffset );

    TARGETIO_TRACE( "TIOTaskFileWrite::OnInitiate", "OnInitiate:  Write %lld to %s", m_AmountToWrite, fileName );

    if( strlen(fileName) <= 0 || m_AmountToWrite < 0 || m_WriteOffset < 0 )
    {
        OnComplete( tmapi::result::RESULT_INVALID_PARAMETER );
    }
    else
    {
        //int
        nn::Result res = nn::dmnt::TargetIO_FileOpen( fileName, (nn::fs::OpenMode_Write|nn::fs::OpenMode_AllowAppend), (int32_t)nn::tma::target_io::TARGET_IO_CREATE_ALWAYS, &m_FileHandle );

        if( res.IsSuccess() )
        {
            //HandlePacket( pPacket );
        }
        else  //Error opening the file
        {
            TARGETIO_TRACE( "TIOTaskFileWrite::OnInitiate", "TargetIO_FileOpen FAILED:  %x", res.GetInnerValueForDebug() );
            OnComplete( ConvertToTMAPI( res ) );
        }
    }
}

//==============================================================================

void TIOTaskFileWrite::OnRecvPacket( tmipc::Packet* pPacket )
{
//    TARGETIO_TRACE( "TIOTaskFileWrite::OnRecvPacket", "OnRecvPacket" );

    HandlePacket( pPacket );
}

//==============================================================================
// Set file size

void TIOTaskFileSetSize::OnInitiate( tmipc::Packet* pPacket )
{
    tmapi::result Result = tmapi::result::RESULT_OK;
    char fileName[nn::tma::target_io::MAX_TARGET_IO_PATH];
    pPacket->ReadString( fileName, sizeof(fileName), NULL );
    int64_t size;
    pPacket->ReadS64( size );

    if( size < 0 || strlen(fileName) <= 0 )
    {
        Result = tmapi::result::RESULT_INVALID_PARAMETER;
    }
    else
    {
        nn::Result res = nn::dmnt::TargetIO_FileSetSize( fileName, size );
        Result = ConvertToTMAPI( res );
    }

    OnComplete( Result );
}

//==============================================================================
// Delete file

void TIOTaskFileDelete::OnInitiate( tmipc::Packet* pPacket )
{
    tmapi::result Result = tmapi::result::RESULT_OK;

    char fileName[nn::tma::target_io::MAX_TARGET_IO_PATH];
    pPacket->ReadString( fileName, sizeof(fileName), NULL );

    if( strlen(fileName) <= 0  )
    {
        Result = tmapi::result::RESULT_INVALID_PARAMETER;
    }
    else
    {
        nn::Result res = nn::dmnt::TargetIO_FileDelete( fileName );
        Result = ConvertToTMAPI( res );
    }
    OnComplete( Result );
}

//==============================================================================
// Move file

void TIOTaskFileMove::OnInitiate( tmipc::Packet* pPacket )
{
    tmapi::result Result = tmapi::result::RESULT_OK;

    char fromName[nn::tma::target_io::MAX_TARGET_IO_PATH];
    pPacket->ReadString( fromName, sizeof(fromName), NULL );

    char toName[nn::tma::target_io::MAX_TARGET_IO_PATH];
    pPacket->ReadString( toName, sizeof(toName), NULL );

    if( ( strlen(fromName) <= 0 ) || ( strlen(toName) <= 0 ) )
    {
        Result = tmapi::result::RESULT_INVALID_PARAMETER;
    }
    else
    {
        nn::Result res = nn::dmnt::TargetIO_FileMove( fromName, toName );
        Result = ConvertToTMAPI( res );
    }

    OnComplete( Result );
}

//==============================================================================
// Get file information

void TIOTaskFileGetInformation::OnInitiate( tmipc::Packet* pPacket )
{
    char fileName[nn::tma::target_io::MAX_TARGET_IO_PATH];
    pPacket->ReadString( fileName, sizeof(fileName), NULL );

    if( strlen(fileName) <= 0 )
    {
        OnComplete( tmapi::result::RESULT_INVALID_PARAMETER );
    }
    else
    {
        nn::tma::target_io::FILE_INFORMATION info;
        int32_t IsDirectory;
        nn::Result res = nn::dmnt::TargetIO_FileGetInformation( fileName, &IsDirectory, &info.m_FileSize, &info.m_CreationTime, &info.m_LastAccessTime, &info.m_LastWriteTime );
        if( res.IsFailure() )
        {
            OnComplete( ConvertToTMAPI( res ) );
        }
        else
        {
            tmipc::Packet* pPacket = AllocSendPacket();
            pPacket->WriteS32( tmapi::result::RESULT_OK );
            pPacket->WriteData( &info, sizeof(info) );
            m_pServicesManager->Send( pPacket );
            Complete();
        }
    }
}

//==============================================================================
// Set file attributes

void TIOTaskFileSetAttributes::OnInitiate( tmipc::Packet* pPacket )
{
    tmapi::result Result = tmapi::result::RESULT_OK;
    char fileName[nn::tma::target_io::MAX_TARGET_IO_PATH];
    pPacket->ReadString( fileName, sizeof(fileName), NULL );

    if( strlen(fileName) <= 0 )
    {
        Result = tmapi::result::RESULT_INVALID_PARAMETER;
    }
    else
    {
        nn::tma::target_io::FileAttributes attributes;
        pPacket->ReadData( &attributes, sizeof(attributes) );
        nn::Result res = nn::dmnt::TargetIO_FileSetAttributes( fileName, &attributes, sizeof(s32) );
        Result = ConvertToTMAPI( res );
    }

    OnComplete( Result );
}

//==============================================================================
// Set file time

void TIOTaskFileSetTime::OnInitiate( tmipc::Packet* pPacket )
{
    tmapi::result Result = tmapi::result::RESULT_OK;
    char fileName[nn::tma::target_io::MAX_TARGET_IO_PATH];
    pPacket->ReadString( fileName, sizeof(fileName), NULL );

    if( strlen(fileName) <= 0 )
    {
        Result = tmapi::result::RESULT_INVALID_PARAMETER;
    }
    else
    {
        uint64_t createTime;
        uint64_t accessTime;
        uint64_t modifyTime;
        pPacket->ReadU64( createTime );
        pPacket->ReadU64( accessTime );
        pPacket->ReadU64( modifyTime );

        nn::Result res = nn::dmnt::TargetIO_FileSetTime( fileName, createTime, accessTime, modifyTime );
        Result = ConvertToTMAPI( res );
    }

    OnComplete( Result );
}

//==============================================================================
// Directory tasks
//==============================================================================

void TIOTaskDirectoryCreate::OnInitiate( tmipc::Packet* pPacket )
{
    tmapi::result Result = tmapi::result::RESULT_OK;
    char pathName[nn::tma::target_io::MAX_TARGET_IO_PATH];
    pPacket->ReadString( pathName, sizeof(pathName), NULL );

    if( strlen( pathName ) <= 0 )
    {
        Result = tmapi::result::RESULT_INVALID_PARAMETER;
    }
    else
    {
        u32 access;
        pPacket->ReadU32( access );

        nn::Result res = nn::dmnt::TargetIO_DirectoryCreate( pathName, (nn::fs::OpenMode)access );
        Result = ConvertToTMAPI( res );
    }
    OnComplete( Result );
}

//==============================================================================
// Delete directory

void TIOTaskDirectoryDelete::OnInitiate( tmipc::Packet* pPacket )
{
    tmapi::result Result = tmapi::result::RESULT_OK;
    char pathName[nn::tma::target_io::MAX_TARGET_IO_PATH];
    pPacket->ReadString( pathName, sizeof(pathName), NULL );

    if( strlen( pathName ) <= 0 )
    {
        Result = tmapi::result::RESULT_INVALID_PARAMETER;
    }
    else
    {
        nn::Result res = nn::dmnt::TargetIO_DirectoryDelete( pathName );
        Result = ConvertToTMAPI( res );
    }

    OnComplete( Result );
}

//==============================================================================
// Move directory

void TIOTaskDirectoryMove::OnInitiate( tmipc::Packet* pPacket )
{
    tmapi::result Result = tmapi::result::RESULT_OK;

    char fromName[nn::tma::target_io::MAX_TARGET_IO_PATH];
    pPacket->ReadString( fromName, sizeof(fromName), NULL );

    char toName[nn::tma::target_io::MAX_TARGET_IO_PATH];
    pPacket->ReadString( toName, sizeof(toName), NULL );

    if( ( strlen(fromName) <= 0 ) || ( strlen(toName) <= 0 ) )
    {
        Result = tmapi::result::RESULT_INVALID_PARAMETER;
    }
    else
    {
        nn::Result res = nn::dmnt::TargetIO_DirectoryRename( fromName, toName );
        Result = ConvertToTMAPI( res );
    }

    OnComplete( Result );
}

//==============================================================================
// Get directory information

void TIOTaskDirectoryGetInformation::OnInitiate( tmipc::Packet* pPacket )
{
    char path[nn::tma::target_io::MAX_TARGET_IO_PATH];
    pPacket->ReadString( path, sizeof(path), NULL );

    if( strlen( path ) <= 0 )
    {
        OnComplete( tmapi::result::RESULT_INVALID_PARAMETER );
    }
    else
    {
        nn::tma::target_io::FILE_INFORMATION info;
        int32_t IsDirectory;
        nn::Result res = nn::dmnt::TargetIO_FileGetInformation( path, &IsDirectory, &info.m_FileSize, &info.m_CreationTime, &info.m_LastAccessTime, &info.m_LastWriteTime );
        tmapi::result Result = ConvertToTMAPI( res );

        if( IsDirectory )
        {
            info.m_FileAttributes = nn::tma::target_io::TARGET_IO_FILE_ATTRIBUTE_DIRECTORY;
        }
        tmipc::Packet* pPacket = AllocSendPacket();
        pPacket->WriteS32( Result );
        if( Result == tmapi::result::RESULT_OK )
        {
            pPacket->WriteData( &info, sizeof(info) );
        }
        m_pServicesManager->Send( pPacket );
        Complete();
    }
}

//==============================================================================
// Search task
//==============================================================================

static void shimEntry( nn::fs::DirectoryEntry* pFrom, nn::tma::target_io::DIRECTORY_ENTRY* pTo )
{
    if( pFrom->directoryEntryType == nn::fs::DirectoryEntryType_Directory )
    {
        pTo->m_FileAttributes = nn::tma::target_io::TARGET_IO_FILE_ATTRIBUTE_DIRECTORY;
    }
    else  //pFrom->directoryEntryType == nn::fs::DirectoryEntryType_File
    {
        pTo->m_FileAttributes = nn::tma::target_io::TARGET_IO_FILE_ATTRIBUTE_NORMAL;
    }

    pTo->m_FileSize = pFrom->fileSize;
    memcpy( pTo->m_FileName, pFrom->name, sizeof(pTo->m_FileName) );
}

//==============================================================================

void TIOTaskSearchFetch::OnInitiate( tmipc::Packet* pPacket )
{
    tmapi::result Result = tmapi::result::RESULT_OK;
    int32_t NumberOfEntriesRead = 0;

    char searchDir[nn::tma::target_io::MAX_TARGET_IO_PATH];
    pPacket->ReadString( searchDir, sizeof(searchDir), NULL );

    //==================================================================
    // First, see if there's anything here.
    int32_t NumberOfEntries;
    nn::Result res = nn::dmnt::TargetIO_DirectoryGetCount( searchDir, &NumberOfEntries ) ;
    tmipc::Packet* pReplyPacket = AllocSendPacket();

    if( res.IsSuccess() == false || NumberOfEntries == 0 )
    {
        if( res.IsSuccess() )
        {
            Result = tmapi::result::RESULT_TARGET_IO_SEARCH_COMPLETE;
        }
        else
        {
            Result = ConvertToTMAPI( res );
        }

        pReplyPacket->WriteS32( Result );
        m_pServicesManager->Send( pReplyPacket );

        TARGETIO_TRACE( "TIOTaskSearchFetch::OnInitiate", "COMPLETE - Sent Packet for result=FALSE or NumberOfEntries=0 for %s", searchDir );
        Complete();
    }

    else
    {
        //=========================================================================
        //  Open the directory and start our search.
        res = nn::dmnt::TargetIO_DirectoryOpen( searchDir, &m_DirHandle );
        if( res.IsSuccess() )
        {
            bool RepyRes = HandleReplyPacket( pReplyPacket );

            m_pServicesManager->Send( pReplyPacket );

            // Do we need to write more?
            if( RepyRes == false )
            {
                TARGETIO_TRACE( "TIOTaskSearchFetch::OnInitiate", "COMPLETE - HandleReplyPacket returned FALSE for %s", searchDir );
                Complete();
            }
            else
            {
                SetNeedPackets();
            }
        }
        else  //Error opening the directory
        {
            Result = ConvertToTMAPI( res );
            pReplyPacket->WriteS32( Result );
            m_pServicesManager->Send( pReplyPacket );
            TARGETIO_TRACE( "TIOTaskSearchFetch::OnInitiate", "COMPLETE - Error opening directory %s", searchDir );
            Complete();
        }
    }
}

//==============================================================================

bool TIOTaskSearchFetch::HandleReplyPacket( tmipc::Packet* pPacket )
{
//    bool Ret = true;
    nn::fs::DirectoryEntry entry;
    nn::Result searchRes = nn::dmnt::TargetIO_DirectoryGetNext( &m_DirHandle, &entry );
    if( searchRes.IsSuccess() == false )
    {
        //TARGETIO_TRACE( "TIOTaskSearchFetch::HandleReplyPacket", "TargetIO_DirectoryGetNext returned %x", searchRes.GetInnerValueForDebug() );
        // Close this
        nn::dmnt::TargetIO_DirectoryClose( &m_DirHandle );
        pPacket->WriteS32( tmapi::result::RESULT_TARGET_IO_SEARCH_COMPLETE );
    }
    else
    {
        pPacket->WriteS32( tmapi::result::RESULT_OK );
        nn::tma::target_io::DIRECTORY_ENTRY writeEntry;
        shimEntry( &entry, &writeEntry );
        pPacket->WriteData( &writeEntry, sizeof(writeEntry) );
        //TARGETIO_TRACE( "TIOTaskSearchFetch::HandleReplyPacket", "TargetIO_DirectoryGetNext found %s", entry.name );
    }

    return ( searchRes.IsSuccess() );
}

//==============================================================================

void TIOTaskSearchFetch::OnSendPacket( tmipc::Packet* pPacket )
{
    if( HandleReplyPacket( pPacket ) == false )
    {
        //TARGETIO_TRACE( "TIOTaskSearchFetch::OnSendPacket", "COMPLETE - HandleReplyPacket returned FALSE" );
        Complete();
    }
}

//==============================================================================


//==============================================================================
}} // namespace
//==============================================================================
