﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "settings_task.h"
#include <nn/nn_SdkLog.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/settings/system/settings_FirmwareVersion.h>

//==============================================================================
namespace tma { namespace settings {
//==============================================================================

//==============================================================================

Task::Task()
: m_Success( false )
, m_TaskMode( Mode::UNKNOWN )
{
    m_TaskType = tmipc::TaskType_SettingsTask;
    memset( m_Category, '\0', nn::settings::fwdbg::SettingsNameLengthMax );
    memset( m_Setting, '\0', nn::settings::fwdbg::SettingsNameLengthMax );
    memset( m_Value, '\0', nn::settings::fwdbg::SettingsNameLengthMax );
}

//==============================================================================

Task::~Task()
{
}

//==============================================================================

void Task::OnInitiate( tmipc::Packet* pPacket )
{
    s32 Length = 0;
    s32 Mode = (s32)tma::settings::Mode::UNKNOWN;
    pPacket->ReadS32( Mode );
    pPacket->ReadString( m_Category, nn::settings::fwdbg::SettingsNameLengthMax, &Length );
    pPacket->ReadString( m_Setting, nn::settings::fwdbg::SettingsNameLengthMax, &Length );
    m_TaskMode = static_cast<tma::settings::Mode>( Mode );

    nn::settings::system::FirmwareVersion version;
    nn::settings::system::GetFirmwareVersion(&version);
    // Info: If adding the operation toward the new value of fwdbg settings,
    //       it is necessary to diverge by firmware version.
    if (version.major <= 1)
    {
        m_Success = false;
        return;
    }

    switch( m_TaskMode )
    {
        case tma::settings::Mode::SET:
            pPacket->ReadData( m_Value, nn::settings::fwdbg::SettingsNameLengthMax );
            HandleSet();
            break;
        case tma::settings::Mode::GET:
            HandleGet();
            break;
        default:
            break;
    }
}

//==============================================================================

void Task::OnRecvPacket( tmipc::Packet* pPacket )
{
    ASSERT(0);
}

//==============================================================================

void Task::OnSendPacket( tmipc::Packet* pPacket )
{
    pPacket->WriteS32( (s32)m_TaskMode );
    pPacket->WriteU8( m_Success ); // operation status
    switch( m_TaskMode )
    {
        case Mode::GET:
            pPacket->WriteData( m_Value, nn::settings::fwdbg::SettingsNameLengthMax );
            break;
        default:
            break;
    }

    Complete();
}

//==============================================================================

void Task::HandleSet()
{
    u8 Temp[nn::settings::fwdbg::SettingsNameLengthMax];
    memset( Temp, '\0', nn::settings::fwdbg::SettingsNameLengthMax );
    nn::settings::fwdbg::SetSettingsItemValue( m_Category, m_Setting, m_Value, nn::settings::fwdbg::SettingsNameLengthMax );
    nn::settings::fwdbg::GetSettingsItemValue( Temp, nn::settings::fwdbg::SettingsNameLengthMax, m_Category, m_Setting );
    m_Success = true;

    s32 iCount = nn::settings::fwdbg::SettingsNameLengthMax;
    while( iCount-- )
    {
        if( m_Value[iCount] != Temp[iCount] )
        {
            m_Success = false;
        }
    }
}

//==============================================================================

void Task::HandleGet()
{
    m_Success = nn::settings::fwdbg::GetSettingsItemValue( m_Value, nn::settings::fwdbg::SettingsNameLengthMax, m_Category, m_Setting ) > 0;
}

//==============================================================================
}}
//==============================================================================
