﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\tmagent.h"
#include "htcs_task.h"
#include "htcs_service.h"
#include "htcs_opcodes.h"

//==============================================================================
namespace tma { namespace htcs {
//==============================================================================

enum
{
    MaxNumberOfSystemEventsToAllow = 32,
    MaxNumberOfSocketsToCloseOnConnect = 256
};

#define REPORT_ON_TASK_LIST TMA_MACRO_VALUE(0)

#if REPORT_ON_TASK_LIST
#define TASK_LIST_REPORT(...) NN_SDK_LOG( "[tma][waiting_task] - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )
#else
#define TASK_LIST_REPORT(...)
#endif

#define REPORT_ON_EVENTS TMA_MACRO_VALUE(0)

#if REPORT_ON_EVENTS
#define EVENT_REPORT(...) NN_SDK_LOG( "[tma][events] - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )
#else
#define EVENT_REPORT(...)
#endif

//==============================================================================
// Waiting tasks.  Task information we need to track.
//==============================================================================

struct waiting_task
{
    waiting_task*    m_pNext;
    int             m_Socket;
    uint32_t        m_Id;

    //==============================================================================

    explicit waiting_task( HTCSTask* pTask )
    {
        m_Id = pTask->GetTaskId();
        m_Socket = pTask->GetSocket();
        m_pNext = NULL;
        TASK_LIST_REPORT("%d created", m_Id );
    }

    virtual ~waiting_task()
    {
        TASK_LIST_REPORT(" %d deleted", m_Id );
    }

    //==============================================================================

    uint32_t GetId()
    {
        return m_Id;
    }

    //==============================================================================

    virtual int GetSize()
    {
        return sizeof( waiting_task );
    }

    //==============================================================================

    bool Is( uint32_t TaskId )
    {
        return (m_Id == TaskId);
    }

    //==============================================================================

    bool IsSocket( int descriptor )
    {
        return ( m_Socket == descriptor );
    }

    //==============================================================================

    void Report( int AtIndex )
    {
        if( AtIndex > 0 )
        {
            NN_SDK_LOG(", " );
        }
        NN_SDK_LOG("%d", m_Id );
    }

    //==============================================================================
    // Static list maintenance functionality
    //==============================================================================

    static uint32_t add( waiting_task* pAddTask, waiting_task** pAddToList )
    {
        //waiting_task* pRet = new waiting_task( pTask );
        waiting_task* pList = *pAddToList;
        if( pList == NULL )
        {
            *pAddToList = pAddTask;
        }
        else
        {
            while( pList->m_pNext != NULL )
            {
                pList = pList->m_pNext;
            }
            pList->m_pNext = pAddTask;
        }
        return pAddTask->GetId();
    }

    //==============================================================================

    static waiting_task* find( uint32_t Find, waiting_task* pList )
    {
        while( pList != NULL )
        {
            if( pList->Is( Find ) )
            {
                break;
            }
            pList = pList->m_pNext;
        }

        return pList;
    }

    //==============================================================================

    static waiting_task* find( tmipc::Task* pTask, waiting_task* pList )
    {
        return find( pTask->GetTaskId(), pList );
    }

    //==============================================================================

    static waiting_task* findBySocket( int descriptor, waiting_task* pList )
    {
        while( pList != NULL )
        {
            if( pList->IsSocket( descriptor ) )
            {
                break;
            }
            pList = pList->m_pNext;
        }

        return pList;
    }

    //==============================================================================

    static void remove( waiting_task* pTask, waiting_task** pFromList )
    {
        waiting_task* pList = *pFromList;
        if( pList == pTask )
        {
            *pFromList = pTask->m_pNext;
        }
        else
        {
            waiting_task* pPrev = pList;
            pList = pList->m_pNext;
            while( pList != pTask )
            {
                pPrev = pList;
                pList = pList->m_pNext;
            }
            if( pPrev != NULL )
            {
                pPrev->m_pNext = pList->m_pNext;
            }
        }

        delete pTask;
    }

    //==============================================================================

    static void clear( waiting_task** pClearList )
    {
        waiting_task* pList = *pClearList;
        while( pList != NULL )
        {
            waiting_task* pNext = pList->m_pNext;
            delete pList;
            pList = pNext;
        }

        *pClearList = NULL;
    }

    //==============================================================================

    static void report( const char* pTag, waiting_task* pList )
    {
#if REPORT_ON_TASK_LIST
        NN_SDK_LOG("[tma] [waiting_task] [%s]: ", pTag );
        int TaskCount = 0;
        while( pList != NULL )
        {
            pList->Report( TaskCount );
            TaskCount += 1;
            pList = pList->m_pNext;
        }
        NN_SDK_LOG(" (%d tasks)\n", TaskCount );
#else
        NN_UNUSED( pTag );
        NN_UNUSED( pList );
#endif
    }

    //==============================================================================

    static void memory_use( waiting_task* pList, int* pNumberOfTasks, int* pSize )
    {
        *pNumberOfTasks = *pSize = 0;
        while( pList != NULL )
        {
            *pSize += pList->GetSize();
            *pNumberOfTasks += 1;
            pList = pList->m_pNext;
        }
    }

    //==============================================================================
};

//==============================================================================

struct accept_waiting_task : waiting_task
{
    nn::htcs::SockAddrHtcs  m_AcceptedAddr;
    int                     m_Results;
    int                     m_LastError;

    explicit accept_waiting_task( HTCSAcceptTask* pTask ) : waiting_task( pTask )
    {
        m_Results = -1;
        m_LastError = nn::htcs::HTCS_EINTR;
        memset( &m_AcceptedAddr, 0, sizeof(m_AcceptedAddr) );
    }

    virtual ~accept_waiting_task()
    {
        TASK_LIST_REPORT("accept_waiting_task %d deleted", m_Id );
    }

    void* operator new(std::size_t size)
    {
        return s_Allocate(size);
    }

    void operator delete(void* p, size_t size) NN_NOEXCEPT
    {
        return s_Deallocate(p, size);
    }

    int GetResults( nn::htcs::SockAddrHtcs* pAddr, int* pError )
    {
        memcpy( pAddr, &m_AcceptedAddr, sizeof(m_AcceptedAddr) );
        *pError = m_LastError;
        return m_Results;
    }

    void OnComplete( HTCSAcceptTask* pCompletedTask )
    {
        m_Results = pCompletedTask->GetResults( &m_AcceptedAddr, &m_LastError );
    }
};

//==============================================================================

struct recv_waiting_task : waiting_task
{
    void*   m_Buffer;
    int64_t m_AmountRead;
    int     m_Results;
    int     m_LastError;
    int32_t m_PreRead;
    int32_t m_PostRead;
    bool    m_MappedMemoryRecv;

    explicit recv_waiting_task( HTCSRecvTask* pTask ) : waiting_task( pTask )
    {
        m_Results = -1;
        m_LastError = nn::htcs::HTCS_EINTR;
        m_Buffer = NULL;
        m_AmountRead = 0;
        m_PreRead = 0;
        m_PostRead = 0;
        m_MappedMemoryRecv = false;
    }

    explicit recv_waiting_task( HTCSRecvLargeTask* pTask ) : waiting_task( pTask )
    {
        m_Results = -1;
        m_LastError = nn::htcs::HTCS_EINTR;
        m_Buffer = NULL;
        m_AmountRead = 0;
        m_PreRead = 0;
        m_PostRead = 0;
        m_MappedMemoryRecv = true;
    }

    virtual ~recv_waiting_task()
    {
        if( m_Buffer != NULL )
        {
            size_t AllocatedSize = (size_t)m_AmountRead;

            // Was this a mapped-memory recv?
            if( m_MappedMemoryRecv )
            {
                AllocatedSize = m_PreRead + m_PostRead;
            }

            TASK_LIST_REPORT("recv_waiting_task %d deleted (%d bytes)", m_Id, AllocatedSize );
            s_Deallocate( m_Buffer, AllocatedSize );
        }
    }

    virtual int GetSize()
    {
        return ( (int)sizeof( recv_waiting_task ) + (int32_t)m_AmountRead + m_PreRead + m_PostRead );
    }

    int GetResults( void* pBuffer, size_t bufferByteSize, int64_t* pReceivedSize, int* pError )
    {
        // Was this a mapped-memory recv?
        if( m_MappedMemoryRecv )
        {
            // We just have to fix up the unaligned portions of the memory.
            if( m_PreRead > 0 )
            {
                memcpy( pBuffer, m_Buffer, m_PreRead );
            }

            if( m_PostRead > 0 )
            {
                int64_t AlignedAmountRead = m_AmountRead - ( m_PreRead + m_PostRead );
                int64_t Offset = m_PreRead + AlignedAmountRead;
                void* pWriteTo = (void*)((int64_t)pBuffer + Offset );
                void* pReadFrom = (void*)((int64_t)m_Buffer + m_PreRead );
                memcpy( pWriteTo, pReadFrom, m_PostRead );
            }
        }
        else
        {
            int CopySize = (int) ( ( m_AmountRead < (int64_t) bufferByteSize ) ? m_AmountRead : bufferByteSize );
            memcpy( pBuffer, m_Buffer, CopySize );
        }

        *pReceivedSize = m_AmountRead;
        *pError = m_LastError;
        return m_Results;
    }

    void OnComplete( HTCSRecvTask* pCompletedTask )
    {
        m_Results = pCompletedTask->GetResults( &m_Buffer, &m_AmountRead, &m_LastError );
    }

    void OnComplete( HTCSRecvLargeTask* pCompletedTask )
    {
        m_MappedMemoryRecv = true;
        m_Results = pCompletedTask->GetResults( &m_Buffer, &m_PreRead, &m_PostRead, &m_AmountRead, &m_LastError );
    }
};

//==============================================================================

struct send_waiting_task : waiting_task
{
    nn::htcs::ssize_t       m_Results;
    int                     m_LastError;

    explicit send_waiting_task( HTCSSendTask* pTask ) : waiting_task( pTask )
    {
        m_Results = -1;
        m_LastError = nn::htcs::HTCS_EINTR;
    }

    virtual ~send_waiting_task()
    {
    }

    virtual int GetSize()
    {
        return sizeof( send_waiting_task );
    }

    nn::htcs::ssize_t GetResults( int* pError )
    {
        *pError = m_LastError;
        return m_Results;
    }

    void OnComplete( HTCSSendTask* pCompletedTask )
    {
        m_Results = (nn::htcs::ssize_t)pCompletedTask->GetResults( &m_LastError );
    }
};


//==============================================================================
// Task list maintenance functionality
//==============================================================================

uint32_t AgentHTCSService::RememberTask( waiting_task* pTask )
{
    ScopedLock Lock( m_TaskMutex );
    uint32_t Ret = waiting_task::add( pTask, &m_pWaitingTasks );
    waiting_task::report( "remember", m_pWaitingTasks );
    return Ret;
}

//==============================================================================

waiting_task* AgentHTCSService::RecallTask( uint32_t TaskId )
{
    ScopedLock Lock( m_TaskMutex );
    waiting_task* pRet = waiting_task::find( TaskId, m_pWaitingTasks );
    if( pRet == NULL )
    {
        TASK_LIST_REPORT("[tma] AgentHTCSService::RecallTask UNABLE TO FIND task %d.", TaskId );
    }

    return pRet;
}

//==============================================================================

void AgentHTCSService::ForgetTask( uint32_t TaskId )
{
    ScopedLock Lock( m_TaskMutex );
    waiting_task* pWaiting = waiting_task::find( TaskId, m_pWaitingTasks );
    if( pWaiting != NULL )
    {
        TASK_LIST_REPORT("AgentHTCSService::ForgetTask deleting task %d", TaskId );
        waiting_task::remove( pWaiting, &m_pWaitingTasks );
        waiting_task::report( "forget", m_pWaitingTasks );
    }
    else
    {
        TASK_LIST_REPORT("[tma] AgentHTCSService::ForgetTask UNABLE TO FIND task %d.", TaskId );
    }
}

//==============================================================================

void AgentHTCSService::OnClose( int descriptor )
{
    ScopedLock Lock( m_TaskMutex );
    waiting_task* pWaiting = waiting_task::findBySocket( descriptor, m_pWaitingTasks );
    while( pWaiting != NULL )
    {
        if( m_pServicesManager != NULL )
        {
            m_pServicesManager->CancelTask( pWaiting->m_Id );
        }
        TASK_LIST_REPORT("AgentHTCSService::close removing task %d", pWaiting->m_Id );
        waiting_task::remove( pWaiting, &m_pWaitingTasks );
        waiting_task::report( "close", m_pWaitingTasks );
        pWaiting = waiting_task::findBySocket( descriptor, m_pWaitingTasks );
    }
}

//==============================================================================

//==============================================================================

AgentHTCSService::AgentHTCSService()
{
    DEJA_TRACE( "AgentHTCSService::AgentHTCSService", "AgentHTCSService" );
    m_ServiceId = HashString( "HTCSService" );
    m_pWaitingTasks = NULL;
    m_TaskMutex.Create();
    m_NumberOfEventsCreated = 0;
    m_EventCountMutex.Create();
    m_pCloseOnConnectSockets = (int32_t*)s_Allocate( MaxNumberOfSocketsToCloseOnConnect * sizeof( int32_t ) );
    memset( m_pCloseOnConnectSockets, -1, MaxNumberOfSocketsToCloseOnConnect * sizeof( int32_t ) );
    m_NumberOfSocketsToCloseOnConnect = 0;
    m_CloseOnConnectMutex.Create();
}

//==============================================================================

AgentHTCSService::~AgentHTCSService()
{
    DEJA_TRACE( "AgentHTCSService::~AgentHTCSService", "~AgentHTCSService" );
    m_TaskMutex.Destroy();
    m_EventCountMutex.Destroy();
    m_CloseOnConnectMutex.Destroy();
    s_Deallocate( m_pCloseOnConnectSockets, MaxNumberOfSocketsToCloseOnConnect * sizeof( int32_t ) );
}

//==============================================================================

void AgentHTCSService::Init( )
{
    DEJA_TRACE( "AgentHTCSService::Init", "Init" );
    Create();
}

//==============================================================================

void AgentHTCSService::Kill()
{
    DEJA_TRACE( "AgentHTCSService::Kill", "Kill" );
    {
        ScopedLock Lock( m_TaskMutex );
        waiting_task::clear( &m_pWaitingTasks );
    }

    Destroy();
}

//==============================================================================

void AgentHTCSService::MemoryReport()
{
    int NumberOfTasks = 0;
    int TaskSize = 0;
    {
        ScopedLock Lock( m_TaskMutex );
        waiting_task::memory_use( m_pWaitingTasks, &NumberOfTasks, &TaskSize );
    }
    NN_SDK_LOG( "[tma] Memory Report:  htcs_service %d (%d active signal tasks)\n",  TaskSize, NumberOfTasks );
}

//==============================================================================
// Event monitoring.

void AgentHTCSService::OnSignalEventCreated()
{
    ScopedLock Lock( m_EventCountMutex );
    m_NumberOfEventsCreated += 1;
    EVENT_REPORT("%d system events allocated (OnSignalEventCreated)", m_NumberOfEventsCreated );
}

//==============================================================================

void AgentHTCSService::OnSignalEventDestroyed()
{
    ScopedLock Lock( m_EventCountMutex );
    m_NumberOfEventsCreated -= 1;
    if( m_NumberOfEventsCreated < 0 )
    {
        m_NumberOfEventsCreated = 0;
    }
    EVENT_REPORT( "%d system events allocated (OnSignalEventDestroyed)", m_NumberOfEventsCreated );
}

//==============================================================================

bool AgentHTCSService::EventAvailable()
{
    ScopedLock Lock( m_EventCountMutex );
    return (m_NumberOfEventsCreated < MaxNumberOfSystemEventsToAllow);
}

//==============================================================================

void AgentHTCSService::OnSocketCloseFailed( int descriptor )
{
    //NN_SDK_LOG( "[tma][htcs] OnSocketCloseFailed - %d\n", descriptor );
    if( descriptor >= 0 )
    {
        ScopedLock Lock( m_CloseOnConnectMutex );
        if( m_NumberOfSocketsToCloseOnConnect < MaxNumberOfSocketsToCloseOnConnect )
        {
            //=================================================================================
            // The close socket operation only fails if we lose connection to target manager.
            // Because nn::htcs::close is an "always succeed" call by design, we must remember
            // to delete this guy next time we get the opportunity.
            //=================================================================================
            // Don't need to duplicate anyone.
            for( int32_t Index = 0; Index < m_NumberOfSocketsToCloseOnConnect; Index += 1 )
            {
                if( m_pCloseOnConnectSockets[Index] == descriptor )
                {
                    return;
                }
            }

            m_pCloseOnConnectSockets[m_NumberOfSocketsToCloseOnConnect] = descriptor;
            m_NumberOfSocketsToCloseOnConnect += 1;
        }
    }
}

//==============================================================================

void AgentHTCSService::OnSocketCreated( int descriptor )
{
    NN_UNUSED( descriptor );

    ScopedLock Lock( m_CloseOnConnectMutex );
    if( m_NumberOfSocketsToCloseOnConnect )
    {
        //=================================================================================================
        // We support "virtual sockets" in HTCS, which hold their own until a socket is required.  At that
        // point, libnn_htcs waits until it can successfully create a socket.  So if we've successfully
        // created a socket, we're connected to target manager.  Go through any socket closes that were
        // cancelled (due to sleep or disconnection) and close them.
        //=================================================================================================
        for( int32_t Index = 0; Index < m_NumberOfSocketsToCloseOnConnect; Index += 1 )
        {
            if( m_pCloseOnConnectSockets[Index] >= 0 )
            {
                //NN_SDK_LOG( "[tma][htcs] AgentHTCSService OnSocketCreated() closing socket %d\n", descriptor );
                HTCSCloseTask* pRet = CreateTask<HTCSCloseTask>( m_pCloseOnConnectSockets[Index] );

                // Let the system clean this guy up when he's finished.
                pRet->SetExternallyOwned( false );
                pRet->Close();

                // Init this slot for next time.
                m_pCloseOnConnectSockets[Index] = -1;
            }
        }
        m_NumberOfSocketsToCloseOnConnect = 0;
    }
}

//==============================================================================

tmipc::Task* AgentHTCSService::OnNewTask( tmipc::Packet* pPacket )
{
    (void)pPacket;
    DEJA_TRACE( "AgentHTCSService::OnNewTask", "OnNewTask" );
    TMA_PRINTF( "AgentHTCSService::OnNewTask - unhandled\n" );

    //TODO:
    return NULL;
}

//==============================================================================

template <class T> T* AgentHTCSService::CreateTask( int Socket )
{
    DEJA_TRACE( "AgentHTCSService::CreateTask", "CreateTask" );
    void* pMem = s_Allocate( sizeof( T ) );
    T* pRet = new (pMem) T( Socket );
    pRet->SetServicesManager( m_pServicesManager );
    pRet->SetServiceId( m_ServiceId );
    pRet->SetTaskId( m_pServicesManager->AllocTaskId() );

    return pRet;
}

//==============================================================================

HTCSSocketTask* AgentHTCSService::Socket()
{
    DEJA_TRACE( "AgentHTCSService::Socket", "Socket" );
    HTCSSocketTask* pRet = CreateTask<HTCSSocketTask>( -1 );
    pRet->Socket();
    return pRet;
}

//==============================================================================

void AgentHTCSService::OnSocketClosed( int descriptor )
{
    TASK_LIST_REPORT("AgentHTCSService::OnSocketClosed socket %d", descriptor );
    OnClose( descriptor );
}

//==============================================================================

HTCSCloseTask* AgentHTCSService::Close( int descriptor )
{
    TASK_LIST_REPORT("AgentHTCSService::Close socket %d", descriptor );
    DEJA_TRACE( "AgentHTCSService::Close", "Close socket %d", descriptor );
    HTCSCloseTask* pRet = CreateTask<HTCSCloseTask>( descriptor );
    pRet->Close();

    return pRet;
}

//==============================================================================

HTCSConnectTask* AgentHTCSService::Connect( int descriptor, const nn::htcs::SockAddrHtcs* pAddr )
{
    DEJA_TRACE( "AgentHTCSService::Connect", "Connect socket %d", descriptor );
    HTCSConnectTask* pRet = CreateTask<HTCSConnectTask>( descriptor );
    pRet->Connect( pAddr );
    return pRet;
}

//==============================================================================

HTCSBindTask* AgentHTCSService::Bind( int descriptor, const nn::htcs::SockAddrHtcs*  pAddr )
{
    DEJA_TRACE( "AgentHTCSService::Bind", "Bind socket %d", descriptor );
    HTCSBindTask* pRet = CreateTask<HTCSBindTask>( descriptor );
    pRet->Bind( pAddr );
    return pRet;
}

//==============================================================================

HTCSListenTask* AgentHTCSService::Listen( int descriptor, int backlogCount )
{
    DEJA_TRACE( "AgentHTCSService::Listen", "Listen socket %d", descriptor );
    HTCSListenTask* pRet = CreateTask<HTCSListenTask>( descriptor );
    pRet->Listen( backlogCount );
    return pRet;
}

//==============================================================================

DeprecatedAcceptTask* AgentHTCSService::Accept( int descriptor, nn::htcs::SockAddrHtcs* pAddr )
{
    DEJA_TRACE( "AgentHTCSService::Accept", "Accept socket %d", descriptor );
    DeprecatedAcceptTask* pRet = CreateTask<DeprecatedAcceptTask>( descriptor );
    pRet->Accept( pAddr );
    //TMA_PRINTF( "AgentHTCSService::Accept socket %d\n", descriptor );
    return pRet;
}

//==============================================================================

DeprecatedRecvTask* AgentHTCSService::Recv( int descriptor, void* pBuffer, size_t bufferByteSize, int flags )
{
    DEJA_TRACE( "AgentHTCSService::Recv", "Recv socket %d", descriptor );
    DeprecatedRecvTask* pRet = CreateTask<DeprecatedRecvTask>( descriptor );
    pRet->Recv( pBuffer, bufferByteSize, flags );
    return pRet;
}

//==============================================================================

DeprecatedSendTask* AgentHTCSService::Send( int descriptor, const void* pBuffer, size_t bufferByteSize, int flags )
{
    DEJA_TRACE( "AgentHTCSService::Send", "Send socket %d", descriptor );
    DeprecatedSendTask* pRet = CreateTask<DeprecatedSendTask>( descriptor );
    pRet->Send( pBuffer, bufferByteSize, flags );
    return pRet;
}

//==============================================================================

int AgentHTCSService::AcceptStart( int descriptor, nn::os::NativeHandle* pWaitHandle, uint32_t* pTaskId )
{
    if( EventAvailable() == false )
    {
        EVENT_REPORT( "AcceptStart FAILED due to too many system events allocated" );
        return -1;
    }

    HTCSAcceptTask* pNewTask = CreateTask<HTCSAcceptTask>( descriptor );
    OnSignalEventCreated();
    accept_waiting_task* pSave = new accept_waiting_task( pNewTask );
    RememberTask( pSave );
    *pTaskId = pNewTask->GetTaskId();
    pNewTask->Start( pWaitHandle );
    return 0;
}

//==============================================================================

int AgentHTCSService::AcceptResults( uint32_t TaskId, nn::htcs::SockAddrHtcs* pAddress, int* pLastError )
{
    // Set these as defaults:
    int Ret = -1;
    *pLastError = nn::htcs::HTCS_EINTR;
    {
        ScopedLock Lock( m_TaskMutex );
        accept_waiting_task* pTask = (accept_waiting_task*)waiting_task::find( TaskId, m_pWaitingTasks );
        if( pTask != NULL )
        {
            Ret = pTask->GetResults( pAddress, pLastError );
            waiting_task::remove( pTask, &m_pWaitingTasks );
        }
    }

    TASK_LIST_REPORT("AgentHTCSService::AcceptResults returned %d from task %d.", Ret, TaskId );

    return Ret;
}

//==============================================================================

void AgentHTCSService::OnAcceptComplete( HTCSAcceptTask* pTask )
{
    accept_waiting_task* pTaskResults = (accept_waiting_task*)RecallTask( pTask->GetTaskId() );
    if( pTaskResults != NULL )
    {
        pTaskResults->OnComplete( pTask );
        TASK_LIST_REPORT("AgentHTCSService::OnAcceptComplete caching results from task %d.", pTask->GetTaskId() );
    }
    else
    {
        TASK_LIST_REPORT("[tma] AgentHTCSService::OnAcceptComplete FAILED to cache results from task %d.", pTask->GetTaskId() );
    }
}

//==============================================================================

int AgentHTCSService::RecvStart( int descriptor, int64_t bufferByteSize, int flags, nn::os::NativeHandle* pWaitHandle, uint32_t* pTaskId )
{
    if( EventAvailable() == false )
    {
        EVENT_REPORT( "RecvStart FAILED due to too many system events allocated" );
        return -1;
    }
    HTCSRecvTask* pNewTask = CreateTask<HTCSRecvTask>( descriptor );
    OnSignalEventCreated();
    void* pMem = s_Allocate(sizeof(recv_waiting_task));
    recv_waiting_task* pSave = new (pMem) recv_waiting_task( pNewTask );
    RememberTask( pSave );
    *pTaskId = pNewTask->GetTaskId();
    pNewTask->Start( bufferByteSize, flags, pWaitHandle );
    TASK_LIST_REPORT("AgentHTCSService::RecvStart - Task %d, read %d bytes", *pTaskId, bufferByteSize );
    return 0;
}

//==============================================================================

int AgentHTCSService::RecvResults( uint32_t TaskId, void* pBuffer, size_t bufferByteSize, int64_t* pReceivedSize, int* pLastError )
{
    TASK_LIST_REPORT("AgentHTCSService::RecvResults - Task %d", TaskId );
    // Set these as defaults:
    int Ret = -1;
    *pLastError = nn::htcs::HTCS_EINTR;
    {
        ScopedLock Lock( m_TaskMutex );
        recv_waiting_task* pTask = (recv_waiting_task*)waiting_task::find( TaskId, m_pWaitingTasks );
        if( pTask != NULL )
        {
            Ret = pTask->GetResults( pBuffer, bufferByteSize, pReceivedSize, pLastError );
            waiting_task::remove( pTask, &m_pWaitingTasks );
        }
    }

    TASK_LIST_REPORT("AgentHTCSService::RecvResults returned %d from task %d.", Ret, TaskId );
    return Ret;
}

//==============================================================================

int AgentHTCSService::RecvLargeStart( int descriptor, int32_t unalignedStartSize, int32_t unalignedEndSize, int64_t alignedSendSize, nn::os::NativeHandle alignedMemoryHandle,
    int flags, nn::os::NativeHandle* pWaitHandle, uint32_t* pTaskId )
{
    if( EventAvailable() == false )
    {
        EVENT_REPORT( "RecvLargeStart FAILED due to too many system events allocated" );
        nn::os::CloseNativeHandle( alignedMemoryHandle );
        return -1;
    }

    HTCSRecvLargeTask* pNewTask = CreateTask<HTCSRecvLargeTask>( descriptor );
    OnSignalEventCreated();
    void* pMem = s_Allocate(sizeof(recv_waiting_task));
    recv_waiting_task* pSave = new (pMem) recv_waiting_task( pNewTask );
    RememberTask( pSave );
    *pTaskId = pNewTask->GetTaskId();
    pNewTask->Start( unalignedStartSize, unalignedEndSize, alignedSendSize, alignedMemoryHandle, flags, pWaitHandle );

    TASK_LIST_REPORT("AgentHTCSService::RecvLargeStart - Task %d, recv %d bytes", *pTaskId, unalignedStartSize + unalignedEndSize + alignedSendSize );
    return 0;
}

//==============================================================================

void AgentHTCSService::OnRecvComplete( HTCSRecvTask* pTask )
{
    TASK_LIST_REPORT("AgentHTCSService::OnRecvComplete - Task %d", pTask->GetTaskId() );
    recv_waiting_task* pTaskResults = (recv_waiting_task*)RecallTask( pTask->GetTaskId() );
    if( pTaskResults != NULL )
    {
        pTaskResults->OnComplete( pTask );
        TASK_LIST_REPORT("AgentHTCSService::OnRecvComplete caching results from task %d.", pTask->GetTaskId() );
    }
}

//==============================================================================

void AgentHTCSService::OnRecvComplete( HTCSRecvLargeTask* pTask )
{
    TASK_LIST_REPORT("AgentHTCSService::OnRecvComplete - Task %d", pTask->GetTaskId() );
    recv_waiting_task* pTaskResults = (recv_waiting_task*)RecallTask( pTask->GetTaskId() );
    if( pTaskResults != NULL )
    {
        pTaskResults->OnComplete( pTask );
        TASK_LIST_REPORT("AgentHTCSService::OnRecvComplete caching results from task %d.", pTask->GetTaskId() );
    }
}

//==============================================================================

int AgentHTCSService::SendStart( int descriptor, const void* pBuffer, int32_t bufferByteSize, int flags, nn::os::NativeHandle* pWaitHandle, uint32_t* pTaskId )
{
    if( EventAvailable() == false )
    {
        EVENT_REPORT( "SendStart FAILED due to too many system events allocated" );
        return -1;
    }
    HTCSSendTask* pNewTask = CreateTask<HTCSSendTask>( descriptor );
    OnSignalEventCreated();
    void* pMem = s_Allocate(sizeof(send_waiting_task));
    send_waiting_task* pSave = new (pMem) send_waiting_task( pNewTask );
    RememberTask( pSave );
    *pTaskId = pNewTask->GetTaskId();
    pNewTask->Start( pBuffer, bufferByteSize, flags, pWaitHandle );
    TASK_LIST_REPORT("AgentHTCSService::SendStart - Task %d, send %d bytes", *pTaskId, bufferByteSize );
    return 0;
}

//==============================================================================

int AgentHTCSService::SendLargeStart( int descriptor, const void* pUnalignedStart, int32_t unalignedStartSize, const void* pUnalignedEnd, int32_t unalignedEndSize,
    nn::os::NativeHandle alignedMemoryHandle, int64_t alignedSendSize, int flags, nn::os::NativeHandle* pWaitHandle, uint32_t* pTaskId )
{
    if( EventAvailable() == false )
    {
        EVENT_REPORT( "SendLargeStart FAILED due to too many system events allocated" );
        nn::os::CloseNativeHandle( alignedMemoryHandle );
        return -1;
    }
    HTCSSendLargeTask* pNewTask = CreateTask<HTCSSendLargeTask>( descriptor );
    OnSignalEventCreated();
    void* pMem = s_Allocate(sizeof(send_waiting_task));
    send_waiting_task* pSave = new (pMem) send_waiting_task( pNewTask );
    RememberTask( pSave );
    *pTaskId = pNewTask->GetTaskId();
    pNewTask->Start( pUnalignedStart, unalignedStartSize, pUnalignedEnd, unalignedEndSize, alignedMemoryHandle, alignedSendSize, flags, pWaitHandle );

    TASK_LIST_REPORT("AgentHTCSService::SendLargeStart - Task %d, send %d bytes", *pTaskId, unalignedStartSize + unalignedEndSize + alignedSendSize );
    return 0;
}

//==============================================================================

int AgentHTCSService::SendResults( uint32_t TaskId, int* pLastError )
{
    TASK_LIST_REPORT("AgentHTCSService::SendResults - Task %d", TaskId );
    // Set these as defaults:
    int Ret = -1;
    *pLastError = nn::htcs::HTCS_EINTR;
    {
        ScopedLock Lock( m_TaskMutex );
        send_waiting_task* pTask = (send_waiting_task*)waiting_task::find( TaskId, m_pWaitingTasks );
        if( pTask != NULL )
        {
            Ret = (int)pTask->GetResults( pLastError );
            waiting_task::remove( pTask, &m_pWaitingTasks );
        }
    }

    TASK_LIST_REPORT("AgentHTCSService::SendResults returned %d from task %d.", Ret, TaskId );
    return Ret;
}

//==============================================================================

void AgentHTCSService::OnSendComplete( HTCSSendTask* pTask )
{
    TASK_LIST_REPORT("AgentHTCSService::OnSendComplete - Task %d", pTask->GetTaskId() );
    send_waiting_task* pTaskResults = (send_waiting_task*)RecallTask( pTask->GetTaskId() );
    if( pTaskResults != NULL )
    {
        pTaskResults->OnComplete( pTask );
        TASK_LIST_REPORT("AgentHTCSService::OnSendComplete caching results from task %d.", pTask->GetTaskId() );
    }
}

//==============================================================================

HTCSShutdownTask* AgentHTCSService::Shutdown( int descriptor, int how )
{
    DEJA_TRACE( "AgentHTCSService::Shutdown", "Shutdown socket %d", descriptor );
    HTCSShutdownTask* pRet = CreateTask<HTCSShutdownTask>( descriptor );
    pRet->Shutdown( how );
    return pRet;
}

//==============================================================================

HTCSFcntlTask* AgentHTCSService::Fcntl( int descriptor, int command, int value )
{
    DEJA_TRACE( "AgentHTCSService::Fcntl", "Fcntl socket %d", descriptor );
    HTCSFcntlTask* pRet = CreateTask<HTCSFcntlTask>( descriptor );
    pRet->Fcntl( command, value );
    return pRet;
}

//==============================================================================
}}
//==============================================================================
