﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\tmagent.h"
#include "htcs_service.h"
#include <nn/util/util_FormatString.h>
#include <nn/nn_SdkAssert.h>
#include <nn/htcs/htcs_Types.h>
#include <nn/sf/sf_NativeHandle.h>

//==============================================================================
namespace nn { namespace tma { namespace htcs {
//==============================================================================

void Initialize() NN_NOEXCEPT
{
}

//==============================================================================

void Finalize() NN_NOEXCEPT
{
}

//==============================================================================
// All these tasks return their results in this field, so we can handle these this way every time.
template <class T> s32 CompleteTask( T* pTask, int& ErrorCode )
{
    pTask->WaitComplete( -1 );
    s32 Result = pTask->GetResult();
    ErrorCode = pTask->GetErrorCode();
    pTask->~T();
    ::tma::s_Deallocate( pTask, sizeof( T ) );
    return Result;
}

//==============================================================================

int Socket( int& ErrorCode ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return CompleteTask( HostService.Socket(), ErrorCode );
}

//==============================================================================

int Close( int descriptor, int& ErrorCode ) NN_NOEXCEPT
{
    if( descriptor < 0 )
    {
        ErrorCode = nn::htcs::HTCS_EBADF;
        return -1;
    }
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return CompleteTask( HostService.Close( descriptor ), ErrorCode );
}

//==============================================================================

int Connect( int descriptor, const nn::htcs::SockAddrHtcs* address, int& ErrorCode ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return CompleteTask( HostService.Connect( descriptor, address ), ErrorCode );
}

//==============================================================================

int Bind( int descriptor, const nn::htcs::SockAddrHtcs* address, int& ErrorCode ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return CompleteTask( HostService.Bind( descriptor, address ), ErrorCode );
}

//==============================================================================

int Listen( int descriptor, int backlogCount, int& ErrorCode ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return CompleteTask( HostService.Listen( descriptor, backlogCount ), ErrorCode );
}

//==============================================================================

int Accept( int descriptor, nn::htcs::SockAddrHtcs* address, int& ErrorCode ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return CompleteTask( HostService.Accept( descriptor, address ), ErrorCode );
}

//==============================================================================

nn::htcs::ssize_t Recv( int descriptor, void* buffer, size_t bufferByteSize, int flags, int& ErrorCode ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return CompleteTask( HostService.Recv( descriptor, buffer, bufferByteSize, flags ), ErrorCode );
}

//==============================================================================

nn::htcs::ssize_t Send( int descriptor, const void* buffer, size_t bufferByteSize, int flags, int& ErrorCode ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return CompleteTask( HostService.Send( descriptor, buffer, bufferByteSize, flags ), ErrorCode );
}
//==============================================================================

int AcceptStart( int descriptor, nn::os::NativeHandle* pWaitHandle, uint32_t* pTaskId ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return HostService.AcceptStart( descriptor, pWaitHandle, pTaskId );
}

//==============================================================================

int AcceptResults( uint32_t TaskId, nn::htcs::SockAddrHtcs* address, int* pLastError ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return HostService.AcceptResults( TaskId, address, pLastError );
}

//==============================================================================

int RecvStart( int descriptor, int64_t bufferByteSize, int flags, nn::os::NativeHandle* pWaitHandle, uint32_t* pTaskId ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return HostService.RecvStart( descriptor, bufferByteSize, flags, pWaitHandle, pTaskId );
}

//==============================================================================

int RecvLargeStart( int descriptor, int32_t unalignedStartSize, int32_t unalignedEndSize, int64_t alignedSize,
    nn::os::NativeHandle alignedMemoryHandle, int flags, nn::os::NativeHandle* pWaitHandle, uint32_t* pTaskId ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    int Ret = HostService.RecvLargeStart( descriptor, unalignedStartSize, unalignedEndSize, alignedSize,
        alignedMemoryHandle, flags, pWaitHandle, pTaskId );
    return Ret;
}

//==============================================================================

nn::htcs::ssize_t RecvResults( uint32_t TaskId, void* pBuffer, size_t bufferByteSize, int64_t* pReceivedSize, int* pLastError ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return HostService.RecvResults( TaskId, pBuffer, bufferByteSize, pReceivedSize, pLastError );
}

//==============================================================================

int SendLargeStart( int descriptor, const void* unalignedStart, int32_t unalignedStartSize, const void* unalignedEnd, int32_t unalignedEndSize,
        nn::os::NativeHandle alignedMemoryHandle, int64_t alignedSize, int flags, nn::os::NativeHandle* pWaitHandle, uint32_t* pTaskId ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    int Ret = HostService.SendLargeStart( descriptor, unalignedStart, unalignedStartSize, unalignedEnd, unalignedEndSize,
        alignedMemoryHandle, alignedSize, flags, pWaitHandle, pTaskId  );

    return Ret;
}

//==============================================================================

int SendStart( int descriptor, const void* buffer, size_t bufferByteSize, int flags, nn::os::NativeHandle* pWaitHandle, uint32_t* pTaskId ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return HostService.SendStart( descriptor, buffer, (int32_t)bufferByteSize, flags, pWaitHandle, pTaskId );
}

//==============================================================================

nn::htcs::ssize_t SendResults( uint32_t TaskId, int* pLastError ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return HostService.SendResults( TaskId, pLastError );
}

//==============================================================================

int Shutdown( int descriptor, int how, int& ErrorCode ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return CompleteTask( HostService.Shutdown( descriptor, how ), ErrorCode );
}

//==============================================================================

int Fcntl( int descriptor, int command, int value, int& ErrorCode ) NN_NOEXCEPT
{
    ::tma::htcs::AgentHTCSService&  HostService = ::tma::htcs::GetHTCSService();
    return CompleteTask( HostService.Fcntl( descriptor, command, value ), ErrorCode );
}

//==============================================================================
// Peer name methods
//==============================================================================
namespace
{
    const nn::htcs::HtcsPeerName PeerNameAny = { "" };
    const nn::htcs::HtcsPeerName DefaultHostName = { "" };
}

//==============================================================================

const nn::htcs::HtcsPeerName GetPeerNameAny() NN_NOEXCEPT
{
    return PeerNameAny;
}

//==============================================================================

const nn::htcs::HtcsPeerName GetDefaultHostName() NN_NOEXCEPT
{
    return DefaultHostName;
}

//==============================================================================
}}} // namespace
//==============================================================================
