﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// このファイルはユーザ定義

#include "..\tmagent.h"
#include "htcs_api.h"
#include "htcs_HtcsManagerImpl.h"
#include "htcs_HtcsSocketImpl.h"
#include <nn/tma/tma_Result.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/os/os_Mutex.h>
#include <mutex>
#include <nn/util/util_Exchange.h>

namespace nn { namespace tma {

namespace {

struct SocketImplAllocatorTag;
typedef nn::sf::ExpHeapStaticAllocator<4096 * 8, SocketImplAllocatorTag> SocketImplAllocator;

// SocketImplAllocator を静的コンストラクタで初期化するためのヘルパー
class SocketImplAllocatorInitializer
{
public:
    SocketImplAllocatorInitializer() NN_NOEXCEPT
    {
        SocketImplAllocator::Initialize(nn::lmem::CreationOption_ThreadSafe);
    }
} g_SocketImplAllocatorInitializer;

struct MapEntry
{
    Bit64 processId;
    HtcsManagerImpl* p;
};

struct StaticMutex
{
    os::MutexType _mutex;

    void lock() NN_NOEXCEPT
    {
        os::LockMutex(&_mutex);
    }

    void unlock() NN_NOEXCEPT
    {
        os::UnlockMutex(&_mutex);
    }
};

StaticMutex g_MapMutex = { NN_OS_MUTEX_INITIALIZER(false) };
MapEntry g_MapEntries[30] = {};

void AddToMap(Bit64 processId, HtcsManagerImpl* p) NN_NOEXCEPT
{
    std::lock_guard<decltype(g_MapMutex)> lk(g_MapMutex);
    for (auto&& e : g_MapEntries)
    {
        if (!e.p)
        {
            e.processId = processId;
            e.p = p;
            return;
        }
    }
}

HtcsManagerImpl* GetFromMap(Bit64 processId) NN_NOEXCEPT
{
    std::lock_guard<decltype(g_MapMutex)> lk(g_MapMutex);
    for (auto&& e : g_MapEntries)
    {
        if (e.processId == processId && e.p)
        {
            return util::Exchange(&e.p, nullptr);
        }
    }
    return nullptr;
}

}

nn::Result HtcsManagerImpl::Socket(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutSocket) NN_NOEXCEPT
{
#if 0
    int ErrorCode = 0;
    int32_t ret = htcs::Socket( ErrorCode );
    pOutSocket.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        //TODO:  SAVE SOCKET HERE
        m_SocketPrimitiveList.Save( ret );
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutErrorCode);
    NN_UNUSED(pOutSocket);
    return ResultUnknown();
#endif
}

nn::Result HtcsManagerImpl::Close(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t descriptor) NN_NOEXCEPT
{
#if 0
    int ErrorCode = 0;
    int32_t ret = htcs::Close( descriptor, ErrorCode );
    pOutIntResult.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        m_SocketPrimitiveList.Remove( descriptor );
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutErrorCode);
    NN_UNUSED(pOutIntResult);
    NN_UNUSED(descriptor);
    return ResultUnknown();
#endif
}

nn::Result HtcsManagerImpl::Connect(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t descriptor, const nn::htcs::SockAddrHtcs& address) NN_NOEXCEPT
{
#if 0
    int ErrorCode = 0;
    int32_t ret = htcs::Connect( descriptor, &address, ErrorCode );
    pOutIntResult.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutErrorCode);
    NN_UNUSED(pOutIntResult);
    NN_UNUSED(descriptor);
    NN_UNUSED(address);
    return ResultUnknown();
#endif
}

nn::Result HtcsManagerImpl::Bind(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t descriptor, const nn::htcs::SockAddrHtcs& address) NN_NOEXCEPT
{
#if 0
    int ErrorCode = 0;
    int32_t ret = htcs::Bind( descriptor, &address, ErrorCode );
    pOutIntResult.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutErrorCode);
    NN_UNUSED(pOutIntResult);
    NN_UNUSED(descriptor);
    NN_UNUSED(address);
    return ResultUnknown();
#endif
}

nn::Result HtcsManagerImpl::Listen(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t descriptor, std::int32_t backlogCount) NN_NOEXCEPT
{
#if 0
    int ErrorCode = 0;
    int32_t ret = htcs::Listen( descriptor, backlogCount, ErrorCode );
    pOutIntResult.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutErrorCode);
    NN_UNUSED(pOutIntResult);
    NN_UNUSED(descriptor);
    NN_UNUSED(backlogCount);
    return ResultUnknown();
#endif
}

nn::Result HtcsManagerImpl::Accept(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, nn::sf::Out<nn::htcs::SockAddrHtcs> pOutAddress, std::int32_t descriptor) NN_NOEXCEPT
{
#if 0
    int ErrorCode = 0;
    int32_t ret = htcs::Accept(descriptor, pOutAddress.GetPointer(), ErrorCode);
    pOutIntResult.Set(ret);
    if (ret < 0)
    {
        pOutErrorCode.Set(ErrorCode);
    }
    else
    {
        m_SocketPrimitiveList.Save(ret);
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutErrorCode);
    NN_UNUSED(pOutIntResult);
    NN_UNUSED(pOutAddress);
    NN_UNUSED(descriptor);
    return ResultUnknown();
#endif
}

nn::Result HtcsManagerImpl::Recv(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int64_t> pOutReceivedSize, const nn::sf::OutBuffer& buffer, std::int32_t descriptor, std::int32_t flags) NN_NOEXCEPT
{
#if 0
    int ErrorCode = 0;
    int64_t ret = htcs::Recv( descriptor, buffer.GetPointerUnsafe(), buffer.GetSize(), flags, ErrorCode );
    pOutReceivedSize.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutErrorCode);
    NN_UNUSED(pOutReceivedSize);
    NN_UNUSED(buffer);
    NN_UNUSED(descriptor);
    NN_UNUSED(flags);
    return ResultUnknown();
#endif
}

nn::Result HtcsManagerImpl::Send(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int64_t> pOutSentSize, std::int32_t descriptor, const nn::sf::InBuffer& buffer, std::int32_t flags) NN_NOEXCEPT
{
#if 0
    int ErrorCode = 0;
    int64_t ret = htcs::Send( descriptor, buffer.GetPointerUnsafe(), buffer.GetSize(), flags, ErrorCode );
    pOutSentSize.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutErrorCode);
    NN_UNUSED(pOutSentSize);
    NN_UNUSED(descriptor);
    NN_UNUSED(buffer);
    NN_UNUSED(flags);
    return ResultUnknown();
#endif
}

nn::Result HtcsManagerImpl::Shutdown(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t descriptor, std::int32_t how) NN_NOEXCEPT
{
#if 0
    int ErrorCode = 0;
    int32_t ret = htcs::Shutdown( descriptor, how, ErrorCode );
    pOutIntResult.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutErrorCode);
    NN_UNUSED(pOutIntResult);
    NN_UNUSED(descriptor);
    NN_UNUSED(how);
    return ResultUnknown();
#endif
}

nn::Result HtcsManagerImpl::Fcntl(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t descriptor, std::int32_t command, std::int32_t value) NN_NOEXCEPT
{
#if 0
    int ErrorCode = 0;
    int32_t ret = htcs::Fcntl( descriptor, command, value, ErrorCode );
    pOutIntResult.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutErrorCode);
    NN_UNUSED(pOutIntResult);
    NN_UNUSED(descriptor);
    NN_UNUSED(command);
    NN_UNUSED(value);
    return ResultUnknown();
#endif
}

nn::Result HtcsManagerImpl::GetPeerNameAny(nn::sf::Out<nn::htcs::HtcsPeerName> pOutValue) NN_NOEXCEPT
{
    pOutValue.Set(htcs::GetPeerNameAny());
    NN_RESULT_SUCCESS;
}

nn::Result HtcsManagerImpl::GetDefaultHostName(nn::sf::Out<nn::htcs::HtcsPeerName> pOutValue) NN_NOEXCEPT
{
    pOutValue.Set(htcs::GetDefaultHostName());
    NN_RESULT_SUCCESS;
}

//===============================================================================================================

nn::Result HtcsManagerImpl::CreateSocketOld(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<nn::sf::SharedPointer<nn::tma::ISocket>> pOutSocket) NN_NOEXCEPT
{
    return CreateSocket(pOutErrorCode, pOutSocket, false);
}

nn::Result HtcsManagerImpl::CreateSocket(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<nn::sf::SharedPointer<nn::tma::ISocket>> pOutSocket, bool enableDisconnectionEmulation) NN_NOEXCEPT
{
    if (enableDisconnectionEmulation && ::tma::GetEnableDisconnectionEmulationFlag())
    {
        // TODO: It is matched with the behavior of disconnection state
        pOutErrorCode.Set(nn::htcs::HTCS_ENETDOWN);
        NN_RESULT_SUCCESS;
    }

    int ErrorCode = 0;
    int32_t socket = htcs::Socket(ErrorCode);
    if( socket < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
        NN_RESULT_SUCCESS;
    }

    auto p = nn::sf::ObjectFactory<SocketImplAllocator::Policy>::CreateSharedEmplaced<
        ISocket,
        HtcsSocketImpl
    >(this, socket);

    pOutSocket.Set(std::move(p));
    pOutErrorCode.Set(0);

    NN_RESULT_SUCCESS;
}
//===============================================================================================================


nn::Result HtcsManagerImpl::RegisterProcessId(Bit64 processId) NN_NOEXCEPT
{
    this->AddReference();
    AddToMap(processId, this);
    NN_RESULT_SUCCESS;
}

nn::Result HtcsManagerImpl::MonitorManager(Bit64 processId) NN_NOEXCEPT
{
    auto pTarget = GetFromMap(processId);
    if (!pTarget)
    {
        NN_RESULT_SUCCESS;
    }
    this->m_MonitorTarget = decltype(m_MonitorTarget)(pTarget, false);
    NN_RESULT_SUCCESS;
}

HtcsManagerImpl::~HtcsManagerImpl() NN_NOEXCEPT
{
    if (m_MonitorTarget)
    {
        m_MonitorTarget->m_SocketList.CancelWait();
        m_MonitorTarget->m_SocketPrimitiveList.Clear();
    }
}

//========================================================================================================

HtcsManagerImpl::SocketList::SocketList() NN_NOEXCEPT
    : m_Mutex(false)
{
}

void HtcsManagerImpl::SocketList::Register(HtcsSocketImpl* p) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_Sockets.push_back(*p);
}

void HtcsManagerImpl::SocketList::Unregister(HtcsSocketImpl* p) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_Sockets.erase(m_Sockets.iterator_to(*p));
}

void HtcsManagerImpl::SocketList::CancelWait() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    for (auto&& e : m_Sockets)
    {
        e.CancelWait();
    }
}

//===============================================================================================================

HtcsManagerImpl::SocketPrimitiveList::SocketPrimitiveList() NN_NOEXCEPT
    : m_Mutex(false)
{
}

void HtcsManagerImpl::SocketPrimitiveList::Save(std::int32_t p) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_Sockets.push_back(p);
}

void HtcsManagerImpl::SocketPrimitiveList::Remove(std::int32_t p) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    for( std::vector<std::int32_t>::iterator it = m_Sockets.begin(); it != m_Sockets.end(); it++ )
    {
        if(*it == p )
        {
            m_Sockets.erase( it );
            break;
        }
    }
}

void HtcsManagerImpl::SocketPrimitiveList::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    for( std::vector<std::int32_t>::iterator it = m_Sockets.begin(); it != m_Sockets.end(); it++ )
    {
        //e.CancelWait();
        int32_t errorCode;
        htcs::Close( *it, errorCode );
    }
    m_Sockets.clear();
}

//================================================================================================

nn::sf::SharedPointer<IHtcsManager> CreateHtcsManagerImpl() NN_NOEXCEPT
{
    return nn::sf::ObjectFactory<SocketImplAllocator::Policy>::CreateSharedEmplaced<
        IHtcsManager,
        HtcsManagerImpl
    >();
}


}}
