﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "..\tmipc\tmipc_packet.h"
#include "..\tmipc\tmipc_task.h"

//==============================================================================
namespace tma { namespace hio {
//==============================================================================

typedef uint64_t TMA_HOSTIO_FILE_HANDLE;
#define INVALID_HOSTIO_FILE_HANDLE (uint64_t)-1

// Our IO types as returned from target manager (services\hio\target_io_service.h)
enum IOType
{
    IOType_Directory,
    IOType_File,
};

//==============================================================================

class HostIOTask : public tmipc::Task
{
public:
    explicit        HostIOTask          ( TMA_HOSTIO_FILE_HANDLE    Handle );
                   ~HostIOTask          ();
virtual void        OnInitiate          ( tmipc::Packet*    pPacket ) = 0;
virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket ) = 0;
virtual void        OnSendPacket        ( tmipc::Packet*    pPacket ) = 0;

        s32         GetResult               ();

protected:
    TMA_HOSTIO_FILE_HANDLE  m_Handle;
    s32                     m_Result;
    int64_t                 m_OperationValue;
};


//==============================================================================

class HostIOReadTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioFileRead };

    explicit        HostIOReadTask          ( TMA_HOSTIO_FILE_HANDLE    Handle );
                   ~HostIOReadTask          ();
virtual void        OnInitiate              ( tmipc::Packet*    pPacket );
virtual void        OnRecvPacket            ( tmipc::Packet*    pPacket );
virtual void        OnSendPacket            ( tmipc::Packet*    pPacket );

        void        Read                    ( int32_t*          pBytesRead,
                                              int64_t           Offset,
                                              void*             pDataBuffer,
                                              int64_t           AmountToRead );

private:
    int32_t         m_NumberOfPacketsExpected;
    int32_t*        m_pBytesRead;
    void*           m_WriteAddr;
    void*           m_pDataBuffer;
};

//==============================================================================

class HostIOWriteTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioFileWrite };

    explicit        HostIOWriteTask         ( TMA_HOSTIO_FILE_HANDLE   Handle );
                   ~HostIOWriteTask         ();
virtual void        OnInitiate              ( tmipc::Packet*    pPacket );
virtual void        OnRecvPacket            ( tmipc::Packet*    pPacket );
virtual void        OnSendPacket            ( tmipc::Packet*    pPacket );

        void        Write                   ( int32_t*          pBytesWritten,
                                              int64_t           Offset,
                                              void*             pDataBuffer,
                                              int64_t           AmountToWrite );

private:
    void            PopulatePacket          ( tmipc::Packet*    pPacket );
    int32_t*        m_pBytesWritten;

    //Internal-use variables.
    int64_t         m_Offset;
    char*           m_pReadFromBuffer;
    int64_t         m_AmountLeftToWrite;
};

//=============================================================================

class HostIOExistsTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioFileExists };

    explicit        HostIOExistsTask      ( TMA_HOSTIO_FILE_HANDLE    Handle );

    virtual void    OnInitiate          ( tmipc::Packet*    pPacket );
    virtual void    OnRecvPacket        ( tmipc::Packet*    pPacket );
    virtual void    OnSendPacket        ( tmipc::Packet*    pPacket );

    void            Exists              ( const char*       pFullPath );
    bool            GetExistsResult     ();
};

//=============================================================================

class HostIOOpenTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioFileOpen };

    explicit        HostIOOpenTask      ( TMA_HOSTIO_FILE_HANDLE    Handle );

    virtual void    OnInitiate          ( tmipc::Packet*    pPacket );
    virtual void    OnRecvPacket        ( tmipc::Packet*    pPacket );
    virtual void    OnSendPacket        ( tmipc::Packet*    pPacket );

    void            Open                ( uint32_t          OpenFlags,
                                          const char*       pFullPath );
    TMA_HOSTIO_FILE_HANDLE GetHandle    ();
};

//=============================================================================

class HostIOCloseTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioFileClose };

    explicit        HostIOCloseTask     ( TMA_HOSTIO_FILE_HANDLE    Handle );

    virtual void    OnInitiate          ( tmipc::Packet*    pPacket );
    virtual void    OnRecvPacket        ( tmipc::Packet*    pPacket );
    virtual void    OnSendPacket        ( tmipc::Packet*    pPacket );

    void            Close               ();
    TMA_HOSTIO_FILE_HANDLE GetHandle    ();
};

//=============================================================================

class HostIOGetSizeTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioFileGetSize };

    explicit        HostIOGetSizeTask   ( TMA_HOSTIO_FILE_HANDLE    Handle );

    virtual void    OnInitiate          ( tmipc::Packet*    pPacket );
    virtual void    OnRecvPacket        ( tmipc::Packet*    pPacket );
    virtual void    OnSendPacket        ( tmipc::Packet*    pPacket );

    void            GetSize             ();

    TMA_HOSTIO_FILE_HANDLE GetHandle    ();
    int64_t         GetSizeResult       ();
};

//=============================================================================

class HostIOCreateTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioFileCreate };

    explicit        HostIOCreateTask    ( TMA_HOSTIO_FILE_HANDLE    Handle );

    virtual void    OnInitiate          ( tmipc::Packet*    pPacket );
    virtual void    OnRecvPacket        ( tmipc::Packet*    pPacket );
    virtual void    OnSendPacket        ( tmipc::Packet*    pPacket );

    void            Create              ( const char*       pFullPath,
                                          s64               Size );
};

//=============================================================================

class HostIODeleteTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioFileDelete };

    explicit        HostIODeleteTask    ( TMA_HOSTIO_FILE_HANDLE    Handle );

    virtual void    OnInitiate          ( tmipc::Packet*    pPacket );
    virtual void    OnRecvPacket        ( tmipc::Packet*    pPacket );
    virtual void    OnSendPacket        ( tmipc::Packet*    pPacket );

    void            Delete              ( const char*       pFullPath );
};

//=============================================================================

class HostIORenameTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioFileRename };

    explicit        HostIORenameTask    ( TMA_HOSTIO_FILE_HANDLE    Handle );

    virtual void    OnInitiate          ( tmipc::Packet*    pPacket );
    virtual void    OnRecvPacket        ( tmipc::Packet*    pPacket );
    virtual void    OnSendPacket        ( tmipc::Packet*    pPacket );

    void            Rename              ( const char*       pFrom,
                                          const char*       pTo );
};

//=============================================================================

class HostIOGetIOTypeTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioGetIoType };

    explicit        HostIOGetIOTypeTask ( TMA_HOSTIO_FILE_HANDLE    Handle );

    virtual void    OnInitiate          ( tmipc::Packet*    pPacket );
    virtual void    OnRecvPacket        ( tmipc::Packet*    pPacket );
    virtual void    OnSendPacket        ( tmipc::Packet*    pPacket );

    void            GetIOType           ( const char* pFullPath );
    s32             GetIOTypeResult     ();
};

//=============================================================================

class HostIOFlushTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioFileFlush };

    explicit        HostIOFlushTask     ( TMA_HOSTIO_FILE_HANDLE    Handle );

    virtual void    OnInitiate          ( tmipc::Packet*    pPacket );
    virtual void    OnRecvPacket        ( tmipc::Packet*    pPacket );
    virtual void    OnSendPacket        ( tmipc::Packet*    pPacket );

    void            Flush               ();
};

//=============================================================================

class HostIOSetSizeTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioFileSetSize };

    explicit        HostIOSetSizeTask   ( TMA_HOSTIO_FILE_HANDLE    Handle );

    virtual void    OnInitiate          ( tmipc::Packet*    pPacket );
    virtual void    OnRecvPacket        ( tmipc::Packet*    pPacket );
    virtual void    OnSendPacket        ( tmipc::Packet*    pPacket );

    void            SetSize             ( s64 Size );
};

//=============================================================================

class HostIOGetSizeFromPathTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioFileGetSizeFromPath };

    explicit        HostIOGetSizeFromPathTask( TMA_HOSTIO_FILE_HANDLE Handle );

    virtual void    OnInitiate          ( tmipc::Packet*    pPacket );
    virtual void    OnRecvPacket        ( tmipc::Packet*    pPacket );
    virtual void    OnSendPacket        ( tmipc::Packet*    pPacket );

    void            GetSize             ( const char*       pFullPath );
    int64_t         GetSizeResult       ();
};

//=============================================================================

class HostIOSetPriorityTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioFileSetPriority };

    explicit        HostIOSetPriorityTask ( TMA_HOSTIO_FILE_HANDLE    Handle );

    virtual void    OnInitiate          ( tmipc::Packet*    pPacket );
    virtual void    OnRecvPacket        ( tmipc::Packet*    pPacket );
    virtual void    OnSendPacket        ( tmipc::Packet*    pPacket );

    void            SetPriority         ( s32 Priority );
};

//=============================================================================

class HostIOGetPriorityTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioFileGetPriority };

    explicit        HostIOGetPriorityTask ( TMA_HOSTIO_FILE_HANDLE    Handle );

    virtual void    OnInitiate          ( tmipc::Packet*    pPacket );
    virtual void    OnRecvPacket        ( tmipc::Packet*    pPacket );
    virtual void    OnSendPacket        ( tmipc::Packet*    pPacket );

    void            GetPriority         ();
    s32             GetPriorityResult   ();
};

//=============================================================================

class HostIOGetFileTimeStampTask : public HostIOTask
{
public:
    enum { TaskType = tmipc::TaskType_HioGetFileTimeStamp };

    explicit        HostIOGetFileTimeStampTask ( TMA_HOSTIO_FILE_HANDLE     Handle );

    virtual void    OnInitiate          ( tmipc::Packet*    pPacket );
    virtual void    OnRecvPacket        ( tmipc::Packet*    pPacket );
    virtual void    OnSendPacket        ( tmipc::Packet*    pPacket );

    void            GetFileTimeStamp    ( const char*       pFullPath );
    void            GetTimeStampResults ( u64*              pCreateTile,
                                          u64*              pAccessTime,
                                          u64*              pModifyTime );

    u64                     m_CreateTime;
    u64                     m_AccessTime;
    u64                     m_ModifyTime;
};

//==============================================================================
}}
//==============================================================================
