﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\tmagent.h"
#include "hio_task.h"
#include "hio_service.h"
#include "hio_opcodes.h"

//==============================================================================
namespace tma { namespace hio {
//==============================================================================

//==============================================================================

AgentHostIOService::AgentHostIOService()
{
    DEJA_TRACE( "AgentHostIOService::AgentHostIOService", "AgentHostIOService" );
    m_ServiceId = HashString( "HostIOService" );
}

//==============================================================================

AgentHostIOService::~AgentHostIOService()
{
    DEJA_TRACE( "AgentHostIOService::~AgentHostIOService", "~AgentHostIOService" );
}

//==============================================================================

void AgentHostIOService::Init()
{
    DEJA_TRACE( "AgentHostIOService::Init", "Init" );
    Create();
}

//==============================================================================

void AgentHostIOService::Kill()
{
    DEJA_TRACE( "AgentHostIOService::Kill", "Kill" );
    Destroy();
}

//==============================================================================

tmipc::Task* AgentHostIOService::OnNewTask( tmipc::Packet* pPacket )
{
    (void)pPacket;

    DEJA_TRACE( "AgentHostIOService::OnInitiate", "OnInitiate" );
    return( NULL );
}

//==============================================================================

HostIOExistsTask* AgentHostIOService::FileExists( const char* pFullPath )
{
    DEJA_TRACE( "AgentHostIOService::FileExists", "FileExists for path %s", pFullPath );
    void* pMem = s_Allocate( sizeof( HostIOExistsTask ) );
    HostIOExistsTask* pTask = new (pMem) HostIOExistsTask( INVALID_HOSTIO_FILE_HANDLE );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->Exists( pFullPath );
    return pTask;
}

//==============================================================================

HostIOCreateTask* AgentHostIOService::CreateFile( const char* pFullPath, int64_t Size )
{
    DEJA_TRACE( "AgentHostIOService::CreateFile", "CreateFile %s with size %lld", pFullPath, Size );
    void* pMem = s_Allocate( sizeof( HostIOCreateTask ) );
    HostIOCreateTask* pTask = new (pMem) HostIOCreateTask( INVALID_HOSTIO_FILE_HANDLE );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->Create( pFullPath, Size );
    return pTask;
}

//==============================================================================

HostIODeleteTask* AgentHostIOService::DeleteFile( const char* pFullPath )
{
    DEJA_TRACE( "AgentHostIOService::DeleteFile", "DeleteFile %s", pFullPath );
    void* pMem = s_Allocate( sizeof( HostIODeleteTask ) );
    HostIODeleteTask* pTask = new (pMem) HostIODeleteTask( INVALID_HOSTIO_FILE_HANDLE );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->Delete( pFullPath );
    return pTask;
}

//==============================================================================

HostIORenameTask* AgentHostIOService::RenameFile( const char* pFromPath, const char* pToPath )
{
    DEJA_TRACE( "AgentHostIOService::RenameFile", "RenameFile from %s to %s", pFromPath, pToPath );
    void* pMem = s_Allocate( sizeof( HostIORenameTask ) );
    HostIORenameTask* pTask = new (pMem) HostIORenameTask( INVALID_HOSTIO_FILE_HANDLE );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->Rename( pFromPath, pToPath );
    return pTask;
}

//==============================================================================

HostIOOpenTask* AgentHostIOService::OpenFile( uint32_t OpenFlags, const char* pFullPath )
{
    TMA_TRACE( "AgentHostIOService", "OpenFile %s", pFullPath );
    DEJA_TRACE( "AgentHostIOService::OpenFile", "OpenFile %s", pFullPath );
    void* pMem = s_Allocate( sizeof( HostIOOpenTask ) );
    HostIOOpenTask* pTask = new (pMem) HostIOOpenTask( INVALID_HOSTIO_FILE_HANDLE );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->Open( OpenFlags, pFullPath );
    return pTask;
}

//==============================================================================

HostIOGetSizeTask* AgentHostIOService::GetFileSize( TMA_HOSTIO_FILE_HANDLE FileHandle )
{
    DEJA_TRACE( "AgentHostIOService::GetFileSize", "GetFileSize for handle %lld", FileHandle );
    void* pMem = s_Allocate( sizeof( HostIOGetSizeTask ) );
    HostIOGetSizeTask* pTask = new (pMem) HostIOGetSizeTask( FileHandle );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->GetSize();
    return pTask;
}

//==============================================================================

HostIOGetSizeFromPathTask* AgentHostIOService::GetFileSize( const char* pFullPath )
{
    DEJA_TRACE( "AgentHostIOService::GetFileSize", "GetFileSize for path %s", pFullPath );
    void* pMem = s_Allocate( sizeof( HostIOGetSizeFromPathTask ) );
    HostIOGetSizeFromPathTask* pTask = new (pMem) HostIOGetSizeFromPathTask( INVALID_HOSTIO_FILE_HANDLE );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->GetSize( pFullPath );
    return pTask;
}

//==============================================================================

HostIOGetFileTimeStampTask* AgentHostIOService::GetFileTimeStamp( const char* pFullPath )
{
    DEJA_TRACE( "AgentHostIOService::GetFileTimeStamp", "GetFileTimeStamp for path %s", pFullPath );
    void* pMem = s_Allocate( sizeof( HostIOGetFileTimeStampTask ) );
    HostIOGetFileTimeStampTask* pTask = new (pMem) HostIOGetFileTimeStampTask( INVALID_HOSTIO_FILE_HANDLE );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->GetFileTimeStamp( pFullPath );
    return pTask;
}

//==============================================================================

HostIOSetSizeTask* AgentHostIOService::SetFileSize( TMA_HOSTIO_FILE_HANDLE FileHandle,  s64 Size )
{
    DEJA_TRACE( "AgentHostIOService::SetFileSize", "SetFileSize for handle %lld", FileHandle );
    void* pMem = s_Allocate( sizeof( HostIOSetSizeTask ) );
    HostIOSetSizeTask* pTask = new (pMem) HostIOSetSizeTask( FileHandle );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->SetSize( Size );
    return pTask;
}

//==============================================================================

HostIOReadTask* AgentHostIOService::ReadFile( int32_t* pBytesRead, TMA_HOSTIO_FILE_HANDLE FileHandle, int64_t Offset, void* pDataBuffer, int64_t AmountToRead )
{
    DEJA_TRACE( "AgentHostIOService::ReadFile", "ReadFile: Offset = %lld, ReadToBuffer = %p, AmountToRead = %lld", Offset, pDataBuffer, AmountToRead );
    void* pMem = s_Allocate( sizeof( HostIOReadTask ) );
    HostIOReadTask* pTask = new (pMem) HostIOReadTask( FileHandle );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->Read( pBytesRead, Offset, pDataBuffer, AmountToRead );
    return pTask;
}

//==============================================================================

HostIOWriteTask* AgentHostIOService::WriteFile( int32_t* pBytesWritten, TMA_HOSTIO_FILE_HANDLE FileHandle, int64_t Offset, void* pDataBuffer, int64_t AmountToWrite )
{
    DEJA_TRACE( "AgentHostIOService::WriteFile", "WriteFile: Offset = %lld, ReadFromBuffer = %p, AmountToWrite = %lld", Offset, pDataBuffer, AmountToWrite );
    void* pMem = s_Allocate( sizeof( HostIOWriteTask ) );
    HostIOWriteTask* pTask = new (pMem) HostIOWriteTask( FileHandle );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->Write( pBytesWritten, Offset, pDataBuffer, AmountToWrite );
    return pTask;
}

//==============================================================================

HostIOSetPriorityTask* AgentHostIOService::SetFilePriority( TMA_HOSTIO_FILE_HANDLE FileHandle, s32 Priority )
{
    DEJA_TRACE( "AgentHostIOService::SetFilePriority", "SetFilePriority for handle %lld", FileHandle );
    void* pMem = s_Allocate( sizeof( HostIOSetPriorityTask ) );
    HostIOSetPriorityTask* pTask = new (pMem) HostIOSetPriorityTask( FileHandle );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->SetPriority( Priority );
    return pTask;

}
//==============================================================================

HostIOGetPriorityTask* AgentHostIOService::GetFilePriority( TMA_HOSTIO_FILE_HANDLE FileHandle )
{
    DEJA_TRACE( "AgentHostIOService::GetFilePriority", "GetFilePriority for handle %lld", FileHandle );
    void* pMem = s_Allocate( sizeof( HostIOGetPriorityTask ) );
    HostIOGetPriorityTask* pTask = new (pMem) HostIOGetPriorityTask( FileHandle );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->GetPriority();
    return pTask;
}

//==============================================================================

HostIOFlushTask* AgentHostIOService::FlushFile( TMA_HOSTIO_FILE_HANDLE FileHandle )
{
    DEJA_TRACE( "AgentHostIOService::FlushFile", "FlushFile for handle %lld", FileHandle );
    void* pMem = s_Allocate( sizeof( HostIOFlushTask ) );
    HostIOFlushTask* pTask = new (pMem) HostIOFlushTask( FileHandle );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->Flush();
    return pTask;
}

//==============================================================================

HostIOCloseTask* AgentHostIOService::CloseFile( TMA_HOSTIO_FILE_HANDLE FileHandle )
{
    DEJA_TRACE( "AgentHostIOService::CloseFile", "CloseFile for handle %lld", FileHandle );
    void* pMem = s_Allocate( sizeof( HostIOCloseTask ) );
    HostIOCloseTask* pTask = new (pMem) HostIOCloseTask( FileHandle );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->Close();
    return pTask;
}

//==============================================================================

HostIOGetIOTypeTask* AgentHostIOService::GetIOType( const char* pFullPath )
{
    DEJA_TRACE( "AgentHostIOService::GetIOType", "GetIOType for path %s", pFullPath );
    void* pMem = s_Allocate( sizeof( HostIOGetIOTypeTask ) );
    HostIOGetIOTypeTask* pTask = new (pMem) HostIOGetIOTypeTask( INVALID_HOSTIO_FILE_HANDLE );
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->GetIOType( pFullPath );
    return pTask;
}

//==============================================================================
}}
//==============================================================================
