﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\tmagent.h"
#include "dbg_Threads.h"
#include <nn/osdbg.h>
#include <nn/dmnt/dmnt_Api.h>
#include "..\dbghlp\dbghlp.h"
#include "..\threadfrozen\thread_frozen_api.h"

//==============================================================================
namespace tma { namespace dbg {
//==============================================================================

extern tmapi::result CheckProcessHandleForError( nn::svc::Handle ProcessHandle );

enum
{
    HIGHEST_APP_PRIORITY    = 28,
    APP_PRIORITY_RANGE      = nn::os::ThreadPriorityRangeSize,
    DEFAULT_THREAD_PRIORITY = nn::os::DefaultThreadPriority,
};

std::map<nn::Bit64, nn::svc::ThreadContext> ThreadDefinition::s_ThreadContextCache;

//==============================================================================

void ThreadDefinition::ClearThreadCache()
{
    s_ThreadContextCache.clear();
}

//==============================================================================

nn::Result ThreadDefinition::GetThreadContext( tma::dbg::thread_context* pThreadContext, nn::svc::Handle ProcessHandle, nn::Bit64 ThreadId, nn::Bit32 ControlFlags)
{
    TMA_PRINTF( "ThreadDefinition::GetThreadContext - Thread: %lld, Flags: %x\n", ThreadId, ControlFlags );

    nn::Result result;
    nn::svc::ThreadContext nnThreadContext;
    auto Context = s_ThreadContextCache.find( ThreadId );
    if( Context != s_ThreadContextCache.end() )
    {
        nnThreadContext = Context->second;
        result = nn::ResultSuccess();
    }
    else
    {
        result = dbghlp::GetDebugThreadContext( &nnThreadContext, ProcessHandle, ThreadId, ControlFlags );
        s_ThreadContextCache.insert( std::pair<nn::Bit64, nn::svc::ThreadContext>( ThreadId, nnThreadContext ) );
    }

    if( result.IsFailure() )
    {
        TMA_PRINTF( "ThreadDefinition::GetThreadContext - FAILURE: Result = %d\n", result.GetInnerValueForDebug() );
        return( result );
    }

    ASSERT( sizeof(tma::dbg::thread_context) >= sizeof(nn::svc::ThreadContext) );
    memcpy( (void*)pThreadContext, (void*)&nnThreadContext, sizeof(nnThreadContext) );

    if( pThreadContext->sp == 0 ) // !pProcess->Is64Bit() ) // Fixup the context info
    {
        pThreadContext->sp  = pThreadContext->cpuRegisters[ ARM32_REGISTER_SP ]; // Copy registers to named locations
        pThreadContext->lr  = pThreadContext->cpuRegisters[ ARM32_REGISTER_LR ];
        pThreadContext->cpuRegisters[ ARM32_REGISTER_PC      ]  = pThreadContext->pc; // Copy registers to numeric locations
        pThreadContext->cpuRegisters[ ARM32_REGISTER_CONTROL ]  = pThreadContext->cpsr;
    }

    return( result );
}

//==============================================================================

s32 ThreadDefinition::GetRegister( nn::svc::Handle ProcessHandle, s32 ThreadId, s32 iRegister, u64* pRegister )
{
    ASSERT( pRegister );

    tma::dbg::thread_context ThreadContext;
    nn::Result result = GetThreadContext( &ThreadContext, ProcessHandle, ThreadId, ALL_REGISTERS );

    if( result.IsSuccess() )
    {
        *pRegister = ThreadContext.cpuRegisters[ iRegister ];
    }
    else
    {
        return CheckProcessHandleForError( ProcessHandle );
    }
    return tmapi::RESULT_OK;
}

//==============================================================================

s32 ThreadDefinition::GetState( nn::svc::Handle ProcessHandle, s32 ThreadId, u64* pSP, u64* pIP )
{
    ASSERT( pSP );
    ASSERT( pIP );

    TMA_PRINTF( "ThreadDefinition::GetState - Handle: %x, ThreadId: %d\n", *(u32*)&ProcessHandle, ThreadId );

    tma::dbg::thread_context ThreadContext;
    nn::Result result = GetThreadContext( &ThreadContext, ProcessHandle, ThreadId, ALL_REGISTERS );

    if( result.IsSuccess() )
    {
        *pSP = ThreadContext.sp;
        *pIP = ThreadContext.pc;
    }
    else
    {
        // Defaults.
        *pSP = 0;
        *pIP = 0;

        return CheckProcessHandleForError( ProcessHandle );
    }
    return tmapi::RESULT_OK;
}

//==============================================================================

ThreadDefinition::ThreadDefinition( u64 ThreadId )
:   m_ThreadId( ThreadId )
,   m_Core( 0 )
,   m_pNext( NULL )
{
    memset( m_Name, 0, sizeof(m_Name) );
    memset( &m_Info, 0, sizeof(m_Info) );

    //Give ourselves a default name.
    // NOTICE that sprintf crashes the agent when we call it here.
    //TMA_SPRINTF( m_Name, "Thread %d", PRINTABLE_THREAD_ID( ThreadId ) );
    strcpy( m_Name, "Thread " );

    //===================================================================
    // We need to run through the loop below at least one time.
    // If we won't, then this is an easy one.
    //===================================================================
    if (ThreadId == 0)
    {
        strcat( m_Name, "0" );
    }

    else
    {
        //We'll need a temp buffer
        char TempBuffer[(32 - 8)]; //The "-8" is for the "Thread " bit that's already there.

        //Add the EOL, which we'll copy over at the last step.
        TempBuffer[(sizeof(TempBuffer) - 1)] = 0;

        //Keep track of where we are.
        int WriteToIndex = sizeof(TempBuffer) - 2;

        //Now loop through until we've converted the number.
        u32 Value = PRINTABLE_THREAD_ID( ThreadId );
        while (Value != 0 && WriteToIndex >= 0)
        {
            int Remainder = Value % 10;
            TempBuffer[WriteToIndex] = (Remainder + '0');
            WriteToIndex -= 1;
            Value = Value / 10;
        }

        //Now just copy the converted string over.
        char* pCopyFrom = &TempBuffer[WriteToIndex + 1];
        strcat( m_Name, pCopyFrom );
    }

}

//==============================================================================

ThreadDefinition::~ThreadDefinition()
{
    //Don't delete the guy we're linked to.  Someone else does that.
}

//==============================================================================

nn::osdbg::ThreadInfo ThreadDefinition::GetThreadInfo()
{
    if( m_Info._stackSize == 0 )
    {
        nn::osdbg::ThreadInfo threadInfo;
        nn::Result result = dbghlp::InitializeThreadInfo( &threadInfo, m_Info._debugHandle, &m_Info._debugInfoCreateProcess, &m_Info._debugInfoCreateThread );

        if( ! result.IsSuccess() )
        {
            NN_SDK_LOG("tma::dbg::ThreadDefinition::GetThreadInfo: tma::dbghelp::InitializeThreadInfo FAILED for thread %d:  Error = 0x%x\n",
                m_ThreadId, result.GetInnerValueForDebug() );
        }
        else
        {
            m_Info = threadInfo;
            //tma::dbg::MemToLog( &m_Info, sizeof(m_Info), "Updated m_Info:\n" ); /////////////// TEMPORARY
        }
    }
    return m_Info;
}

//==============================================================================

tmapi::result ThreadDefinition::Update( nn::svc::Handle ProcessHandle, nn::dmnt::ThreadData& Data )
{
    s_ThreadContextCache.insert( std::pair<nn::Bit64, nn::svc::ThreadContext>( m_ThreadId, Data.m_Context ) );

    m_SP = Data.m_SP;
    m_IP = Data.m_IP;

    if( m_SP == 0 ) // !pProcess->Is64Bit() ) // Fixup the context info
    {
        tma::dbg::thread_context Context;
        GetThreadContext( &Context, ProcessHandle, m_ThreadId, ALL_REGISTERS );
        m_SP =  Context.cpuRegisters[ ARM32_REGISTER_SP ];
    }

    m_Priority = Data.m_Priority;

    bool Frozen = false;
    bool Success = tma::threadfrozen::GetThreadFrozen( m_ThreadId, Frozen );
    m_Status = (Success && Frozen) ? 'F': Data.m_Status;

    m_Core          = Data.m_Core;
    m_IdealCore     = Data.m_IdealCore;
    m_AffinityMask  = Data.m_AffinityMask;

    if( strlen( Data.m_Name ) > 0 )
    {
        SetName( Data.m_Name );
    }

    if( Data.m_StackSize != 0 )
    {
        m_Info._stackSize = Data.m_StackSize;
    }

    if( Data.m_Stack != 0 )
    {
        m_Info._stack = Data.m_Stack;
    }

    return tmapi::RESULT_OK;
}

//==============================================================================

nn::Result ThreadDefinition::UpdateAll( nn::svc::Handle ProcessHandle, ThreadDefinitionCollection* pThreadCollection, s32 NumberOfThreads )
{
//NN_SDK_LOG("[tma] ThreadDefinition::UpdateAll %d threads\n", NumberOfThreads );

    s32 Index = 0;
    ThreadDefinition* pList = pThreadCollection->GetThreadList();
    void* pMem = s_Allocate( sizeof(nn::dmnt::ThreadData) * NumberOfThreads );
    nn::dmnt::ThreadData* pThreadData = new (pMem) nn::dmnt::ThreadData[NumberOfThreads];
    memset( pThreadData, 0, sizeof(nn::dmnt::ThreadData) * NumberOfThreads );

    pMem = s_Allocate( sizeof(nn::osdbg::ThreadInfo) * NumberOfThreads );
    nn::osdbg::ThreadInfo* pThreadInfo = new (pMem) nn::osdbg::ThreadInfo[NumberOfThreads];
    memset( pThreadInfo, 0, sizeof(nn::osdbg::ThreadInfo) * NumberOfThreads );
    while( pList )
    {
        pThreadInfo[Index++] = pList->m_Info;
        pList = pList->GetNext();
    }

    dbghlp::GetAllDebugThreadInfo( ProcessHandle, pThreadData, sizeof(ThreadData), pThreadInfo, NumberOfThreads );

    Index = 0;
    pList = pThreadCollection->GetThreadList();
    while( pList )
    {
        nn::dmnt::ThreadData Thread = pThreadData[Index++];
        ThreadDefinition* SpecificDef = pThreadCollection->Find( Thread.m_ThreadId );
        if( SpecificDef != NULL )
        {
            SpecificDef->Update( ProcessHandle, Thread );
        }
        pList = pList->GetNext();
    }
    delete[]( pThreadInfo );
    delete[]( pThreadData );

    // Errors can be returned selectively for different types of thread information given a specific state. Do our best, rely on solid defaults.
    return nn::ResultSuccess();
}

//==============================================================================

u64  ThreadDefinition::GetIP()
{
    return m_IP;
}

//==============================================================================

u64  ThreadDefinition::GetSP()
{
    return m_SP;
}

//==============================================================================

char ThreadDefinition::GetStatus()
{
    return m_Status;
}

//==============================================================================

u64  ThreadDefinition::GetThreadId()
{
    return m_ThreadId;
}

//==============================================================================

u8   ThreadDefinition::GetCore( )
{
    return m_Core;
}

//==============================================================================

u32  ThreadDefinition::GetIdealCore( )
{
    return m_IdealCore;
}

//==============================================================================

u32  ThreadDefinition::GetAffinityMask( )
{
    return m_AffinityMask;
}

//==============================================================================

u16  ThreadDefinition::GetPriority()
{
    return m_Priority;
}

//==============================================================================

char*   ThreadDefinition::GetName()
{
    return m_Name;
}

//==============================================================================

void ThreadDefinition::SetInfo( nn::osdbg::ThreadInfo* pInfo )
{
    m_Info = *pInfo;
}

//==============================================================================

u64 ThreadDefinition::GetStackAddr()
{
    //m_Info._stack = (u64)nn::osdbg::GetThreadStackAddress( &m_Info );
    GetThreadInfo(); // Ensure that the ThreadInfo has been initialized
    return m_Info._stack;
}

//==============================================================================

u64 ThreadDefinition::GetStackSize()
{
    //m_Info._stackSize = (u64)nn::osdbg::GetThreadStackSize( &m_Info );
    GetThreadInfo(); // Ensure that the ThreadInfo has been initialized
    return m_Info._stackSize;
}

//==============================================================================

ThreadDefinition* ThreadDefinition::GetNext()
{
    return m_pNext;
}

//==============================================================================

void ThreadDefinition::SetNext(ThreadDefinition* pNext)
{
    m_pNext = pNext;
}

//==============================================================================

void ThreadDefinition::SetName( char* pName )
{
    if( pName != NULL)
    {
        int Length = strlen( pName );
        int Index = 0;
        for ( ; Index < Length; Index += 1)
        {
            if (Index == nn::os::ThreadNameLengthMax - 1)
            {
                break;
            }
            m_Name[Index] = pName[Index];
        }

        //Insure an EOL.
        for (; Index < nn::os::ThreadNameLengthMax; Index += 1 )
        {
            m_Name[Index] = 0;
        }
    }
}

//==============================================================================

void ThreadDefinition::ToLog()
{
    tma::dbg::MemToLog( this, sizeof(ThreadDefinition), "ThreadDefinition:\n" );
}

//==============================================================================
// Thread definition collection functionality
//==============================================================================
ThreadDefinitionCollection::ThreadDefinitionCollection( )
:   m_ThreadList(NULL)
{
    m_Mutex.Create();
}

ThreadDefinitionCollection::~ThreadDefinitionCollection()
{
    Clear();
    m_Mutex.Destroy();
}


//==============================================================================
// List management.
//==============================================================================

void ThreadDefinitionCollection::Add( u64 ThreadId, nn::osdbg::ThreadInfo* pInfo )
{
    //NN_SDK_LOG("[tma] ThreadDefinitionCollection:Add thread %d\n", ThreadId );
    //TMA_TRACE("ThreadDefinitionCollection::Add", "Adding %d", PRINTABLE_THREAD_ID( ThreadId ) );
    ScopedLock lock(m_Mutex);
    ThreadDefinition* pThread = Find( ThreadId );
    if ( pThread == NULL )
    {
        void* pMem = s_Allocate( sizeof(ThreadDefinition) );
        pThread = new (pMem) ThreadDefinition( ThreadId );
        Add( pThread );
    }

    pThread->SetInfo( pInfo );
}

//==============================================================================

void ThreadDefinitionCollection::Remove( u64 ThreadId )
{
//    TMA_TRACE("ThreadDefinitionCollection::Remove", "Remove %d", ThreadId );
    ScopedLock lock( m_Mutex );
    ThreadDefinition* pList = m_ThreadList;
    while ( pList )
    {
        if ( pList->GetThreadId() == ThreadId )
        {
            Remove( pList );
            return;
        }
        pList = pList->GetNext();
    }
    //Good place for a "couldn't find this one" assert.
}

//==============================================================================
// Private function that removes the passed definition from our list and returns
// the next object in this list.
// NOTICE that this assumes the list is locked.
void ThreadDefinitionCollection::Remove( ThreadDefinition* pRemove )
{
    tma::threadfrozen::ClearThreadFrozen( pRemove->GetThreadId() );
    //Save the entries on the back side of this node
     ThreadDefinition* pSaveNext = pRemove->GetNext();

    //Check for the special case of removing the head.
    if( m_ThreadList == pRemove )
    {
        m_ThreadList = pSaveNext;
        pRemove->~ThreadDefinition();
        tma::s_Deallocate( pRemove, sizeof( ThreadDefinition ) );
    }
    else
    {
        ThreadDefinition* pPrev = m_ThreadList;
        ThreadDefinition* pList = m_ThreadList->GetNext();
        while ( pList )
        {
            if ( pList == pRemove )
            {
                //Cut it out of our list.
                pPrev->SetNext( pSaveNext );
                pList->~ThreadDefinition();
                tma::s_Deallocate( pList, sizeof( ThreadDefinition ) );
                return;
            }

            pPrev = pList;
            pList = pList->GetNext();
        }
    }
}

//==============================================================================
// Private function that add the passed definition to our list.
// NOTICE that this assumes the list is locked.
void ThreadDefinitionCollection::Add( ThreadDefinition* pAdd )
{
    tma::threadfrozen::SetThreadFrozen( pAdd->GetThreadId(), false );
    if ( m_ThreadList == NULL )
    {
        m_ThreadList = pAdd;
    }
    else
    {
        ThreadDefinition* pList = m_ThreadList;
        while (pList->GetNext() != NULL)
        {
            pList = pList->GetNext();
        }
        pList->SetNext( pAdd );
    }
}

//==============================================================================
// Private function that checks our list to see if we already know about this thread.
// NOTICE that this assumes the list is locked.
ThreadDefinition* ThreadDefinitionCollection::Find( u64 ThreadId )
{
//    TMA_TRACE("ThreadDefinitionCollection::Find", "Looking for %d", PRINTABLE_THREAD_ID(ThreadId) );
    ThreadDefinition* pList = m_ThreadList;
    while ( pList )
    {
        if ( pList->GetThreadId() == ThreadId )
        {
            return pList;
        }
        pList = pList->GetNext();
    }

    return NULL;
}

//==============================================================================

bool ThreadDefinitionCollection::Exists( u64 ThreadId )
{
    return ( Find( ThreadId ) != NULL );
}

//==============================================================================

void ThreadDefinitionCollection::Clear()
{
//    TMA_TRACE("ThreadDefinitionCollection::Clear", "Clear" );
    ScopedLock lock( m_Mutex );
    ThreadDefinition* pList = m_ThreadList;
    while ( pList )
    {
        ThreadDefinition* pNext = pList->GetNext();
        //delete pList;
        pList->~ThreadDefinition();
        tma::s_Deallocate( pList, sizeof( ThreadDefinition ) );
        pList = pNext;
    }
    m_ThreadList = NULL;
}

//==============================================================================

void ThreadDefinitionCollection::Update( int32_t NumberOfActiveThreads, uint64_t* pActiveThreadIds )
{
//    TMA_TRACE("ThreadDefinitionCollection::Update", "Update:  Number of threads = %d", NumberOfActiveThreads );
    ScopedLock lock( m_Mutex );

    //Go through this list and see who we're missing.
    for ( int Index = 0; Index < NumberOfActiveThreads; Index += 1 )
    {
        uint64_t ThreadId = pActiveThreadIds[Index];
        if( Exists( ThreadId ) == false )
        {
            //TMA_TRACE("ThreadDefinitionCollection::Update", "Adding %d", PRINTABLE_THREAD_ID(ThreadId) );
            void* pMem = s_Allocate( sizeof(ThreadDefinition) );
            ThreadDefinition* pNewThread = new (pMem) ThreadDefinition( ThreadId );
            Add( pNewThread );
        }
    }


    //Now go through our list and see who we have that we shouldn't
    ThreadDefinition* pList = m_ThreadList;
    while ( pList )
    {
        s32 ThreadId = pList->GetThreadId();
        ThreadDefinition* pNext = pList->GetNext();
        int FindIndex = 0;
        for ( ; FindIndex < NumberOfActiveThreads; FindIndex += 1 )
        {
            if( ThreadId == pActiveThreadIds[FindIndex] )
            {
                break;
            }
        }
        if( FindIndex == NumberOfActiveThreads )
        {
            //Didn't find him, so cut him out.
            Remove( pList );
        }

        pList = pNext;
    }

}

//==============================================================================

ThreadDefinition* ThreadDefinitionCollection::Get( s32 AtIndex )
{
    //TMA_TRACE("ThreadDefinitionCollection::Get", "Get %d", AtIndex );
    ScopedLock lock( m_Mutex );
    s32 Index = 0;
    ThreadDefinition* pList = m_ThreadList;
    while ( pList )
    {
        if ( Index == AtIndex )
        {
            return pList;
        }
        pList = pList->GetNext();
        Index += 1;
    }
    return NULL;
}

//==============================================================================

ThreadDefinition* ThreadDefinitionCollection::GetThreadList()
{
    return m_ThreadList;
}

//==============================================================================

s32 ThreadDefinitionCollection::Count()
{
    //TMA_TRACE("ThreadDefinitionCollection::Count", "Count" );
    ScopedLock lock( m_Mutex );
    s32 Count = 0;
    ThreadDefinition* pList = m_ThreadList;
    while ( pList )
    {
        Count += 1;
        pList = pList->GetNext();
    }
    return Count;
}

//==============================================================================

u64 ThreadDefinitionCollection::GetDefaultId()
{
    ScopedLock lock( m_Mutex );
    if( m_ThreadList != NULL )
    {
        return m_ThreadList->GetThreadId();
    }

    return 0;
}

//==============================================================================

bool ThreadDefinitionCollection::GetPerThreadStepInfo( u64 ThreadId, per_thread_step_info& PTSI )
{
    //TMA_TRACE("ThreadDefinitionCollection::GetPerThreadStepInfo", "GetPerThreadStepInfo( %lld )", ThreadId );
    ScopedLock lock( m_Mutex );

    ThreadDefinition* pList = m_ThreadList;
    while ( pList )
    {
        if ( pList->m_ThreadId == ThreadId )
        {
            PTSI = pList->m_PTSI;
            return true;
        }
        pList = pList->GetNext();
    }
    return false;
}

//==============================================================================

bool ThreadDefinitionCollection::SetPerThreadStepInfo( u64 ThreadId, per_thread_step_info& PTSI )
{
    //TMA_TRACE("ThreadDefinitionCollection::SetPerThreadStepInfo", "SetPerThreadStepInfo( %lld )", ThreadId );
    ScopedLock lock( m_Mutex );

    ThreadDefinition* pList = m_ThreadList;
    while ( pList )
    {
        if ( pList->m_ThreadId == ThreadId )
        {
            pList->m_PTSI = PTSI;
            return true;
        }
        pList = pList->GetNext();
    }
    return false;
}

//==============================================================================

void ThreadDefinitionCollection::ToLog()
{
    ScopedLock lock( m_Mutex );

    ThreadDefinition* pList = m_ThreadList;
    while ( pList )
    {
        pList->ToLog();

        pList = pList->GetNext();
    }
}

//==============================================================================
}} // namespace
//==============================================================================
