﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include "dbg_opcodes.h"
#include "dbg_Task.h"
#include "dbg_Command.h"
#include "dbg_Modules.h"
#include "dbg_Threads.h"
#include "dbg_CommandArgs.h"
#include "dbg_Exception.h"
#include "dbg_Service.h"
#include "../tm_result.h"

// TODO: this is temporary, until a proper #define (or whatnot) is
//       created and [ideally] centrally located.  For now, 0 means
//       "No Process"
static const u64 NoProcessId(0);

enum
{
    DEFAULT_RETURN_BUFFER_SIZE = 2048,
    SIZE_OF_DEBUG_PACKET_HEADER = sizeof(s32) + // MessageType
                                  sizeof(u64) + // ProcessId
                                  sizeof(s32),  // ResultCode
    MAX_SIZE_DEBUG_PACKET_DATA = ( (tmipc::MAX_PACKET_DATA) - (SIZE_OF_DEBUG_PACKET_HEADER) ),
};

//==============================================================================
namespace tma { namespace dbg {
//==============================================================================

Thread  DebugTask::m_WorkThread;
MQ      DebugTask::m_WorkMQ;
enum
{
    StackSize = (8192 * 10)
};

//==============================================================================

void DebugTask::Initialize()
{
    //TMA_TRACE("DebugTask", "Install");
    m_WorkMQ.Create( 32 );
    m_WorkThread.Start( WorkThread, NULL, StackSize, NN_SYSTEM_THREAD_PRIORITY(tma, DbgWorkThread), NN_SYSTEM_THREAD_NAME(tma, DbgWorkThread) );
}

//==============================================================================

void DebugTask::Finalize()
{
    //TMA_TRACE("DebugTask", "Remove");
    // Notify the thread that it's time to quit
    m_WorkMQ.Send( NULL );
    m_WorkThread.Join();
    m_WorkThread.Destroy();
    m_WorkMQ.Destroy();
}

//==============================================================================

DebugTask::DebugTask( ProcessMgr* pProcessMgr )
{
    m_pProcessMgr = pProcessMgr;
    m_pThread = NULL;
    m_TaskType = tmipc::TaskType_DebugTask;
}

//==============================================================================

DebugTask::~DebugTask()
{
    m_pProcessMgr = NULL;
    if( m_pThread != NULL )
    {
        m_pThread->Destroy();
        delete m_pThread;
    }
}

//==============================================================================

void DebugTask::SendReply( tmipc::Task* pTask, tma::dbg::TMAgent_message_type MessageType, u64 ProcessId, s32 ResultCode, void* Data, s32 DataSize )
{
    tmipc::Packet* pPacket = pTask->AllocSendPacket();
    pPacket->WriteS32( MessageType );
    pPacket->WriteU64( ProcessId );
    pPacket->WriteS32( ResultCode );
    switch( MessageType )
    {
        case tma::dbg::TMAGENT_MESSAGE_TYPE_NUMERIC_S32:
            pPacket->WriteS32( *(s32*)Data );
            break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_NUMERIC_U32:
            pPacket->WriteU32( *(u32*)Data );
            break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_NUMERIC_S64:
            pPacket->WriteS64( *(s64*)Data );
            break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_NUMERIC_U64:
            pPacket->WriteU64( *(u64*)Data );
            break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_MODULE:
            pPacket->WriteU64(    ((tma::dbg::ModuleDefinition*)Data)->GetAddress() );
            pPacket->WriteU64(    ((tma::dbg::ModuleDefinition*)Data)->GetSize() );
            pPacket->WriteData(   ((tma::dbg::ModuleDefinition*)Data)->GetID(), tma::dbg::ModuleDefinition::MODULE_ID_SIZE );
            pPacket->WriteString( ((tma::dbg::ModuleDefinition*)Data)->GetName() );
            break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_PROCESS:
            pPacket->WriteU64(  ((tma::dbg::ProcessDefinition*)Data)->m_PID );
            pPacket->WriteS32(  ((tma::dbg::ProcessDefinition*)Data)->m_NumberOfModules );
            pPacket->WriteU32(  ((tma::dbg::ProcessDefinition*)Data)->m_NumThreads );
            pPacket->WriteU32(  ((tma::dbg::ProcessDefinition*)Data)->m_LastExceptionId );
            pPacket->WriteU8(   ((tma::dbg::ProcessDefinition*)Data)->m_64BitFlags );
            pPacket->WriteU8(   ((tma::dbg::ProcessDefinition*)Data)->m_State );
            pPacket->WriteData( ((tma::dbg::ProcessDefinition*)Data)->m_Name, PROCESS_NAME_MAX_LENGTH );
            break;

        case tma::dbg::TMAGENT_MESSAGE_TYPE_THREAD:
            pPacket->WriteS32(  ((tma::dbg::ThreadData*)Data)->m_Data.m_ThreadId );
            pPacket->WriteU64(  ((tma::dbg::ThreadData*)Data)->m_Data.m_IP );
            pPacket->WriteU64(  ((tma::dbg::ThreadData*)Data)->m_Data.m_SP );
            pPacket->WriteU32(  ((tma::dbg::ThreadData*)Data)->m_Data.m_Status );
            pPacket->WriteU32(  ((tma::dbg::ThreadData*)Data)->m_Data.m_Core );
            pPacket->WriteU32(  ((tma::dbg::ThreadData*)Data)->m_Data.m_IdealCore );
            pPacket->WriteU32(  ((tma::dbg::ThreadData*)Data)->m_Data.m_Priority );
            pPacket->WriteU32(  ((tma::dbg::ThreadData*)Data)->m_Data.m_PriorityBase );
            pPacket->WriteU32(  ((tma::dbg::ThreadData*)Data)->m_Data.m_Flags );
            pPacket->WriteData( ((tma::dbg::ThreadData*)Data)->m_Data.m_Name, nn::os::ThreadNameLengthMax );
            pPacket->WriteU32(  ((tma::dbg::ThreadData*)Data)->m_Data.m_AffinityMask );
            break;

        case tma::dbg::TMAGENT_MESSAGE_TYPE_MEMORY_SEGMENT:
        case tma::dbg::TMAGENT_MESSAGE_TYPE_RAW:
            pPacket->WriteData( Data, DataSize );
            break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_RESULT:
            break;
        case tma::dbg::TMAGENT_MESSAGE_TYPE_UNKNOWN:
            break;
        default:
            pPacket->WriteString( (const char*)Data );
            break;
    }

    pTask->GetServicesManager()->Send( pPacket );
    TMA_TRACE( "DebugTask", "SendReply - Sent" );
}

//==============================================================================

void* DebugTask::WorkThread( void* pArgs )
{
    //TMA_TRACE( "DebugTask", "LoadExecutableThread - Start" );

    s32 Result = 0;
    bool bDone = false;
    while( !bDone )
    {
        // Read the next work item.
        WorkTaskInfo* pInfo = reinterpret_cast<WorkTaskInfo*>(m_WorkMQ.Receive());

        // Process the parameter.
        if( pInfo != NULL )
        {
            if( pInfo->m_Execute != NULL )
            {
                pInfo->m_Execute( pInfo );
            }
            else
            {
                NN_SDK_LOG("ERROR:  WorkTaskInfo->m_Execute = NULL for command %d\n", pInfo->m_CommandCode );
            }

            pInfo->m_pDebugTask->Complete();

            // Deallocate the WorkInfo struct.
            delete pInfo;
            pInfo = NULL;
        }
        else  //NULL means close this thread.
        {
            //TMA_TRACE( "DebugTask", "LoadExecutableThread - Received NULL packet." );
            bDone = true;
        }
    }

    //TMA_TRACE( "DebugTask", "LoadExecutableThread - Exit" );
    return NULL;
}

//==============================================================================

void DebugTask::IssueSimpleCommand( WorkTaskInfo* pInfo )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    char ReturnBuffer[DEFAULT_RETURN_BUFFER_SIZE];
    memset( ReturnBuffer, 0, DEFAULT_RETURN_BUFFER_SIZE );
    tma::dbg::Command Command( pInfo->m_CommandCode, pInfo->m_ProcessId, pInfo->m_ParamValue, 0, NULL );
    TMAgent_message_type Type = pInfo->m_pProcessMgr->HandleCommand( &Command, ResultCode, ReturnBuffer );
    SendReply( pInfo->m_pDebugTask, Type, pInfo->m_ProcessId, ResultCode, ReturnBuffer, 0 );
}

//==============================================================================

void DebugTask::SetBreakpoint( WorkTaskInfo* pInfo )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    TMA_PRINTF( "HandleBreakpointCommand - Addr: %llx, State: %d, Instr: %08x\n", pInfo->m_BreakPoint.m_Address, pInfo->m_BreakPoint.m_State, pInfo->m_BreakPoint.m_Instruction );

    tma::dbg::Command Command( TMA_OP_SET_BKPT, pInfo->m_ProcessId, 0, sizeof(pInfo->m_BreakPoint), &pInfo->m_BreakPoint );
    tma::dbg::TMAgent_message_type Type = pInfo->m_pProcessMgr->HandleCommand( &Command, ResultCode, &pInfo->m_BreakPoint );
    SendReply( pInfo->m_pDebugTask, Type, pInfo->m_ProcessId, ResultCode, &pInfo->m_BreakPoint, sizeof(pInfo->m_BreakPoint) );

    TMA_PRINTF( "HandleBreakpointCommand - Addr: %llx, State: %d, Instr: %08x\n", pInfo->m_BreakPoint.m_Address, pInfo->m_BreakPoint.m_State, pInfo->m_BreakPoint.m_Instruction );
}

//==============================================================================

void DebugTask::Step( WorkTaskInfo* pInfo )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    tma::dbg::Command Command( TMA_OP_STEP_DEBUG, pInfo->m_ProcessId, 0, sizeof(StepDebugCommandArgs), &pInfo->m_StepArgs );
    tma::dbg::TMAgent_message_type Type = pInfo->m_pProcessMgr->HandleCommand( &Command, ResultCode, NULL );
    SendReply( pInfo->m_pDebugTask, Type, pInfo->m_ProcessId, ResultCode, NULL, 0 );
}

//==============================================================================

void DebugTask::WriteMemory( WorkTaskInfo* pInfo )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    tma::dbg::Command Command( TMA_OP_WRITE_MEM, pInfo->m_ProcessId, 0, sizeof(WriteMemoryCommandArgs), &pInfo->m_WriteMemoryArgs );
    tma::dbg::TMAgent_message_type Type = pInfo->m_pProcessMgr->HandleCommand( &Command, ResultCode, NULL );
    SendReply( pInfo->m_pDebugTask, Type, pInfo->m_ProcessId, ResultCode, NULL, 0 );
}

//==============================================================================

void DebugTask::ReadMemory( WorkTaskInfo* pInfo )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    u8      ReturnBuffer[DEFAULT_RETURN_BUFFER_SIZE];
    u8*     pReturnBuffer = ReturnBuffer;

    if( pInfo->m_ReadMemoryArgs.m_DataSize > DEFAULT_RETURN_BUFFER_SIZE )
    {
        void* pMem =      s_Allocate( pInfo->m_ReadMemoryArgs.m_DataSize);
        pReturnBuffer = new (pMem) u8[pInfo->m_ReadMemoryArgs.m_DataSize];
    }
    memset( pReturnBuffer, 0, (size_t)pInfo->m_ReadMemoryArgs.m_DataSize );

    TMA_TRACE( "DebugTask", "HandleReadMemCommand( %llx )", pInfo->m_ReadMemoryArgs.m_Address );

    tma::dbg::Command Command( TMA_OP_READ_MEM, pInfo->m_ProcessId, 0, sizeof(ReadMemCommandArgs), &pInfo->m_ReadMemoryArgs );
    tma::dbg::TMAgent_message_type Type = pInfo->m_pProcessMgr->HandleCommand( &Command, ResultCode, pReturnBuffer );
    SendReply( pInfo->m_pDebugTask, Type, pInfo->m_ProcessId, ResultCode, pReturnBuffer, pInfo->m_ReadMemoryArgs.m_DataSize );
    if( pReturnBuffer != (void*)ReturnBuffer )
    {
        delete[] pReturnBuffer;
    }
}

//==============================================================================

void DebugTask::ReadRegisterDefinitions( WorkTaskInfo* pInfo )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    u8      ReturnBuffer[DEFAULT_RETURN_BUFFER_SIZE];
    u8*     pReturnBuffer = ReturnBuffer;

    if( pInfo->m_ReadRegDefsArgs.m_DataSize > DEFAULT_RETURN_BUFFER_SIZE )
    {
        void* pMem =      s_Allocate( (size_t)pInfo->m_ReadRegDefsArgs.m_DataSize );
        pReturnBuffer = new (pMem) u8[(size_t)pInfo->m_ReadRegDefsArgs.m_DataSize];
    }
    memset( pReturnBuffer, 0, (size_t)pInfo->m_ReadRegDefsArgs.m_DataSize );

    TMA_TRACE( "DebugTask", "HandleReadRegDefsCommand( %d )", pInfo->m_ReadRegDefsArgs.m_DataSize );

    tma::dbg::Command Command( TMA_OP_READ_REG_DEFS, pInfo->m_ProcessId, 0, sizeof(ReadRegDefsCommandArgs), &pInfo->m_ReadRegDefsArgs);
    tma::dbg::TMAgent_message_type Type = pInfo->m_pProcessMgr->HandleCommand( &Command, ResultCode, pReturnBuffer );

    s32 DataSize = pInfo->m_ReadRegDefsArgs.m_DataSize;

    if( ResultCode != tmapi::result::RESULT_OK )
    {
        DataSize = 0;

        if( ResultCode == tmapi::result::RESULT_BUFFER_TOO_SMALL )
        {
            DataSize = sizeof(u32); // Enough to return correct size
        }
    }
    SendReply( pInfo->m_pDebugTask, Type, pInfo->m_ProcessId, ResultCode, pReturnBuffer, DataSize );

    TMA_TRACE( "DebugTask", "HandleReadRegDefsCommand( %d ), ResultCode = %d, DataSize = %d", pInfo->m_ReadRegDefsArgs.m_DataSize, ResultCode, DataSize );

    if( pReturnBuffer != (void*)ReturnBuffer )
    {
        delete[] pReturnBuffer;
    }
}

//==============================================================================

void DebugTask::ReadRegisterData( WorkTaskInfo* pInfo )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    u8      ReturnBuffer[DEFAULT_RETURN_BUFFER_SIZE];
    u8*     pReturnBuffer = ReturnBuffer;

    if( pInfo->m_ReadRegisterArgs.m_DataSize > DEFAULT_RETURN_BUFFER_SIZE )
    {
        void* pMem =      s_Allocate( (size_t)pInfo->m_ReadRegisterArgs.m_DataSize );
        pReturnBuffer = new (pMem) u8[(size_t)pInfo->m_ReadRegisterArgs.m_DataSize];
    }
    memset( pReturnBuffer, 0, (size_t)pInfo->m_ReadRegisterArgs.m_DataSize );

    TMA_TRACE( "DebugTask", "HandleReadRegDataCommand" );

    tma::dbg::Command Command( TMA_OP_READ_REG_DATA, pInfo->m_ProcessId, 0, sizeof(ReadRegDataCommandArgs), &pInfo->m_ReadRegisterArgs );
    tma::dbg::TMAgent_message_type Type = pInfo->m_pProcessMgr->HandleCommand( &Command, ResultCode, pReturnBuffer );
    SendReply( pInfo->m_pDebugTask, Type, pInfo->m_ProcessId, ResultCode, pReturnBuffer, pInfo->m_ReadRegisterArgs.m_DataSize );

    if( pReturnBuffer != (void*)ReturnBuffer )
    {
        delete[] pReturnBuffer;
    }
}

//==============================================================================

void DebugTask::WriteRegisterData( WorkTaskInfo* pInfo )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    tma::dbg::Command Command( TMA_OP_WRITE_REG, pInfo->m_ProcessId, 0, sizeof(WriteRegisterCommandArgs), &pInfo->m_WriteRegisterArgs );
    tma::dbg::TMAgent_message_type Type = pInfo->m_pProcessMgr->HandleCommand( &Command, ResultCode, NULL );
    SendReply( pInfo->m_pDebugTask, Type, pInfo->m_ProcessId, ResultCode, NULL, 0 );
}

//==============================================================================

void DebugTask::WriteDump( WorkTaskInfo* pInfo )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    tma::dbg::Command Command( TMA_OP_WRITE_DUMP, pInfo->m_ProcessId, 0, sizeof(WriteDumpCommandArgs), &pInfo->m_WriteDumpArgs );
    tma::dbg::TMAgent_message_type Type = pInfo->m_pProcessMgr->HandleCommand( &Command, ResultCode, NULL );
    SendReply( pInfo->m_pDebugTask, Type, pInfo->m_ProcessId, ResultCode, NULL, 0 );
}

//==============================================================================

void DebugTask::GetMemorySegmentInfo( WorkTaskInfo* pInfo )
{
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    u8      ReturnBuffer[DEFAULT_RETURN_BUFFER_SIZE];
    u8*     pReturnBuffer = ReturnBuffer;

    if( pInfo->m_ParamValue > DEFAULT_RETURN_BUFFER_SIZE )
    {
        void* pMem =      s_Allocate( (size_t)pInfo->m_ParamValue);
        pReturnBuffer = new (pMem) u8[(size_t)pInfo->m_ParamValue];
    }
    memset( pReturnBuffer, 0, (size_t)pInfo->m_ParamValue );

    tma::dbg::Command Command( TMA_OP_MEMORY_SEGMENT_INFO, pInfo->m_ProcessId, pInfo->m_ParamValue, 0, NULL );
    tma::dbg::TMAgent_message_type Type = pInfo->m_pProcessMgr->HandleCommand( &Command, ResultCode, pReturnBuffer );
    SendReply( pInfo->m_pDebugTask, Type, pInfo->m_ProcessId, ResultCode, pReturnBuffer, pInfo->m_ParamValue );

    if( pReturnBuffer != (void*)ReturnBuffer )
    {
        delete[] pReturnBuffer;
    }
}

//==============================================================================

void* DebugTask::HandleAttachOnStart( void* pArgs )
{
    DebugTask* pThis = (DebugTask*)pArgs;
    u64 ProcessId = 0;
    s32 Result = pThis->m_pProcessMgr->HandleAttachOnLaunch( pThis->m_ProgramId, ProcessId );
    SendReply( pThis, tma::dbg::TMAGENT_MESSAGE_TYPE_NUMERIC_U64, ProcessId, Result, (void*)&ProcessId, 0 );
    pThis->Complete();
    return NULL;
}

//==============================================================================

void* DebugTask::HandleLoadExe( void* pArgs )
{
    DebugTask* pThis = (DebugTask*)pArgs;
    u64 ProcessId = 0;
    s32 Result = pThis->m_pProcessMgr->HandleLoadElf( pThis->m_FileName, pThis->m_Args, ProcessId );
    SendReply( pThis, tma::dbg::TMAGENT_MESSAGE_TYPE_NUMERIC_U64, ProcessId, Result, (void*)&ProcessId, 0 );
    pThis->Complete();
    return NULL;
}

//==============================================================================

void* DebugTask::HandleLoadExe2( void* pArgs )
{
    DebugTask* pThis = (DebugTask*)pArgs;
    u64 ProcessId = 0;
    nn::Result res = pThis->m_pProcessMgr->LoadExe( pThis->m_FileName, pThis->m_Args, ProcessId );

    // Here's what TMS is expecting from the TMA_OP_LOAD_EXE2 command.
    struct ProcessLoadResult
    {
        ProcessLoadResult( u64 ProcessId, nn::Result loadResult )
        {
            m_ProcessId     = ProcessId;
            m_NativeResult  = loadResult.GetInnerValueForDebug();
        }

        u64 m_ProcessId;
        u32 m_NativeResult;
    };

    ProcessLoadResult send( ProcessId, res );

    // Result is good?
    s32 Result = tmapi::result::RESULT_OK;
    if( res.IsFailure() )
    {
        Result = tmapi::result::RESULT_PROCESS_CREATE_FAILED;
    }

    SendReply( pThis, tma::dbg::TMAGENT_MESSAGE_TYPE_RAW, ProcessId, Result, (void*)&send, sizeof(send) );
    pThis->Complete();
    return NULL;
}

//==============================================================================

void DebugTask::AttachByProgramId( WorkTaskInfo* pInfo )
{
    u64 ProcessId = 0;

    // We co-op the process ID field for the program Id:
    s32 Result = pInfo->m_pProcessMgr->HandleAttachByProgramId( pInfo->m_ProcessId, ProcessId );
    SendReply( pInfo->m_pDebugTask, tma::dbg::TMAGENT_MESSAGE_TYPE_NUMERIC_U64, ProcessId, Result, (void*)&ProcessId, 0 );
}

//==============================================================================

void DebugTask::OnInitiate( tmipc::Packet* pPacket )
{
    u32 CommandCode = 0;
    u32 u32Param = 0;
    u64 u64Param = 0;
    u64 ProcessId = 0;
    pPacket->ReadU32( CommandCode );
    pPacket->ReadU64( ProcessId );

    TMA_TRACE( "DebugTask", "OnInitiate( %d )", CommandCode );

    //Report on this command.
    DEJA_TRACE( "DebugTask::OnInitiate", "Received command: %d", CommandCode );
    DEJA_TRACE( "DebugTask::OnInitiate", "Got Command: %(DebugOpcodes)", CommandCode );
//    NN_SDK_LOG( "DebugTask::OnInitiate - Received command: %d\n", CommandCode  );

    //===========================================================================
    // We'll thread these tasks so they don't block other tasks that come in.
    //===========================================================================
    void* pMem = NULL;
    if( CommandCode == TMA_OP_ATTACH_ON_START )
    {
        pPacket->ReadU64( m_ProgramId );
        pMem = s_Allocate( sizeof(Thread) );
        m_pThread = new (pMem) Thread();
        m_pThread->Start( HandleAttachOnStart, this, StackSize, NN_SYSTEM_THREAD_PRIORITY(tma, HandleAttachOnStart), NN_SYSTEM_THREAD_NAME(tma, HandleAttachOnStart) );
        return;
    }
    else if( CommandCode == TMA_OP_LOAD_ELF )
    {
        pPacket->ReadString( m_FileName, sizeof(m_FileName), NULL );
        pPacket->ReadString( m_Args, sizeof(m_Args), NULL );
        pMem = s_Allocate( sizeof(Thread) );
        m_pThread = new (pMem) Thread();
        m_pThread->Start( HandleLoadExe, this, StackSize, NN_SYSTEM_THREAD_PRIORITY(tma, HandleLoadExeThread), NN_SYSTEM_THREAD_NAME(tma, HandleLoadExeThread) );
        return;
    }
    else if( CommandCode == TMA_OP_LOAD_EXE2 )
    {
        pPacket->ReadString( m_FileName, sizeof(m_FileName), NULL );
        pPacket->ReadString( m_Args, sizeof(m_Args), NULL );
        pMem = s_Allocate( sizeof(Thread) );
        m_pThread = new (pMem) Thread();
        m_pThread->Start( HandleLoadExe2, this, StackSize, NN_SYSTEM_THREAD_PRIORITY(tma, HandleLoadExeThread), NN_SYSTEM_THREAD_NAME(tma, HandleLoadExeThread) );
        return;
    }
    //===========================================================================
    // We'll thread these tasks so they don't block other tasks that come in.
    //===========================================================================
    pMem = s_Allocate( sizeof(WorkTaskInfo) );
    WorkTaskInfo* pTaskInfo = new (pMem) WorkTaskInfo( this, CommandCode, m_pProcessMgr );
    pTaskInfo->m_ProcessId = ProcessId;

    // See if we have any additional parameters
    switch( CommandCode )
    {
        case TMA_OP_CONT_DEBUG:
        case TMA_OP_MODULE_INFO:
        case TMA_OP_THREAD_INFO:
        case TMA_OP_PROCESS_INFO:
        case TMA_OP_JIT_PROCESS_INFO:
            {
                u32 ReadU32;
                pPacket->ReadU32( ReadU32 ) ;
                pTaskInfo->m_ParamValue = (u64)ReadU32;
            }
            pTaskInfo->m_Execute = IssueSimpleCommand;
            break;

        case TMA_OP_MEMORY_SEGMENT_INFO:
            {
                u32 ReadU32;
                pPacket->ReadU32( ReadU32 ) ;
                pTaskInfo->m_ParamValue = (u64)ReadU32;
            }
            pTaskInfo->m_Execute = GetMemorySegmentInfo;
            break;

        case TMA_OP_SET_BKPT:
            pPacket->ReadData( &pTaskInfo->m_BreakPoint, sizeof(pTaskInfo->m_BreakPoint) );
            pTaskInfo->m_Execute = SetBreakpoint;
            break;

        case TMA_OP_CLR_BKPT:
        case TMA_OP_START_DEBUG:
            pPacket->ReadU64( pTaskInfo->m_ParamValue );
            pTaskInfo->m_Execute = IssueSimpleCommand;
            break;

        case TMA_OP_STEP_DEBUG:
            pPacket->ReadData( &pTaskInfo->m_StepArgs, sizeof(pTaskInfo->m_StepArgs) );
            pTaskInfo->m_Execute = Step;
            break;

        case TMA_OP_WRITE_MEM:
            pPacket->ReadData( &pTaskInfo->m_WriteMemoryArgs, sizeof(pTaskInfo->m_WriteMemoryArgs) );
            pTaskInfo->m_Execute = WriteMemory;
            break;

        case TMA_OP_READ_MEM:
            pPacket->ReadData( &pTaskInfo->m_ReadMemoryArgs, sizeof(pTaskInfo->m_ReadMemoryArgs) );
            pTaskInfo->m_Execute = ReadMemory;
            break;

        case TMA_OP_READ_REG_DEFS:
            pPacket->ReadData( &pTaskInfo->m_ReadRegDefsArgs, sizeof(pTaskInfo->m_ReadRegDefsArgs) );
            pTaskInfo->m_Execute = ReadRegisterDefinitions;
            break;

        case TMA_OP_READ_REG_DATA:
            pPacket->ReadData( &pTaskInfo->m_ReadRegisterArgs, sizeof(pTaskInfo->m_ReadRegisterArgs) );
            pTaskInfo->m_Execute = ReadRegisterData;
            break;

        case TMA_OP_WRITE_REG:
            pPacket->ReadData( &pTaskInfo->m_WriteRegisterArgs, sizeof(pTaskInfo->m_WriteRegisterArgs) );
            pTaskInfo->m_Execute = WriteRegisterData;
            break;

        case TMA_OP_WRITE_DUMP:
            pPacket->ReadData( &pTaskInfo->m_WriteDumpArgs, sizeof(pTaskInfo->m_WriteDumpArgs) );
            pTaskInfo->m_Execute = WriteDump;
            break;

        case TMA_OP_NUM_JIT_PROCESSES:
        case TMA_OP_NUM_PROCESSES:
            pTaskInfo->m_ProcessId = NoProcessId;
            pTaskInfo->m_Execute = IssueSimpleCommand;
            break;

        case TMA_OP_ATTACH_BY_PROGRAM_ID:
            //=======================================================
            // We co-op the process ID field for the program Id.
            pPacket->ReadU64( pTaskInfo->m_ProcessId );
            pTaskInfo->m_Execute = AttachByProgramId;
            break;

        default:
            pTaskInfo->m_Execute = IssueSimpleCommand;
            break;
    }

    // Add the Thread Parameter to the message queue.
    m_WorkMQ.Send( pTaskInfo );
}// NOLINT(impl/function_size)

//==============================================================================

void DebugTask::ProcessCommand( WorkTaskInfo* pTaskInfo )
{
    m_WorkMQ.Send( pTaskInfo );
}

//==============================================================================

void DebugTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    //TMA_TRACE( "DebugTask", "OnRecvPacket %p", GetTaskId() );

    // Task complete.
    Complete();
}

//==============================================================================

void DebugTask::OnSendPacket( tmipc::Packet* pPacket )
{
    ASSERT( 0 );
}

//==============================================================================

void DebugTask::Start( TMAgent_message_type MessageType, process_id ProcessId,  void* pData, s32 SizeOfData )
{
    //TMA_TRACE( "DebugTask", "Start %p", GetTaskId() );

    tmipc::Packet* pPacket = AllocSendPacket( true );
    pPacket->WriteS32( MessageType );
    pPacket->WriteU64( ProcessId );
    switch( MessageType )
    {
        case tma::dbg::TMAGENT_MESSAGE_TYPE_EXCEPTION:
            pPacket->WriteS32( ((tma::dbg::Exception*)pData)->m_ThreadId );
            pPacket->WriteU64( ((tma::dbg::Exception*)pData)->m_ExceptionAddress );
            pPacket->WriteU64( ((tma::dbg::Exception*)pData)->m_IP );
            pPacket->WriteU64( ((tma::dbg::Exception*)pData)->m_SP );
            pPacket->WriteS32( ((tma::dbg::Exception*)pData)->m_ExceptionId );
            pPacket->WriteU8( ((tma::dbg::Exception*)pData)->m_ModulesChanged );
            break;

        default:
            pPacket->WriteData( pData, SizeOfData );
            break;
    }

    m_pServicesManager->SubmitTask( this, pPacket );
}

//==============================================================================
// Mini-coredump functionality.
//==============================================================================

MiniCoreDump::MiniCoreDump( ProcessMgr* pProcessMgr )
{
    //NN_SDK_LOG( "[tma] - MiniCoreDump::MiniCoreDump\n" );
    m_pProcessMgr = pProcessMgr;
}

//==============================================================================

MiniCoreDump::~MiniCoreDump()
{
    //NN_SDK_LOG( "[tma] - MiniCoreDump::~MiniCoreDump\n" );
    m_pProcessMgr = NULL;
}

//==============================================================================

void MiniCoreDump::OnInitiate( tmipc::Packet* pPacket )
{
    //NN_SDK_LOG( "[tma] - MiniCoreDump::OnInitiate\n" );

    u64 ProcessId = 0;
    pPacket->ReadU64( ProcessId );
    void* pMem =                 s_Allocate( sizeof(DebugTask::WorkTaskInfo) );
    DebugTask::WorkTaskInfo* pTaskInfo = new (pMem) DebugTask::WorkTaskInfo( this, TMA_OP_WRITE_MINI_DUMP, m_pProcessMgr );
    pPacket->ReadData( &pTaskInfo->m_WriteDumpArgs, sizeof(pTaskInfo->m_WriteDumpArgs) );
    pTaskInfo->m_ProcessId = ProcessId;
    pTaskInfo->m_Execute = DoMiniDump;
    DebugTask::ProcessCommand( pTaskInfo );
}

//==============================================================================

void MiniCoreDump::DoMiniDump( DebugTask::WorkTaskInfo* pInfo )
{
    //NN_SDK_LOG( "[tma] - MiniCoreDump::DoMiniDump\n" );
    s32 ResultCode = tmapi::RESULT_UNKNOWN;
    tma::dbg::Command Command( TMA_OP_WRITE_MINI_DUMP, pInfo->m_ProcessId, 0, sizeof(WriteDumpCommandArgs), &pInfo->m_WriteDumpArgs );
    tma::dbg::TMAgent_message_type Type = pInfo->m_pProcessMgr->HandleCommand( &Command, ResultCode, NULL );
    //NN_SDK_LOG( "[tma] - MiniCoreDump::DoMiniDump COMPLETE:  Result = %d\n", ResultCode );
    DebugTask::SendReply( pInfo->m_pDebugTask, Type, pInfo->m_ProcessId, ResultCode, NULL, 0 );
}

//==============================================================================

void MiniCoreDump::OnRecvPacket( tmipc::Packet* pPacket )
{
    // Task complete.
    Complete();
}

//==============================================================================

void MiniCoreDump::OnSendPacket( tmipc::Packet* pPacket )
{
    ASSERT( 0 );
}

//==============================================================================
} }
//==============================================================================
