﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dbg_RegisterDefs.h"

//==============================================================================
namespace tma { namespace dbg {
//==============================================================================

//==============================================================================

s32 GetSizeULEB( u64 Value )
{
    s32 Size = 1;

    for( ; ; )
    {
        Value >>= 7;

        if( Value == 0 ) break;

        Size++;
    }
    return( Size );
}

//==============================================================================

s32 GetSizeSLEB( s64 Value )
{
    u8  B;
    s32 Size = 1;

    for( ; ; )
    {
        B       = (char)( Value & 0x7F );
        Value >>= 7;

        if(    ( (Value == 0  ) && ((B & 0x40) == 0) )
            || ( (Value == -1L) && ((B & 0x40) != 0) ) )
        {
            break;
        }
        Size++;
    }
    return( Size );
}

//==============================================================================

void EncodeULEB( u8*& pWrite, u8* pEnd, u64 Value )
{
    (void)pEnd;
    u8  B   = Value & 0x7F;
    Value >>= 7;

    while( Value )
    {
        ASSERT( pWrite < pEnd );
        *(u8*)pWrite = B | 0x80;
        pWrite      += 1;

        B       = Value & 0x7F;
        Value >>= 7;
    }
    ASSERT( pWrite < pEnd );
    *(u8*)pWrite = B;
    pWrite      += 1;
}

//==============================================================================

void EncodeSLEB( u8*& pWrite, u8* pEnd, s64 Value )
{
    (void)pEnd;
    u8  B   = Value & 0x7F;
    Value >>= 7;

    for(;;)
    {
        if(    ( (Value ==  0 ) && ((B & 0x40) == 0) )
            || ( (Value == -1L) && ((B & 0x40) != 0) ) )
        {
            break;
        }
        ASSERT( pWrite < pEnd );
        *(u8*)pWrite = B | 0x80;
        pWrite      += 1;

        B       = Value & 0x7F;
        Value >>= 7;
    }
    ASSERT( pWrite < pEnd );
    *(u8*)pWrite = B;
    pWrite      += 1;
}

//==============================================================================

void EncodeUTF8( u8*& pWrite, u8* pEnd, const char* pString )
{
    if( pString )
    {
        const u8* p = (const u8*)pString;

        while( pWrite < pEnd )
        {
            u8 B = *p++;

            ASSERT( pWrite < pEnd );
            *pWrite++ = B;

            if( B == 0 )
            {
                break;
            }
        }
    }
    else
    {
        ASSERT( pWrite < pEnd );
        *pWrite++ = 0;
    }
}

//==============================================================================

u64 DecodeULEB( u8*& pRead )
{
    u64  Result = 0;
    s32  nBits  = 0;
    u8   B      = 0x80;

    while( B & 0x80 )
    {
        B       = *pRead;
        Result  = Result | ((u64)(B & 0x7F) << nBits);
        pRead  += 1;
        nBits  += 7;
    }
    return(Result);
}

//==============================================================================

s64 DecodeSLEB( u8*& pRead )
{
    u64  Result = 0;
    s32  nBits  = 0;
    u8   B      = 0x80;

    while( B & 0x80 )
    {
        B       = *pRead;
        Result  = Result | ((u64)(B & 0x7F) << nBits);
        pRead  += 1;
        nBits  += 7;
    }

    if( B & 0x40 )
    {
        Result |= -1L << nBits;
    }
    return(Result);
}

//==============================================================================

void SkipLEB( u8*& pRead )
{
    u8 B = *pRead++;

    while( B & 0x80 )
    {
        B = *pRead++;
    }
    return;
}

//==============================================================================

s32 DecodeUTF8( u8*&pRead, u8* pBuffer, s32 BufSize  )
{
    u8* pBuf    = pBuffer;
    u8* pBufEnd = pBuf + BufSize - 1;

    while( pBuf < pBufEnd )
    {
        u8 B    = *pRead++; //TODO - Perform true UTF8 decoding, now assumes ASCII

        if( B == 0 )
        {
            break;
        }
        *pBuf++ = B;
    }
    *pBuf++ = 0;

    return( (s32)( pBuf - pBuffer ) ); // Return length including terminating zero
}

//==============================================================================

u8* SkipUTF8( u8*&pRead )
{
    u8* pStart = pRead;

    while( *pRead++ )
    {
        ; //TODO - Perform true UTF8 decoding, now assumes ASCII
    }
    return( pStart ); // Return starting address of UTF8 string
}

//==============================================================================

void EncodeCategory( u8*& pWrite, u8* pEnd, const char* pName, u32 Flags, const char* pDescription )
{
    EncodeUTF8( pWrite, pEnd, pName        );
    EncodeUTF8( pWrite, pEnd, pDescription );
    EncodeULEB( pWrite, pEnd, (u64)Flags   );
}

//==============================================================================

void EncodeRegister( u8*& pWrite, u8* pEnd, const char* pName, u32 Flags, s32 Offset, s32 FirstBit, s32 Width, s32 RegNumber, const char* pDescription )
{
    EncodeUTF8( pWrite, pEnd, pName          );
    EncodeUTF8( pWrite, pEnd, pDescription   );
    EncodeULEB( pWrite, pEnd, (u64)Flags     );
    EncodeULEB( pWrite, pEnd, (u64)Offset    );
    EncodeULEB( pWrite, pEnd, (u64)FirstBit  );
    EncodeULEB( pWrite, pEnd, (u64)Width     );
    EncodeSLEB( pWrite, pEnd, (s64)RegNumber );
}

//==============================================================================

tmapi::result GetRegisterDefinitions( bool Is64Bit, bool Is64BitAddressSpace, u32& BufferSize, void* pReturnBuffer )
{
    // Category flags can be freely defined in this file

    enum register_flags
    {
        // Format flags must be shared with TargetManager and DebugEngine

        FMT_FLAG_BYTE       = 0x00000001,
        FMT_FLAG_HALF       = 0x00000002,
        FMT_FLAG_WORD       = 0x00000004,
        FMT_FLAG_SINGLE     = 0x00000004,   // Same as Word
        FMT_FLAG_DOUBLE     = 0x00000008,
        FMT_FLAG_QUAD       = 0x00000010,

        FMT_FLAG_HEX        = 0x00000020,
        FMT_FLAG_BINARY     = 0x00000040,
        FMT_FLAG_FLOAT      = 0x00000080,
        FMT_FLAG_VECTOR     = 0x00000100,

        // Category flags add to the format flags, but need not be known to TargetManager or DebugEngine

        CAT_FLAG_GPR        = 0x00000200,
        CAT_FLAG_FPR        = 0x00000400,
        CAT_FLAG_STATUS     = 0x00000800,

        // Composite category flags, unique for each 32 or 64-bit category

        CAT_CPU             = CAT_FLAG_GPR | FMT_FLAG_DOUBLE | FMT_FLAG_HEX,
        CAT_CPU_WORD        = CAT_FLAG_GPR | FMT_FLAG_WORD   | FMT_FLAG_HEX,

        CAT_CPSR            = CAT_FLAG_GPR | CAT_FLAG_STATUS | FMT_FLAG_BINARY,
        CAT_FPSCR           = CAT_FLAG_FPR | CAT_FLAG_STATUS | FMT_FLAG_BINARY,

        CAT_FPR_QUAD        = CAT_FLAG_FPR | FMT_FLAG_QUAD   | FMT_FLAG_FLOAT,
        CAT_FPR_DOUBLE      = CAT_FLAG_FPR | FMT_FLAG_DOUBLE | FMT_FLAG_FLOAT,
        CAT_FPR_SINGLE      = CAT_FLAG_FPR | FMT_FLAG_SINGLE | FMT_FLAG_FLOAT,
        CAT_FPR_HALF        = CAT_FLAG_FPR | FMT_FLAG_HALF   | FMT_FLAG_FLOAT,
        CAT_FPR_BYTE        = CAT_FLAG_FPR | FMT_FLAG_BYTE   | FMT_FLAG_FLOAT,

        CAT_VEC_DOUBLE      = CAT_FLAG_FPR | FMT_FLAG_DOUBLE | FMT_FLAG_FLOAT | FMT_FLAG_VECTOR,
        CAT_VEC_SINGLE      = CAT_FLAG_FPR | FMT_FLAG_SINGLE | FMT_FLAG_FLOAT | FMT_FLAG_VECTOR,
        CAT_VEC_HALF        = CAT_FLAG_FPR | FMT_FLAG_HALF   | FMT_FLAG_FLOAT | FMT_FLAG_VECTOR,
        CAT_VEC_BYTE        = CAT_FLAG_FPR | FMT_FLAG_BYTE   | FMT_FLAG_FLOAT | FMT_FLAG_VECTOR,
    };

    u32  SizeRegisterDefs = 0;
    u32  BufSize = 16 * 1024;
    u8*  pBuf    = new u8[BufSize];
    u8*  pWrite  = pBuf;
    u8*  pEnd    = pBuf + BufSize;

    EncodeULEB(         pWrite, pEnd, 0 );                                  // Version Number
    EncodeULEB(         pWrite, pEnd, sizeof(tma::dbg::thread_context) );   // Size of register data in bytes
    EncodeULEB(         pWrite, pEnd, ( Is64Bit ? 8 : 4 ) );                // Size of register in bytes
    EncodeULEB(         pWrite, pEnd, ( Is64BitAddressSpace ? 8 : 4 ) );    // Size of address in bytes
    EncodeUTF8(         pWrite, pEnd, "PC" );                               // Name of Instruction Pointer register
    EncodeUTF8(         pWrite, pEnd, "SP" );                               // Name of Stack Pointer register

    if( Is64Bit )
    {
        EncodeUTF8(     pWrite, pEnd, "X30" );                               // Name of Return Address register
    }
    else
    {
        EncodeUTF8(     pWrite, pEnd, "LR" );                               // Name of Return Address register
    }
    EncodeULEB(         pWrite, pEnd, 0 );                                  // Flags

    if( Is64Bit )
    {
        EncodeCategory( pWrite, pEnd, "CPU",        CAT_CPU,            "General Purpose Register - Double" );  // Category name required by Visual Studio
        EncodeCategory( pWrite, pEnd, "CPU WORD",   CAT_CPU_WORD,       "General Purpose Register - Word" );
        EncodeCategory( pWrite, pEnd, "CPSR",       CAT_CPSR,           "Current Program Status Register" );
        EncodeCategory( pWrite, pEnd, "FPSCR",      CAT_FPSCR,          "Floating Point Status and Control Register" );

        EncodeCategory( pWrite, pEnd, "FPR QUAD",   CAT_FPR_QUAD,       "Floating Point Register - Quad" );
        EncodeCategory( pWrite, pEnd, "FPR DOUBLE", CAT_FPR_DOUBLE,     "Floating Point Register - Double" );
        EncodeCategory( pWrite, pEnd, "FPR SINGLE", CAT_FPR_SINGLE,     "Floating Point Register - Single" );
        EncodeCategory( pWrite, pEnd, "FPR HALF",   CAT_FPR_HALF,       "Floating Point Register - Half" );
        EncodeCategory( pWrite, pEnd, "FPR BYTE",   CAT_FPR_BYTE,       "Floating Point Register - Byte" );

        EncodeCategory( pWrite, pEnd, "FPR VEC.D",  CAT_VEC_DOUBLE,     "Vector Register - Double" );
        EncodeCategory( pWrite, pEnd, "FPR VEC.S",  CAT_VEC_SINGLE,     "Vector Register - Single" );
        EncodeCategory( pWrite, pEnd, "FPR VEC.H",  CAT_VEC_HALF,       "Vector Register - Half" );
        EncodeCategory( pWrite, pEnd, "FPR VEC.B",  CAT_VEC_BYTE,       "Vector Register - Byte" );
    }
    else
    {
        EncodeCategory( pWrite, pEnd, "CPU",        CAT_CPU_WORD,       "General Purpose Register" );           // Category name required by Visual Studio
        EncodeCategory( pWrite, pEnd, "CPSR",       CAT_CPSR,           "Current Program Status Register" );
        EncodeCategory( pWrite, pEnd, "FPSCR",      CAT_FPSCR,          "Floating Point Status and Control Register" );

        EncodeCategory( pWrite, pEnd, "FPR DOUBLE", CAT_FPR_DOUBLE,     "Floating Point Register - Double" );
        EncodeCategory( pWrite, pEnd, "FPR SINGLE", CAT_FPR_SINGLE,     "Floating Point Register - Single" );
    }
    EncodeULEB(         pWrite, pEnd, 0 ); // End of Categories

    if( Is64Bit )
    {
        EncodeRegister( pWrite, pEnd, "X0",         CAT_CPU,             0 * sizeof(u64),   0,  64,      0, 0 );
        EncodeRegister( pWrite, pEnd, "X1",         CAT_CPU,             1 * sizeof(u64),   0,  64,      1, 0 );
        EncodeRegister( pWrite, pEnd, "X2",         CAT_CPU,             2 * sizeof(u64),   0,  64,      2, 0 );
        EncodeRegister( pWrite, pEnd, "X3",         CAT_CPU,             3 * sizeof(u64),   0,  64,      3, 0 );
        EncodeRegister( pWrite, pEnd, "X4",         CAT_CPU,             4 * sizeof(u64),   0,  64,      4, 0 );
        EncodeRegister( pWrite, pEnd, "X5",         CAT_CPU,             5 * sizeof(u64),   0,  64,      5, 0 );
        EncodeRegister( pWrite, pEnd, "X6",         CAT_CPU,             6 * sizeof(u64),   0,  64,      6, 0 );
        EncodeRegister( pWrite, pEnd, "X7",         CAT_CPU,             7 * sizeof(u64),   0,  64,      7, 0 );
        EncodeRegister( pWrite, pEnd, "X8",         CAT_CPU,             8 * sizeof(u64),   0,  64,      8, 0 );
        EncodeRegister( pWrite, pEnd, "X9",         CAT_CPU,             9 * sizeof(u64),   0,  64,      9, 0 );
        EncodeRegister( pWrite, pEnd, "X10",        CAT_CPU,            10 * sizeof(u64),   0,  64,     10, 0 );
        EncodeRegister( pWrite, pEnd, "X11",        CAT_CPU,            11 * sizeof(u64),   0,  64,     11, 0 );
        EncodeRegister( pWrite, pEnd, "X12",        CAT_CPU,            12 * sizeof(u64),   0,  64,     12, 0 );
        EncodeRegister( pWrite, pEnd, "X13",        CAT_CPU,            13 * sizeof(u64),   0,  64,     13, 0 );
        EncodeRegister( pWrite, pEnd, "X14",        CAT_CPU,            14 * sizeof(u64),   0,  64,     14, 0 );
        EncodeRegister( pWrite, pEnd, "X15",        CAT_CPU,            15 * sizeof(u64),   0,  64,     15, 0 );
        EncodeRegister( pWrite, pEnd, "X16",        CAT_CPU,            16 * sizeof(u64),   0,  64,     16, 0 );
        EncodeRegister( pWrite, pEnd, "X17",        CAT_CPU,            17 * sizeof(u64),   0,  64,     17, 0 );
        EncodeRegister( pWrite, pEnd, "X18",        CAT_CPU,            18 * sizeof(u64),   0,  64,     18, 0 );
        EncodeRegister( pWrite, pEnd, "X19",        CAT_CPU,            19 * sizeof(u64),   0,  64,     19, 0 );
        EncodeRegister( pWrite, pEnd, "X20",        CAT_CPU,            20 * sizeof(u64),   0,  64,     20, 0 );
        EncodeRegister( pWrite, pEnd, "X21",        CAT_CPU,            21 * sizeof(u64),   0,  64,     21, 0 );
        EncodeRegister( pWrite, pEnd, "X22",        CAT_CPU,            22 * sizeof(u64),   0,  64,     22, 0 );
        EncodeRegister( pWrite, pEnd, "X23",        CAT_CPU,            23 * sizeof(u64),   0,  64,     23, 0 );
        EncodeRegister( pWrite, pEnd, "X24",        CAT_CPU,            24 * sizeof(u64),   0,  64,     24, 0 );
        EncodeRegister( pWrite, pEnd, "X25",        CAT_CPU,            25 * sizeof(u64),   0,  64,     25, 0 );
        EncodeRegister( pWrite, pEnd, "X26",        CAT_CPU,            26 * sizeof(u64),   0,  64,     26, 0 );
        EncodeRegister( pWrite, pEnd, "X27",        CAT_CPU,            27 * sizeof(u64),   0,  64,     27, 0 );
        EncodeRegister( pWrite, pEnd, "X28",        CAT_CPU,            28 * sizeof(u64),   0,  64,     28, 0 );
        EncodeRegister( pWrite, pEnd, "X29",        CAT_CPU,            29 * sizeof(u64),   0,  64,     29, 0 );
        EncodeRegister( pWrite, pEnd, "X30",        CAT_CPU,            30 * sizeof(u64),   0,  64,     30, 0 );
        EncodeRegister( pWrite, pEnd, "SP",         CAT_CPU,            31 * sizeof(u64),   0,  64,     31, 0 );

        EncodeRegister( pWrite, pEnd, "PC",         CAT_CPU,            32 * sizeof(u64),   0,  64,     32, 0 );

        EncodeRegister( pWrite, pEnd, "W0",         CAT_CPU_WORD,        0 * sizeof(u64),   0,  32,      0, 0 );
        EncodeRegister( pWrite, pEnd, "W1",         CAT_CPU_WORD,        1 * sizeof(u64),   0,  32,      1, 0 );
        EncodeRegister( pWrite, pEnd, "W2",         CAT_CPU_WORD,        2 * sizeof(u64),   0,  32,      2, 0 );
        EncodeRegister( pWrite, pEnd, "W3",         CAT_CPU_WORD,        3 * sizeof(u64),   0,  32,      3, 0 );
        EncodeRegister( pWrite, pEnd, "W4",         CAT_CPU_WORD,        4 * sizeof(u64),   0,  32,      4, 0 );
        EncodeRegister( pWrite, pEnd, "W5",         CAT_CPU_WORD,        5 * sizeof(u64),   0,  32,      5, 0 );
        EncodeRegister( pWrite, pEnd, "W6",         CAT_CPU_WORD,        6 * sizeof(u64),   0,  32,      6, 0 );
        EncodeRegister( pWrite, pEnd, "W7",         CAT_CPU_WORD,        7 * sizeof(u64),   0,  32,      7, 0 );
        EncodeRegister( pWrite, pEnd, "W8",         CAT_CPU_WORD,        8 * sizeof(u64),   0,  32,      8, 0 );
        EncodeRegister( pWrite, pEnd, "W9",         CAT_CPU_WORD,        9 * sizeof(u64),   0,  32,      9, 0 );
        EncodeRegister( pWrite, pEnd, "W10",        CAT_CPU_WORD,       10 * sizeof(u64),   0,  32,     10, 0 );
        EncodeRegister( pWrite, pEnd, "W11",        CAT_CPU_WORD,       11 * sizeof(u64),   0,  32,     11, 0 );
        EncodeRegister( pWrite, pEnd, "W12",        CAT_CPU_WORD,       12 * sizeof(u64),   0,  32,     12, 0 );
        EncodeRegister( pWrite, pEnd, "W13",        CAT_CPU_WORD,       13 * sizeof(u64),   0,  32,     13, 0 );
        EncodeRegister( pWrite, pEnd, "W14",        CAT_CPU_WORD,       14 * sizeof(u64),   0,  32,     14, 0 );
        EncodeRegister( pWrite, pEnd, "W15",        CAT_CPU_WORD,       15 * sizeof(u64),   0,  32,     15, 0 );
        EncodeRegister( pWrite, pEnd, "W16",        CAT_CPU_WORD,       16 * sizeof(u64),   0,  32,     16, 0 );
        EncodeRegister( pWrite, pEnd, "W17",        CAT_CPU_WORD,       17 * sizeof(u64),   0,  32,     17, 0 );
        EncodeRegister( pWrite, pEnd, "W18",        CAT_CPU_WORD,       18 * sizeof(u64),   0,  32,     18, 0 );
        EncodeRegister( pWrite, pEnd, "W19",        CAT_CPU_WORD,       19 * sizeof(u64),   0,  32,     19, 0 );
        EncodeRegister( pWrite, pEnd, "W20",        CAT_CPU_WORD,       20 * sizeof(u64),   0,  32,     20, 0 );
        EncodeRegister( pWrite, pEnd, "W21",        CAT_CPU_WORD,       21 * sizeof(u64),   0,  32,     21, 0 );
        EncodeRegister( pWrite, pEnd, "W22",        CAT_CPU_WORD,       22 * sizeof(u64),   0,  32,     22, 0 );
        EncodeRegister( pWrite, pEnd, "W23",        CAT_CPU_WORD,       23 * sizeof(u64),   0,  32,     23, 0 );
        EncodeRegister( pWrite, pEnd, "W24",        CAT_CPU_WORD,       24 * sizeof(u64),   0,  32,     24, 0 );
        EncodeRegister( pWrite, pEnd, "W25",        CAT_CPU_WORD,       25 * sizeof(u64),   0,  32,     25, 0 );
        EncodeRegister( pWrite, pEnd, "W26",        CAT_CPU_WORD,       26 * sizeof(u64),   0,  32,     26, 0 );
        EncodeRegister( pWrite, pEnd, "W27",        CAT_CPU_WORD,       27 * sizeof(u64),   0,  32,     27, 0 );
        EncodeRegister( pWrite, pEnd, "W28",        CAT_CPU_WORD,       28 * sizeof(u64),   0,  32,     28, 0 );
        EncodeRegister( pWrite, pEnd, "W29",        CAT_CPU_WORD,       29 * sizeof(u64),   0,  32,     29, 0 );
        EncodeRegister( pWrite, pEnd, "W30",        CAT_CPU_WORD,       30 * sizeof(u64),   0,  32,     30, 0 );
        EncodeRegister( pWrite, pEnd, "WSP",        CAT_CPU_WORD,       31 * sizeof(u64),   0,  32,     31, 0 );
    }
    else // 32-bit
    {
        EncodeRegister( pWrite, pEnd, "R0",         CAT_CPU_WORD,        0 * sizeof(u64),   0,  32,      0, 0 );
        EncodeRegister( pWrite, pEnd, "R1",         CAT_CPU_WORD,        1 * sizeof(u64),   0,  32,      1, 0 );
        EncodeRegister( pWrite, pEnd, "R2",         CAT_CPU_WORD,        2 * sizeof(u64),   0,  32,      2, 0 );
        EncodeRegister( pWrite, pEnd, "R3",         CAT_CPU_WORD,        3 * sizeof(u64),   0,  32,      3, 0 );
        EncodeRegister( pWrite, pEnd, "R4",         CAT_CPU_WORD,        4 * sizeof(u64),   0,  32,      4, 0 );
        EncodeRegister( pWrite, pEnd, "R5",         CAT_CPU_WORD,        5 * sizeof(u64),   0,  32,      5, 0 );
        EncodeRegister( pWrite, pEnd, "R6",         CAT_CPU_WORD,        6 * sizeof(u64),   0,  32,      6, 0 );
        EncodeRegister( pWrite, pEnd, "R7",         CAT_CPU_WORD,        7 * sizeof(u64),   0,  32,      7, 0 );
        EncodeRegister( pWrite, pEnd, "R8",         CAT_CPU_WORD,        8 * sizeof(u64),   0,  32,      8, 0 );
        EncodeRegister( pWrite, pEnd, "R9",         CAT_CPU_WORD,        9 * sizeof(u64),   0,  32,      9, 0 );
        EncodeRegister( pWrite, pEnd, "R10",        CAT_CPU_WORD,       10 * sizeof(u64),   0,  32,     10, 0 );
        EncodeRegister( pWrite, pEnd, "R11",        CAT_CPU_WORD,       11 * sizeof(u64),   0,  32,     11, 0 );
        EncodeRegister( pWrite, pEnd, "R12",        CAT_CPU_WORD,       12 * sizeof(u64),   0,  32,     12, 0 );

        EncodeRegister( pWrite, pEnd, "SP",         CAT_CPU_WORD,       31 * sizeof(u64),   0,  32,     13, 0 ); // LR/SP reversed in thread_context
        EncodeRegister( pWrite, pEnd, "LR",         CAT_CPU_WORD,       30 * sizeof(u64),   0,  32,     14, 0 );
        EncodeRegister( pWrite, pEnd, "PC",         CAT_CPU_WORD,       32 * sizeof(u64),   0,  32,     15, 0 );
    }

    EncodeRegister(     pWrite, pEnd, "CPSR",       CAT_CPSR,           33 * sizeof(u64),   0,  32,     33, "CPSR Summary" );

    EncodeRegister(     pWrite, pEnd, "N",          CAT_CPSR,           33 * sizeof(u64),  31,   1,     33, "Result < 0" );
    EncodeRegister(     pWrite, pEnd, "Z",          CAT_CPSR,           33 * sizeof(u64),  30,   1,     33, "Result == 0" );
    EncodeRegister(     pWrite, pEnd, "C",          CAT_CPSR,           33 * sizeof(u64),  29,   1,     33, "Carry occurred" );
    EncodeRegister(     pWrite, pEnd, "V",          CAT_CPSR,           33 * sizeof(u64),  28,   1,     33, "Signed overflow occurred" );
    EncodeRegister(     pWrite, pEnd, "Q",          CAT_CPSR,           33 * sizeof(u64),  27,   1,     33, "Saturation occurred" );
    EncodeRegister(     pWrite, pEnd, "J",          CAT_CPSR,           33 * sizeof(u64),  24,   1,     33, "Jazelle mode" );
    EncodeRegister(     pWrite, pEnd, "E",          CAT_CPSR,           33 * sizeof(u64),   9,   1,     33, "Big endian mode" );
    EncodeRegister(     pWrite, pEnd, "A",          CAT_CPSR,           33 * sizeof(u64),   8,   1,     33, "Asynchronous aborts disabled" );
    EncodeRegister(     pWrite, pEnd, "I",          CAT_CPSR,           33 * sizeof(u64),   7,   1,     33, "IRQ interrupts disabled" );
    EncodeRegister(     pWrite, pEnd, "F",          CAT_CPSR,           33 * sizeof(u64),   6,   1,     33, "FIQ interrupts disabled" );
    EncodeRegister(     pWrite, pEnd, "T",          CAT_CPSR,           33 * sizeof(u64),   5,   1,     33, "Thumb mode" );
    EncodeRegister(     pWrite, pEnd, "GE",         CAT_CPSR,           33 * sizeof(u64),  16,   4,     33, "SIMD greater than or equal flags" );
    EncodeRegister(     pWrite, pEnd, "MODE",       CAT_CPSR,           33 * sizeof(u64),   0,   5,     33, "ARM processor mode" );
    EncodeRegister(     pWrite, pEnd, "IT-7:6",     CAT_CPSR,           33 * sizeof(u64),  25,   2,     33, "Thumb If-Then bits" );
    EncodeRegister(     pWrite, pEnd, "IT-5:0",     CAT_CPSR,           33 * sizeof(u64),  10,   6,     33, "Thumb If-Then bits" );

    EncodeRegister(     pWrite, pEnd, "FPCSR",      CAT_FPSCR,          98 * sizeof(u64),   0,  32,     98, "FPCSR Summary" );

    EncodeRegister(     pWrite, pEnd, "N",          CAT_FPSCR,          98 * sizeof(u64),  31,   1,     98, "Result < 0" );
    EncodeRegister(     pWrite, pEnd, "Z",          CAT_FPSCR,          98 * sizeof(u64),  30,   1,     98, "Result == 0" );
    EncodeRegister(     pWrite, pEnd, "C",          CAT_FPSCR,          98 * sizeof(u64),  29,   1,     98, "Carry occurred" );
    EncodeRegister(     pWrite, pEnd, "V",          CAT_FPSCR,          98 * sizeof(u64),  28,   1,     98, "Signed overflow occurred" );
    EncodeRegister(     pWrite, pEnd, "QC",         CAT_FPSCR,          98 * sizeof(u64),  27,   1,     98, "Cumulative Saturation" );
    EncodeRegister(     pWrite, pEnd, "AHP",        CAT_FPSCR,          98 * sizeof(u64),  26,   1,     98, "Alternative Half-Precision" );
    EncodeRegister(     pWrite, pEnd, "DN",         CAT_FPSCR,          98 * sizeof(u64),  25,   1,     98, "Default NaN mode" );
    EncodeRegister(     pWrite, pEnd, "FZ",         CAT_FPSCR,          98 * sizeof(u64),  24,   1,     98, "Flush-to-Zero mode" );
    EncodeRegister(     pWrite, pEnd, "RMode",      CAT_FPSCR,          98 * sizeof(u64),  22,   2,     98, "Rounding Mode" );
    EncodeRegister(     pWrite, pEnd, "Stride",     CAT_FPSCR,          98 * sizeof(u64),  20,   2,     98, "Stride" );
    EncodeRegister(     pWrite, pEnd, "Len",        CAT_FPSCR,          98 * sizeof(u64),  16,   3,     98, "Length" );
    EncodeRegister(     pWrite, pEnd, "IDC",        CAT_FPSCR,          98 * sizeof(u64),   7,   1,     98, "Input Denormal cumulative exception" );
    EncodeRegister(     pWrite, pEnd, "IXC",        CAT_FPSCR,          98 * sizeof(u64),   4,   1,     98, "Inexact cumulative exception" );
    EncodeRegister(     pWrite, pEnd, "UFC",        CAT_FPSCR,          98 * sizeof(u64),   3,   1,     98, "Underflow cumulative exception" );
    EncodeRegister(     pWrite, pEnd, "OFC",        CAT_FPSCR,          98 * sizeof(u64),   2,   1,     98, "Overflow cumulative exception" );
    EncodeRegister(     pWrite, pEnd, "DZC",        CAT_FPSCR,          98 * sizeof(u64),   1,   1,     98, "Division by Zero cumulative exception" );
    EncodeRegister(     pWrite, pEnd, "IOC",        CAT_FPSCR,          98 * sizeof(u64),   0,   1,     98, "Invalid Operation cumulative exception" );

    if( Is64Bit )
    {
        EncodeRegister( pWrite, pEnd, "Q0",         CAT_FPR_QUAD,       (34 +  0) * sizeof(u64),   0, 128, (256 +  0),  0 );
        EncodeRegister( pWrite, pEnd, "Q1",         CAT_FPR_QUAD,       (34 +  2) * sizeof(u64),   0, 128, (256 +  1),  0 );
        EncodeRegister( pWrite, pEnd, "Q2",         CAT_FPR_QUAD,       (34 +  4) * sizeof(u64),   0, 128, (256 +  2),  0 );
        EncodeRegister( pWrite, pEnd, "Q3",         CAT_FPR_QUAD,       (34 +  6) * sizeof(u64),   0, 128, (256 +  3),  0 );
        EncodeRegister( pWrite, pEnd, "Q4",         CAT_FPR_QUAD,       (34 +  8) * sizeof(u64),   0, 128, (256 +  4),  0 );
        EncodeRegister( pWrite, pEnd, "Q5",         CAT_FPR_QUAD,       (34 + 10) * sizeof(u64),   0, 128, (256 +  5),  0 );
        EncodeRegister( pWrite, pEnd, "Q6",         CAT_FPR_QUAD,       (34 + 12) * sizeof(u64),   0, 128, (256 +  6),  0 );
        EncodeRegister( pWrite, pEnd, "Q7",         CAT_FPR_QUAD,       (34 + 14) * sizeof(u64),   0, 128, (256 +  7),  0 );
        EncodeRegister( pWrite, pEnd, "Q8",         CAT_FPR_QUAD,       (34 + 16) * sizeof(u64),   0, 128, (256 +  8),  0 );
        EncodeRegister( pWrite, pEnd, "Q9",         CAT_FPR_QUAD,       (34 + 18) * sizeof(u64),   0, 128, (256 +  9),  0 );
        EncodeRegister( pWrite, pEnd, "Q10",        CAT_FPR_QUAD,       (34 + 20) * sizeof(u64),   0, 128, (256 + 10),  0 );
        EncodeRegister( pWrite, pEnd, "Q11",        CAT_FPR_QUAD,       (34 + 22) * sizeof(u64),   0, 128, (256 + 11),  0 );
        EncodeRegister( pWrite, pEnd, "Q12",        CAT_FPR_QUAD,       (34 + 24) * sizeof(u64),   0, 128, (256 + 12),  0 );
        EncodeRegister( pWrite, pEnd, "Q13",        CAT_FPR_QUAD,       (34 + 26) * sizeof(u64),   0, 128, (256 + 13),  0 );
        EncodeRegister( pWrite, pEnd, "Q14",        CAT_FPR_QUAD,       (34 + 28) * sizeof(u64),   0, 128, (256 + 14),  0 );
        EncodeRegister( pWrite, pEnd, "Q15",        CAT_FPR_QUAD,       (34 + 30) * sizeof(u64),   0, 128, (256 + 15),  0 );
        EncodeRegister( pWrite, pEnd, "Q16",        CAT_FPR_QUAD,       (34 + 32) * sizeof(u64),   0, 128, (256 + 16),  0 );
        EncodeRegister( pWrite, pEnd, "Q17",        CAT_FPR_QUAD,       (34 + 34) * sizeof(u64),   0, 128, (256 + 17),  0 );
        EncodeRegister( pWrite, pEnd, "Q18",        CAT_FPR_QUAD,       (34 + 36) * sizeof(u64),   0, 128, (256 + 18),  0 );
        EncodeRegister( pWrite, pEnd, "Q19",        CAT_FPR_QUAD,       (34 + 38) * sizeof(u64),   0, 128, (256 + 19),  0 );
        EncodeRegister( pWrite, pEnd, "Q20",        CAT_FPR_QUAD,       (34 + 40) * sizeof(u64),   0, 128, (256 + 20),  0 );
        EncodeRegister( pWrite, pEnd, "Q21",        CAT_FPR_QUAD,       (34 + 42) * sizeof(u64),   0, 128, (256 + 21),  0 );
        EncodeRegister( pWrite, pEnd, "Q22",        CAT_FPR_QUAD,       (34 + 44) * sizeof(u64),   0, 128, (256 + 22),  0 );
        EncodeRegister( pWrite, pEnd, "Q23",        CAT_FPR_QUAD,       (34 + 46) * sizeof(u64),   0, 128, (256 + 23),  0 );
        EncodeRegister( pWrite, pEnd, "Q24",        CAT_FPR_QUAD,       (34 + 48) * sizeof(u64),   0, 128, (256 + 24),  0 );
        EncodeRegister( pWrite, pEnd, "Q25",        CAT_FPR_QUAD,       (34 + 50) * sizeof(u64),   0, 128, (256 + 25),  0 );
        EncodeRegister( pWrite, pEnd, "Q26",        CAT_FPR_QUAD,       (34 + 52) * sizeof(u64),   0, 128, (256 + 26),  0 );
        EncodeRegister( pWrite, pEnd, "Q27",        CAT_FPR_QUAD,       (34 + 54) * sizeof(u64),   0, 128, (256 + 27),  0 );
        EncodeRegister( pWrite, pEnd, "Q28",        CAT_FPR_QUAD,       (34 + 56) * sizeof(u64),   0, 128, (256 + 28),  0 );
        EncodeRegister( pWrite, pEnd, "Q29",        CAT_FPR_QUAD,       (34 + 58) * sizeof(u64),   0, 128, (256 + 29),  0 );
        EncodeRegister( pWrite, pEnd, "Q30",        CAT_FPR_QUAD,       (34 + 60) * sizeof(u64),   0, 128, (256 + 30),  0 );
        EncodeRegister( pWrite, pEnd, "Q31",        CAT_FPR_QUAD,       (34 + 62) * sizeof(u64),   0, 128, (256 + 31),  0 );

        EncodeRegister( pWrite, pEnd, "D0",         CAT_FPR_DOUBLE,     (34 +  0) * sizeof(u64),   0,  64, (256 +  0),  0 );
        EncodeRegister( pWrite, pEnd, "D1",         CAT_FPR_DOUBLE,     (34 +  2) * sizeof(u64),   0,  64, (256 +  1),  0 );
        EncodeRegister( pWrite, pEnd, "D2",         CAT_FPR_DOUBLE,     (34 +  4) * sizeof(u64),   0,  64, (256 +  2),  0 );
        EncodeRegister( pWrite, pEnd, "D3",         CAT_FPR_DOUBLE,     (34 +  6) * sizeof(u64),   0,  64, (256 +  3),  0 );
        EncodeRegister( pWrite, pEnd, "D4",         CAT_FPR_DOUBLE,     (34 +  8) * sizeof(u64),   0,  64, (256 +  4),  0 );
        EncodeRegister( pWrite, pEnd, "D5",         CAT_FPR_DOUBLE,     (34 + 10) * sizeof(u64),   0,  64, (256 +  5),  0 );
        EncodeRegister( pWrite, pEnd, "D6",         CAT_FPR_DOUBLE,     (34 + 12) * sizeof(u64),   0,  64, (256 +  6),  0 );
        EncodeRegister( pWrite, pEnd, "D7",         CAT_FPR_DOUBLE,     (34 + 14) * sizeof(u64),   0,  64, (256 +  7),  0 );
        EncodeRegister( pWrite, pEnd, "D8",         CAT_FPR_DOUBLE,     (34 + 16) * sizeof(u64),   0,  64, (256 +  8),  0 );
        EncodeRegister( pWrite, pEnd, "D9",         CAT_FPR_DOUBLE,     (34 + 18) * sizeof(u64),   0,  64, (256 +  9),  0 );
        EncodeRegister( pWrite, pEnd, "D10",        CAT_FPR_DOUBLE,     (34 + 20) * sizeof(u64),   0,  64, (256 + 10),  0 );
        EncodeRegister( pWrite, pEnd, "D11",        CAT_FPR_DOUBLE,     (34 + 22) * sizeof(u64),   0,  64, (256 + 11),  0 );
        EncodeRegister( pWrite, pEnd, "D12",        CAT_FPR_DOUBLE,     (34 + 24) * sizeof(u64),   0,  64, (256 + 12),  0 );
        EncodeRegister( pWrite, pEnd, "D13",        CAT_FPR_DOUBLE,     (34 + 26) * sizeof(u64),   0,  64, (256 + 13),  0 );
        EncodeRegister( pWrite, pEnd, "D14",        CAT_FPR_DOUBLE,     (34 + 28) * sizeof(u64),   0,  64, (256 + 14),  0 );
        EncodeRegister( pWrite, pEnd, "D15",        CAT_FPR_DOUBLE,     (34 + 30) * sizeof(u64),   0,  64, (256 + 15),  0 );
        EncodeRegister( pWrite, pEnd, "D16",        CAT_FPR_DOUBLE,     (34 + 32) * sizeof(u64),   0,  64, (256 + 16),  0 );
        EncodeRegister( pWrite, pEnd, "D17",        CAT_FPR_DOUBLE,     (34 + 34) * sizeof(u64),   0,  64, (256 + 17),  0 );
        EncodeRegister( pWrite, pEnd, "D18",        CAT_FPR_DOUBLE,     (34 + 36) * sizeof(u64),   0,  64, (256 + 18),  0 );
        EncodeRegister( pWrite, pEnd, "D19",        CAT_FPR_DOUBLE,     (34 + 38) * sizeof(u64),   0,  64, (256 + 19),  0 );
        EncodeRegister( pWrite, pEnd, "D20",        CAT_FPR_DOUBLE,     (34 + 40) * sizeof(u64),   0,  64, (256 + 20),  0 );
        EncodeRegister( pWrite, pEnd, "D21",        CAT_FPR_DOUBLE,     (34 + 42) * sizeof(u64),   0,  64, (256 + 21),  0 );
        EncodeRegister( pWrite, pEnd, "D22",        CAT_FPR_DOUBLE,     (34 + 44) * sizeof(u64),   0,  64, (256 + 22),  0 );
        EncodeRegister( pWrite, pEnd, "D23",        CAT_FPR_DOUBLE,     (34 + 46) * sizeof(u64),   0,  64, (256 + 23),  0 );
        EncodeRegister( pWrite, pEnd, "D24",        CAT_FPR_DOUBLE,     (34 + 48) * sizeof(u64),   0,  64, (256 + 24),  0 );
        EncodeRegister( pWrite, pEnd, "D25",        CAT_FPR_DOUBLE,     (34 + 50) * sizeof(u64),   0,  64, (256 + 25),  0 );
        EncodeRegister( pWrite, pEnd, "D26",        CAT_FPR_DOUBLE,     (34 + 52) * sizeof(u64),   0,  64, (256 + 26),  0 );
        EncodeRegister( pWrite, pEnd, "D27",        CAT_FPR_DOUBLE,     (34 + 54) * sizeof(u64),   0,  64, (256 + 27),  0 );
        EncodeRegister( pWrite, pEnd, "D28",        CAT_FPR_DOUBLE,     (34 + 56) * sizeof(u64),   0,  64, (256 + 28),  0 );
        EncodeRegister( pWrite, pEnd, "D29",        CAT_FPR_DOUBLE,     (34 + 58) * sizeof(u64),   0,  64, (256 + 29),  0 );
        EncodeRegister( pWrite, pEnd, "D30",        CAT_FPR_DOUBLE,     (34 + 60) * sizeof(u64),   0,  64, (256 + 30),  0 );
        EncodeRegister( pWrite, pEnd, "D31",        CAT_FPR_DOUBLE,     (34 + 62) * sizeof(u64),   0,  64, (256 + 31),  0 );

        EncodeRegister( pWrite, pEnd, "S0",         CAT_FPR_SINGLE,     (34 +  0) * sizeof(u64),   0,  32, (256 +  0),  0 );
        EncodeRegister( pWrite, pEnd, "S1",         CAT_FPR_SINGLE,     (34 +  2) * sizeof(u64),   0,  32, (256 +  1),  0 );
        EncodeRegister( pWrite, pEnd, "S2",         CAT_FPR_SINGLE,     (34 +  4) * sizeof(u64),   0,  32, (256 +  2),  0 );
        EncodeRegister( pWrite, pEnd, "S3",         CAT_FPR_SINGLE,     (34 +  6) * sizeof(u64),   0,  32, (256 +  3),  0 );
        EncodeRegister( pWrite, pEnd, "S4",         CAT_FPR_SINGLE,     (34 +  8) * sizeof(u64),   0,  32, (256 +  4),  0 );
        EncodeRegister( pWrite, pEnd, "S5",         CAT_FPR_SINGLE,     (34 + 10) * sizeof(u64),   0,  32, (256 +  5),  0 );
        EncodeRegister( pWrite, pEnd, "S6",         CAT_FPR_SINGLE,     (34 + 12) * sizeof(u64),   0,  32, (256 +  6),  0 );
        EncodeRegister( pWrite, pEnd, "S7",         CAT_FPR_SINGLE,     (34 + 14) * sizeof(u64),   0,  32, (256 +  7),  0 );
        EncodeRegister( pWrite, pEnd, "S8",         CAT_FPR_SINGLE,     (34 + 16) * sizeof(u64),   0,  32, (256 +  8),  0 );
        EncodeRegister( pWrite, pEnd, "S9",         CAT_FPR_SINGLE,     (34 + 18) * sizeof(u64),   0,  32, (256 +  9),  0 );
        EncodeRegister( pWrite, pEnd, "S10",        CAT_FPR_SINGLE,     (34 + 20) * sizeof(u64),   0,  32, (256 + 10),  0 );
        EncodeRegister( pWrite, pEnd, "S11",        CAT_FPR_SINGLE,     (34 + 22) * sizeof(u64),   0,  32, (256 + 11),  0 );
        EncodeRegister( pWrite, pEnd, "S12",        CAT_FPR_SINGLE,     (34 + 24) * sizeof(u64),   0,  32, (256 + 12),  0 );
        EncodeRegister( pWrite, pEnd, "S13",        CAT_FPR_SINGLE,     (34 + 26) * sizeof(u64),   0,  32, (256 + 13),  0 );
        EncodeRegister( pWrite, pEnd, "S14",        CAT_FPR_SINGLE,     (34 + 28) * sizeof(u64),   0,  32, (256 + 14),  0 );
        EncodeRegister( pWrite, pEnd, "S15",        CAT_FPR_SINGLE,     (34 + 30) * sizeof(u64),   0,  32, (256 + 15),  0 );
        EncodeRegister( pWrite, pEnd, "S16",        CAT_FPR_SINGLE,     (34 + 32) * sizeof(u64),   0,  32, (256 + 16),  0 );
        EncodeRegister( pWrite, pEnd, "S17",        CAT_FPR_SINGLE,     (34 + 34) * sizeof(u64),   0,  32, (256 + 17),  0 );
        EncodeRegister( pWrite, pEnd, "S18",        CAT_FPR_SINGLE,     (34 + 36) * sizeof(u64),   0,  32, (256 + 18),  0 );
        EncodeRegister( pWrite, pEnd, "S19",        CAT_FPR_SINGLE,     (34 + 38) * sizeof(u64),   0,  32, (256 + 19),  0 );
        EncodeRegister( pWrite, pEnd, "S20",        CAT_FPR_SINGLE,     (34 + 40) * sizeof(u64),   0,  32, (256 + 20),  0 );
        EncodeRegister( pWrite, pEnd, "S21",        CAT_FPR_SINGLE,     (34 + 42) * sizeof(u64),   0,  32, (256 + 21),  0 );
        EncodeRegister( pWrite, pEnd, "S22",        CAT_FPR_SINGLE,     (34 + 44) * sizeof(u64),   0,  32, (256 + 22),  0 );
        EncodeRegister( pWrite, pEnd, "S23",        CAT_FPR_SINGLE,     (34 + 46) * sizeof(u64),   0,  32, (256 + 23),  0 );
        EncodeRegister( pWrite, pEnd, "S24",        CAT_FPR_SINGLE,     (34 + 48) * sizeof(u64),   0,  32, (256 + 24),  0 );
        EncodeRegister( pWrite, pEnd, "S25",        CAT_FPR_SINGLE,     (34 + 50) * sizeof(u64),   0,  32, (256 + 25),  0 );
        EncodeRegister( pWrite, pEnd, "S26",        CAT_FPR_SINGLE,     (34 + 52) * sizeof(u64),   0,  32, (256 + 26),  0 );
        EncodeRegister( pWrite, pEnd, "S27",        CAT_FPR_SINGLE,     (34 + 54) * sizeof(u64),   0,  32, (256 + 27),  0 );
        EncodeRegister( pWrite, pEnd, "S28",        CAT_FPR_SINGLE,     (34 + 56) * sizeof(u64),   0,  32, (256 + 28),  0 );
        EncodeRegister( pWrite, pEnd, "S29",        CAT_FPR_SINGLE,     (34 + 58) * sizeof(u64),   0,  32, (256 + 29),  0 );
        EncodeRegister( pWrite, pEnd, "S30",        CAT_FPR_SINGLE,     (34 + 60) * sizeof(u64),   0,  32, (256 + 30),  0 );
        EncodeRegister( pWrite, pEnd, "S31",        CAT_FPR_SINGLE,     (34 + 62) * sizeof(u64),   0,  32, (256 + 31),  0 );

        EncodeRegister( pWrite, pEnd, "H0",         CAT_FPR_HALF,       (34 +  0) * sizeof(u64),   0,  16, (256 +  0),  0 );
        EncodeRegister( pWrite, pEnd, "H1",         CAT_FPR_HALF,       (34 +  2) * sizeof(u64),   0,  16, (256 +  1),  0 );
        EncodeRegister( pWrite, pEnd, "H2",         CAT_FPR_HALF,       (34 +  4) * sizeof(u64),   0,  16, (256 +  2),  0 );
        EncodeRegister( pWrite, pEnd, "H3",         CAT_FPR_HALF,       (34 +  6) * sizeof(u64),   0,  16, (256 +  3),  0 );
        EncodeRegister( pWrite, pEnd, "H4",         CAT_FPR_HALF,       (34 +  8) * sizeof(u64),   0,  16, (256 +  4),  0 );
        EncodeRegister( pWrite, pEnd, "H5",         CAT_FPR_HALF,       (34 + 10) * sizeof(u64),   0,  16, (256 +  5),  0 );
        EncodeRegister( pWrite, pEnd, "H6",         CAT_FPR_HALF,       (34 + 12) * sizeof(u64),   0,  16, (256 +  6),  0 );
        EncodeRegister( pWrite, pEnd, "H7",         CAT_FPR_HALF,       (34 + 14) * sizeof(u64),   0,  16, (256 +  7),  0 );
        EncodeRegister( pWrite, pEnd, "H8",         CAT_FPR_HALF,       (34 + 16) * sizeof(u64),   0,  16, (256 +  8),  0 );
        EncodeRegister( pWrite, pEnd, "H9",         CAT_FPR_HALF,       (34 + 18) * sizeof(u64),   0,  16, (256 +  9),  0 );
        EncodeRegister( pWrite, pEnd, "H10",        CAT_FPR_HALF,       (34 + 20) * sizeof(u64),   0,  16, (256 + 10),  0 );
        EncodeRegister( pWrite, pEnd, "H11",        CAT_FPR_HALF,       (34 + 22) * sizeof(u64),   0,  16, (256 + 11),  0 );
        EncodeRegister( pWrite, pEnd, "H12",        CAT_FPR_HALF,       (34 + 24) * sizeof(u64),   0,  16, (256 + 12),  0 );
        EncodeRegister( pWrite, pEnd, "H13",        CAT_FPR_HALF,       (34 + 26) * sizeof(u64),   0,  16, (256 + 13),  0 );
        EncodeRegister( pWrite, pEnd, "H14",        CAT_FPR_HALF,       (34 + 28) * sizeof(u64),   0,  16, (256 + 14),  0 );
        EncodeRegister( pWrite, pEnd, "H15",        CAT_FPR_HALF,       (34 + 30) * sizeof(u64),   0,  16, (256 + 15),  0 );
        EncodeRegister( pWrite, pEnd, "H16",        CAT_FPR_HALF,       (34 + 32) * sizeof(u64),   0,  16, (256 + 16),  0 );
        EncodeRegister( pWrite, pEnd, "H17",        CAT_FPR_HALF,       (34 + 34) * sizeof(u64),   0,  16, (256 + 17),  0 );
        EncodeRegister( pWrite, pEnd, "H18",        CAT_FPR_HALF,       (34 + 36) * sizeof(u64),   0,  16, (256 + 18),  0 );
        EncodeRegister( pWrite, pEnd, "H19",        CAT_FPR_HALF,       (34 + 38) * sizeof(u64),   0,  16, (256 + 19),  0 );
        EncodeRegister( pWrite, pEnd, "H20",        CAT_FPR_HALF,       (34 + 40) * sizeof(u64),   0,  16, (256 + 20),  0 );
        EncodeRegister( pWrite, pEnd, "H21",        CAT_FPR_HALF,       (34 + 42) * sizeof(u64),   0,  16, (256 + 21),  0 );
        EncodeRegister( pWrite, pEnd, "H22",        CAT_FPR_HALF,       (34 + 44) * sizeof(u64),   0,  16, (256 + 22),  0 );
        EncodeRegister( pWrite, pEnd, "H23",        CAT_FPR_HALF,       (34 + 46) * sizeof(u64),   0,  16, (256 + 23),  0 );
        EncodeRegister( pWrite, pEnd, "H24",        CAT_FPR_HALF,       (34 + 48) * sizeof(u64),   0,  16, (256 + 24),  0 );
        EncodeRegister( pWrite, pEnd, "H25",        CAT_FPR_HALF,       (34 + 50) * sizeof(u64),   0,  16, (256 + 25),  0 );
        EncodeRegister( pWrite, pEnd, "H26",        CAT_FPR_HALF,       (34 + 52) * sizeof(u64),   0,  16, (256 + 26),  0 );
        EncodeRegister( pWrite, pEnd, "H27",        CAT_FPR_HALF,       (34 + 54) * sizeof(u64),   0,  16, (256 + 27),  0 );
        EncodeRegister( pWrite, pEnd, "H28",        CAT_FPR_HALF,       (34 + 56) * sizeof(u64),   0,  16, (256 + 28),  0 );
        EncodeRegister( pWrite, pEnd, "H29",        CAT_FPR_HALF,       (34 + 58) * sizeof(u64),   0,  16, (256 + 29),  0 );
        EncodeRegister( pWrite, pEnd, "H30",        CAT_FPR_HALF,       (34 + 60) * sizeof(u64),   0,  16, (256 + 30),  0 );
        EncodeRegister( pWrite, pEnd, "H31",        CAT_FPR_HALF,       (34 + 62) * sizeof(u64),   0,  16, (256 + 31),  0 );

        EncodeRegister( pWrite, pEnd, "B0",         CAT_FPR_BYTE,       (34 +  0) * sizeof(u64),   0,   8, (256 +  0),  0 );
        EncodeRegister( pWrite, pEnd, "B1",         CAT_FPR_BYTE,       (34 +  2) * sizeof(u64),   0,   8, (256 +  1),  0 );
        EncodeRegister( pWrite, pEnd, "B2",         CAT_FPR_BYTE,       (34 +  4) * sizeof(u64),   0,   8, (256 +  2),  0 );
        EncodeRegister( pWrite, pEnd, "B3",         CAT_FPR_BYTE,       (34 +  6) * sizeof(u64),   0,   8, (256 +  3),  0 );
        EncodeRegister( pWrite, pEnd, "B4",         CAT_FPR_BYTE,       (34 +  8) * sizeof(u64),   0,   8, (256 +  4),  0 );
        EncodeRegister( pWrite, pEnd, "B5",         CAT_FPR_BYTE,       (34 + 10) * sizeof(u64),   0,   8, (256 +  5),  0 );
        EncodeRegister( pWrite, pEnd, "B6",         CAT_FPR_BYTE,       (34 + 12) * sizeof(u64),   0,   8, (256 +  6),  0 );
        EncodeRegister( pWrite, pEnd, "B7",         CAT_FPR_BYTE,       (34 + 14) * sizeof(u64),   0,   8, (256 +  7),  0 );
        EncodeRegister( pWrite, pEnd, "B8",         CAT_FPR_BYTE,       (34 + 16) * sizeof(u64),   0,   8, (256 +  8),  0 );
        EncodeRegister( pWrite, pEnd, "B9",         CAT_FPR_BYTE,       (34 + 18) * sizeof(u64),   0,   8, (256 +  9),  0 );
        EncodeRegister( pWrite, pEnd, "B10",        CAT_FPR_BYTE,       (34 + 20) * sizeof(u64),   0,   8, (256 + 10),  0 );
        EncodeRegister( pWrite, pEnd, "B11",        CAT_FPR_BYTE,       (34 + 22) * sizeof(u64),   0,   8, (256 + 11),  0 );
        EncodeRegister( pWrite, pEnd, "B12",        CAT_FPR_BYTE,       (34 + 24) * sizeof(u64),   0,   8, (256 + 12),  0 );
        EncodeRegister( pWrite, pEnd, "B13",        CAT_FPR_BYTE,       (34 + 26) * sizeof(u64),   0,   8, (256 + 13),  0 );
        EncodeRegister( pWrite, pEnd, "B14",        CAT_FPR_BYTE,       (34 + 28) * sizeof(u64),   0,   8, (256 + 14),  0 );
        EncodeRegister( pWrite, pEnd, "B15",        CAT_FPR_BYTE,       (34 + 30) * sizeof(u64),   0,   8, (256 + 15),  0 );
        EncodeRegister( pWrite, pEnd, "B16",        CAT_FPR_BYTE,       (34 + 32) * sizeof(u64),   0,   8, (256 + 16),  0 );
        EncodeRegister( pWrite, pEnd, "B17",        CAT_FPR_BYTE,       (34 + 34) * sizeof(u64),   0,   8, (256 + 17),  0 );
        EncodeRegister( pWrite, pEnd, "B18",        CAT_FPR_BYTE,       (34 + 36) * sizeof(u64),   0,   8, (256 + 18),  0 );
        EncodeRegister( pWrite, pEnd, "B19",        CAT_FPR_BYTE,       (34 + 38) * sizeof(u64),   0,   8, (256 + 19),  0 );
        EncodeRegister( pWrite, pEnd, "B20",        CAT_FPR_BYTE,       (34 + 40) * sizeof(u64),   0,   8, (256 + 20),  0 );
        EncodeRegister( pWrite, pEnd, "B21",        CAT_FPR_BYTE,       (34 + 42) * sizeof(u64),   0,   8, (256 + 21),  0 );
        EncodeRegister( pWrite, pEnd, "B22",        CAT_FPR_BYTE,       (34 + 44) * sizeof(u64),   0,   8, (256 + 22),  0 );
        EncodeRegister( pWrite, pEnd, "B23",        CAT_FPR_BYTE,       (34 + 46) * sizeof(u64),   0,   8, (256 + 23),  0 );
        EncodeRegister( pWrite, pEnd, "B24",        CAT_FPR_BYTE,       (34 + 48) * sizeof(u64),   0,   8, (256 + 24),  0 );
        EncodeRegister( pWrite, pEnd, "B25",        CAT_FPR_BYTE,       (34 + 50) * sizeof(u64),   0,   8, (256 + 25),  0 );
        EncodeRegister( pWrite, pEnd, "B26",        CAT_FPR_BYTE,       (34 + 52) * sizeof(u64),   0,   8, (256 + 26),  0 );
        EncodeRegister( pWrite, pEnd, "B27",        CAT_FPR_BYTE,       (34 + 54) * sizeof(u64),   0,   8, (256 + 27),  0 );
        EncodeRegister( pWrite, pEnd, "B28",        CAT_FPR_BYTE,       (34 + 56) * sizeof(u64),   0,   8, (256 + 28),  0 );
        EncodeRegister( pWrite, pEnd, "B29",        CAT_FPR_BYTE,       (34 + 58) * sizeof(u64),   0,   8, (256 + 29),  0 );
        EncodeRegister( pWrite, pEnd, "B30",        CAT_FPR_BYTE,       (34 + 60) * sizeof(u64),   0,   8, (256 + 30),  0 );
        EncodeRegister( pWrite, pEnd, "B31",        CAT_FPR_BYTE,       (34 + 62) * sizeof(u64),   0,   8, (256 + 31),  0 );

        EncodeRegister( pWrite, pEnd, "V0.D",       CAT_VEC_DOUBLE,     (34 +  0) * sizeof(u64),   0, 128, (256 +  0),  0 );
        EncodeRegister( pWrite, pEnd, "V1.D",       CAT_VEC_DOUBLE,     (34 +  2) * sizeof(u64),   0, 128, (256 +  1),  0 );
        EncodeRegister( pWrite, pEnd, "V2.D",       CAT_VEC_DOUBLE,     (34 +  4) * sizeof(u64),   0, 128, (256 +  2),  0 );
        EncodeRegister( pWrite, pEnd, "V3.D",       CAT_VEC_DOUBLE,     (34 +  6) * sizeof(u64),   0, 128, (256 +  3),  0 );
        EncodeRegister( pWrite, pEnd, "V4.D",       CAT_VEC_DOUBLE,     (34 +  8) * sizeof(u64),   0, 128, (256 +  4),  0 );
        EncodeRegister( pWrite, pEnd, "V5.D",       CAT_VEC_DOUBLE,     (34 + 10) * sizeof(u64),   0, 128, (256 +  5),  0 );
        EncodeRegister( pWrite, pEnd, "V6.D",       CAT_VEC_DOUBLE,     (34 + 12) * sizeof(u64),   0, 128, (256 +  6),  0 );
        EncodeRegister( pWrite, pEnd, "V7.D",       CAT_VEC_DOUBLE,     (34 + 14) * sizeof(u64),   0, 128, (256 +  7),  0 );
        EncodeRegister( pWrite, pEnd, "V8.D",       CAT_VEC_DOUBLE,     (34 + 16) * sizeof(u64),   0, 128, (256 +  8),  0 );
        EncodeRegister( pWrite, pEnd, "V9.D",       CAT_VEC_DOUBLE,     (34 + 18) * sizeof(u64),   0, 128, (256 +  9),  0 );
        EncodeRegister( pWrite, pEnd, "V10.D",      CAT_VEC_DOUBLE,     (34 + 20) * sizeof(u64),   0, 128, (256 + 10),  0 );
        EncodeRegister( pWrite, pEnd, "V11.D",      CAT_VEC_DOUBLE,     (34 + 22) * sizeof(u64),   0, 128, (256 + 11),  0 );
        EncodeRegister( pWrite, pEnd, "V12.D",      CAT_VEC_DOUBLE,     (34 + 24) * sizeof(u64),   0, 128, (256 + 12),  0 );
        EncodeRegister( pWrite, pEnd, "V13.D",      CAT_VEC_DOUBLE,     (34 + 26) * sizeof(u64),   0, 128, (256 + 13),  0 );
        EncodeRegister( pWrite, pEnd, "V14.D",      CAT_VEC_DOUBLE,     (34 + 28) * sizeof(u64),   0, 128, (256 + 14),  0 );
        EncodeRegister( pWrite, pEnd, "V15.D",      CAT_VEC_DOUBLE,     (34 + 30) * sizeof(u64),   0, 128, (256 + 15),  0 );
        EncodeRegister( pWrite, pEnd, "V16.D",      CAT_VEC_DOUBLE,     (34 + 32) * sizeof(u64),   0, 128, (256 + 16),  0 );
        EncodeRegister( pWrite, pEnd, "V17.D",      CAT_VEC_DOUBLE,     (34 + 34) * sizeof(u64),   0, 128, (256 + 17),  0 );
        EncodeRegister( pWrite, pEnd, "V18.D",      CAT_VEC_DOUBLE,     (34 + 36) * sizeof(u64),   0, 128, (256 + 18),  0 );
        EncodeRegister( pWrite, pEnd, "V19.D",      CAT_VEC_DOUBLE,     (34 + 38) * sizeof(u64),   0, 128, (256 + 19),  0 );
        EncodeRegister( pWrite, pEnd, "V20.D",      CAT_VEC_DOUBLE,     (34 + 40) * sizeof(u64),   0, 128, (256 + 20),  0 );
        EncodeRegister( pWrite, pEnd, "V21.D",      CAT_VEC_DOUBLE,     (34 + 42) * sizeof(u64),   0, 128, (256 + 21),  0 );
        EncodeRegister( pWrite, pEnd, "V22.D",      CAT_VEC_DOUBLE,     (34 + 44) * sizeof(u64),   0, 128, (256 + 22),  0 );
        EncodeRegister( pWrite, pEnd, "V23.D",      CAT_VEC_DOUBLE,     (34 + 46) * sizeof(u64),   0, 128, (256 + 23),  0 );
        EncodeRegister( pWrite, pEnd, "V24.D",      CAT_VEC_DOUBLE,     (34 + 48) * sizeof(u64),   0, 128, (256 + 24),  0 );
        EncodeRegister( pWrite, pEnd, "V25.D",      CAT_VEC_DOUBLE,     (34 + 50) * sizeof(u64),   0, 128, (256 + 25),  0 );
        EncodeRegister( pWrite, pEnd, "V26.D",      CAT_VEC_DOUBLE,     (34 + 52) * sizeof(u64),   0, 128, (256 + 26),  0 );
        EncodeRegister( pWrite, pEnd, "V27.D",      CAT_VEC_DOUBLE,     (34 + 54) * sizeof(u64),   0, 128, (256 + 27),  0 );
        EncodeRegister( pWrite, pEnd, "V28.D",      CAT_VEC_DOUBLE,     (34 + 56) * sizeof(u64),   0, 128, (256 + 28),  0 );
        EncodeRegister( pWrite, pEnd, "V29.D",      CAT_VEC_DOUBLE,     (34 + 58) * sizeof(u64),   0, 128, (256 + 29),  0 );
        EncodeRegister( pWrite, pEnd, "V30.D",      CAT_VEC_DOUBLE,     (34 + 60) * sizeof(u64),   0, 128, (256 + 30),  0 );
        EncodeRegister( pWrite, pEnd, "V31.D",      CAT_VEC_DOUBLE,     (34 + 62) * sizeof(u64),   0, 128, (256 + 31),  0 );

        EncodeRegister( pWrite, pEnd, "V0.S",       CAT_VEC_SINGLE,     (34 +  0) * sizeof(u64),   0, 128, (256 +  0),  0 );
        EncodeRegister( pWrite, pEnd, "V1.S",       CAT_VEC_SINGLE,     (34 +  2) * sizeof(u64),   0, 128, (256 +  1),  0 );
        EncodeRegister( pWrite, pEnd, "V2.S",       CAT_VEC_SINGLE,     (34 +  4) * sizeof(u64),   0, 128, (256 +  2),  0 );
        EncodeRegister( pWrite, pEnd, "V3.S",       CAT_VEC_SINGLE,     (34 +  6) * sizeof(u64),   0, 128, (256 +  3),  0 );
        EncodeRegister( pWrite, pEnd, "V4.S",       CAT_VEC_SINGLE,     (34 +  8) * sizeof(u64),   0, 128, (256 +  4),  0 );
        EncodeRegister( pWrite, pEnd, "V5.S",       CAT_VEC_SINGLE,     (34 + 10) * sizeof(u64),   0, 128, (256 +  5),  0 );
        EncodeRegister( pWrite, pEnd, "V6.S",       CAT_VEC_SINGLE,     (34 + 12) * sizeof(u64),   0, 128, (256 +  6),  0 );
        EncodeRegister( pWrite, pEnd, "V7.S",       CAT_VEC_SINGLE,     (34 + 14) * sizeof(u64),   0, 128, (256 +  7),  0 );
        EncodeRegister( pWrite, pEnd, "V8.S",       CAT_VEC_SINGLE,     (34 + 16) * sizeof(u64),   0, 128, (256 +  8),  0 );
        EncodeRegister( pWrite, pEnd, "V9.S",       CAT_VEC_SINGLE,     (34 + 18) * sizeof(u64),   0, 128, (256 +  9),  0 );
        EncodeRegister( pWrite, pEnd, "V10.S",      CAT_VEC_SINGLE,     (34 + 20) * sizeof(u64),   0, 128, (256 + 10),  0 );
        EncodeRegister( pWrite, pEnd, "V11.S",      CAT_VEC_SINGLE,     (34 + 22) * sizeof(u64),   0, 128, (256 + 11),  0 );
        EncodeRegister( pWrite, pEnd, "V12.S",      CAT_VEC_SINGLE,     (34 + 24) * sizeof(u64),   0, 128, (256 + 12),  0 );
        EncodeRegister( pWrite, pEnd, "V13.S",      CAT_VEC_SINGLE,     (34 + 26) * sizeof(u64),   0, 128, (256 + 13),  0 );
        EncodeRegister( pWrite, pEnd, "V14.S",      CAT_VEC_SINGLE,     (34 + 28) * sizeof(u64),   0, 128, (256 + 14),  0 );
        EncodeRegister( pWrite, pEnd, "V15.S",      CAT_VEC_SINGLE,     (34 + 30) * sizeof(u64),   0, 128, (256 + 15),  0 );
        EncodeRegister( pWrite, pEnd, "V16.S",      CAT_VEC_SINGLE,     (34 + 32) * sizeof(u64),   0, 128, (256 + 16),  0 );
        EncodeRegister( pWrite, pEnd, "V17.S",      CAT_VEC_SINGLE,     (34 + 34) * sizeof(u64),   0, 128, (256 + 17),  0 );
        EncodeRegister( pWrite, pEnd, "V18.S",      CAT_VEC_SINGLE,     (34 + 36) * sizeof(u64),   0, 128, (256 + 18),  0 );
        EncodeRegister( pWrite, pEnd, "V19.S",      CAT_VEC_SINGLE,     (34 + 38) * sizeof(u64),   0, 128, (256 + 19),  0 );
        EncodeRegister( pWrite, pEnd, "V20.S",      CAT_VEC_SINGLE,     (34 + 40) * sizeof(u64),   0, 128, (256 + 20),  0 );
        EncodeRegister( pWrite, pEnd, "V21.S",      CAT_VEC_SINGLE,     (34 + 42) * sizeof(u64),   0, 128, (256 + 21),  0 );
        EncodeRegister( pWrite, pEnd, "V22.S",      CAT_VEC_SINGLE,     (34 + 44) * sizeof(u64),   0, 128, (256 + 22),  0 );
        EncodeRegister( pWrite, pEnd, "V23.S",      CAT_VEC_SINGLE,     (34 + 46) * sizeof(u64),   0, 128, (256 + 23),  0 );
        EncodeRegister( pWrite, pEnd, "V24.S",      CAT_VEC_SINGLE,     (34 + 48) * sizeof(u64),   0, 128, (256 + 24),  0 );
        EncodeRegister( pWrite, pEnd, "V25.S",      CAT_VEC_SINGLE,     (34 + 50) * sizeof(u64),   0, 128, (256 + 25),  0 );
        EncodeRegister( pWrite, pEnd, "V26.S",      CAT_VEC_SINGLE,     (34 + 52) * sizeof(u64),   0, 128, (256 + 26),  0 );
        EncodeRegister( pWrite, pEnd, "V27.S",      CAT_VEC_SINGLE,     (34 + 54) * sizeof(u64),   0, 128, (256 + 27),  0 );
        EncodeRegister( pWrite, pEnd, "V28.S",      CAT_VEC_SINGLE,     (34 + 56) * sizeof(u64),   0, 128, (256 + 28),  0 );
        EncodeRegister( pWrite, pEnd, "V29.S",      CAT_VEC_SINGLE,     (34 + 58) * sizeof(u64),   0, 128, (256 + 29),  0 );
        EncodeRegister( pWrite, pEnd, "V30.S",      CAT_VEC_SINGLE,     (34 + 60) * sizeof(u64),   0, 128, (256 + 30),  0 );
        EncodeRegister( pWrite, pEnd, "V31.S",      CAT_VEC_SINGLE,     (34 + 62) * sizeof(u64),   0, 128, (256 + 31),  0 );

        EncodeRegister( pWrite, pEnd, "V0.H",       CAT_VEC_HALF,       (34 +  0) * sizeof(u64),   0, 128, (256 +  0),  0 );
        EncodeRegister( pWrite, pEnd, "V1.H",       CAT_VEC_HALF,       (34 +  2) * sizeof(u64),   0, 128, (256 +  1),  0 );
        EncodeRegister( pWrite, pEnd, "V2.H",       CAT_VEC_HALF,       (34 +  4) * sizeof(u64),   0, 128, (256 +  2),  0 );
        EncodeRegister( pWrite, pEnd, "V3.H",       CAT_VEC_HALF,       (34 +  6) * sizeof(u64),   0, 128, (256 +  3),  0 );
        EncodeRegister( pWrite, pEnd, "V4.H",       CAT_VEC_HALF,       (34 +  8) * sizeof(u64),   0, 128, (256 +  4),  0 );
        EncodeRegister( pWrite, pEnd, "V5.H",       CAT_VEC_HALF,       (34 + 10) * sizeof(u64),   0, 128, (256 +  5),  0 );
        EncodeRegister( pWrite, pEnd, "V6.H",       CAT_VEC_HALF,       (34 + 12) * sizeof(u64),   0, 128, (256 +  6),  0 );
        EncodeRegister( pWrite, pEnd, "V7.H",       CAT_VEC_HALF,       (34 + 14) * sizeof(u64),   0, 128, (256 +  7),  0 );
        EncodeRegister( pWrite, pEnd, "V8.H",       CAT_VEC_HALF,       (34 + 16) * sizeof(u64),   0, 128, (256 +  8),  0 );
        EncodeRegister( pWrite, pEnd, "V9.H",       CAT_VEC_HALF,       (34 + 18) * sizeof(u64),   0, 128, (256 +  9),  0 );
        EncodeRegister( pWrite, pEnd, "V10.H",      CAT_VEC_HALF,       (34 + 20) * sizeof(u64),   0, 128, (256 + 10),  0 );
        EncodeRegister( pWrite, pEnd, "V11.H",      CAT_VEC_HALF,       (34 + 22) * sizeof(u64),   0, 128, (256 + 11),  0 );
        EncodeRegister( pWrite, pEnd, "V12.H",      CAT_VEC_HALF,       (34 + 24) * sizeof(u64),   0, 128, (256 + 12),  0 );
        EncodeRegister( pWrite, pEnd, "V13.H",      CAT_VEC_HALF,       (34 + 26) * sizeof(u64),   0, 128, (256 + 13),  0 );
        EncodeRegister( pWrite, pEnd, "V14.H",      CAT_VEC_HALF,       (34 + 28) * sizeof(u64),   0, 128, (256 + 14),  0 );
        EncodeRegister( pWrite, pEnd, "V15.H",      CAT_VEC_HALF,       (34 + 30) * sizeof(u64),   0, 128, (256 + 15),  0 );
        EncodeRegister( pWrite, pEnd, "V16.H",      CAT_VEC_HALF,       (34 + 32) * sizeof(u64),   0, 128, (256 + 16),  0 );
        EncodeRegister( pWrite, pEnd, "V17.H",      CAT_VEC_HALF,       (34 + 34) * sizeof(u64),   0, 128, (256 + 17),  0 );
        EncodeRegister( pWrite, pEnd, "V18.H",      CAT_VEC_HALF,       (34 + 36) * sizeof(u64),   0, 128, (256 + 18),  0 );
        EncodeRegister( pWrite, pEnd, "V19.H",      CAT_VEC_HALF,       (34 + 38) * sizeof(u64),   0, 128, (256 + 19),  0 );
        EncodeRegister( pWrite, pEnd, "V20.H",      CAT_VEC_HALF,       (34 + 40) * sizeof(u64),   0, 128, (256 + 20),  0 );
        EncodeRegister( pWrite, pEnd, "V21.H",      CAT_VEC_HALF,       (34 + 42) * sizeof(u64),   0, 128, (256 + 21),  0 );
        EncodeRegister( pWrite, pEnd, "V22.H",      CAT_VEC_HALF,       (34 + 44) * sizeof(u64),   0, 128, (256 + 22),  0 );
        EncodeRegister( pWrite, pEnd, "V23.H",      CAT_VEC_HALF,       (34 + 46) * sizeof(u64),   0, 128, (256 + 23),  0 );
        EncodeRegister( pWrite, pEnd, "V24.H",      CAT_VEC_HALF,       (34 + 48) * sizeof(u64),   0, 128, (256 + 24),  0 );
        EncodeRegister( pWrite, pEnd, "V25.H",      CAT_VEC_HALF,       (34 + 50) * sizeof(u64),   0, 128, (256 + 25),  0 );
        EncodeRegister( pWrite, pEnd, "V26.H",      CAT_VEC_HALF,       (34 + 52) * sizeof(u64),   0, 128, (256 + 26),  0 );
        EncodeRegister( pWrite, pEnd, "V27.H",      CAT_VEC_HALF,       (34 + 54) * sizeof(u64),   0, 128, (256 + 27),  0 );
        EncodeRegister( pWrite, pEnd, "V28.H",      CAT_VEC_HALF,       (34 + 56) * sizeof(u64),   0, 128, (256 + 28),  0 );
        EncodeRegister( pWrite, pEnd, "V29.H",      CAT_VEC_HALF,       (34 + 58) * sizeof(u64),   0, 128, (256 + 29),  0 );
        EncodeRegister( pWrite, pEnd, "V30.H",      CAT_VEC_HALF,       (34 + 60) * sizeof(u64),   0, 128, (256 + 30),  0 );
        EncodeRegister( pWrite, pEnd, "V31.H",      CAT_VEC_HALF,       (34 + 62) * sizeof(u64),   0, 128, (256 + 31),  0 );

        EncodeRegister( pWrite, pEnd, "V0.B",       CAT_VEC_BYTE,       (34 +  0) * sizeof(u64),   0, 128, (256 +  0),  0 );
        EncodeRegister( pWrite, pEnd, "V1.B",       CAT_VEC_BYTE,       (34 +  2) * sizeof(u64),   0, 128, (256 +  1),  0 );
        EncodeRegister( pWrite, pEnd, "V2.B",       CAT_VEC_BYTE,       (34 +  4) * sizeof(u64),   0, 128, (256 +  2),  0 );
        EncodeRegister( pWrite, pEnd, "V3.B",       CAT_VEC_BYTE,       (34 +  6) * sizeof(u64),   0, 128, (256 +  3),  0 );
        EncodeRegister( pWrite, pEnd, "V4.B",       CAT_VEC_BYTE,       (34 +  8) * sizeof(u64),   0, 128, (256 +  4),  0 );
        EncodeRegister( pWrite, pEnd, "V5.B",       CAT_VEC_BYTE,       (34 + 10) * sizeof(u64),   0, 128, (256 +  5),  0 );
        EncodeRegister( pWrite, pEnd, "V6.B",       CAT_VEC_BYTE,       (34 + 12) * sizeof(u64),   0, 128, (256 +  6),  0 );
        EncodeRegister( pWrite, pEnd, "V7.B",       CAT_VEC_BYTE,       (34 + 14) * sizeof(u64),   0, 128, (256 +  7),  0 );
        EncodeRegister( pWrite, pEnd, "V8.B",       CAT_VEC_BYTE,       (34 + 16) * sizeof(u64),   0, 128, (256 +  8),  0 );
        EncodeRegister( pWrite, pEnd, "V9.B",       CAT_VEC_BYTE,       (34 + 18) * sizeof(u64),   0, 128, (256 +  9),  0 );
        EncodeRegister( pWrite, pEnd, "V10.B",      CAT_VEC_BYTE,       (34 + 20) * sizeof(u64),   0, 128, (256 + 10),  0 );
        EncodeRegister( pWrite, pEnd, "V11.B",      CAT_VEC_BYTE,       (34 + 22) * sizeof(u64),   0, 128, (256 + 11),  0 );
        EncodeRegister( pWrite, pEnd, "V12.B",      CAT_VEC_BYTE,       (34 + 24) * sizeof(u64),   0, 128, (256 + 12),  0 );
        EncodeRegister( pWrite, pEnd, "V13.B",      CAT_VEC_BYTE,       (34 + 26) * sizeof(u64),   0, 128, (256 + 13),  0 );
        EncodeRegister( pWrite, pEnd, "V14.B",      CAT_VEC_BYTE,       (34 + 28) * sizeof(u64),   0, 128, (256 + 14),  0 );
        EncodeRegister( pWrite, pEnd, "V15.B",      CAT_VEC_BYTE,       (34 + 30) * sizeof(u64),   0, 128, (256 + 15),  0 );
        EncodeRegister( pWrite, pEnd, "V16.B",      CAT_VEC_BYTE,       (34 + 32) * sizeof(u64),   0, 128, (256 + 16),  0 );
        EncodeRegister( pWrite, pEnd, "V17.B",      CAT_VEC_BYTE,       (34 + 34) * sizeof(u64),   0, 128, (256 + 17),  0 );
        EncodeRegister( pWrite, pEnd, "V18.B",      CAT_VEC_BYTE,       (34 + 36) * sizeof(u64),   0, 128, (256 + 18),  0 );
        EncodeRegister( pWrite, pEnd, "V19.B",      CAT_VEC_BYTE,       (34 + 38) * sizeof(u64),   0, 128, (256 + 19),  0 );
        EncodeRegister( pWrite, pEnd, "V20.B",      CAT_VEC_BYTE,       (34 + 40) * sizeof(u64),   0, 128, (256 + 20),  0 );
        EncodeRegister( pWrite, pEnd, "V21.B",      CAT_VEC_BYTE,       (34 + 42) * sizeof(u64),   0, 128, (256 + 21),  0 );
        EncodeRegister( pWrite, pEnd, "V22.B",      CAT_VEC_BYTE,       (34 + 44) * sizeof(u64),   0, 128, (256 + 22),  0 );
        EncodeRegister( pWrite, pEnd, "V23.B",      CAT_VEC_BYTE,       (34 + 46) * sizeof(u64),   0, 128, (256 + 23),  0 );
        EncodeRegister( pWrite, pEnd, "V24.B",      CAT_VEC_BYTE,       (34 + 48) * sizeof(u64),   0, 128, (256 + 24),  0 );
        EncodeRegister( pWrite, pEnd, "V25.B",      CAT_VEC_BYTE,       (34 + 50) * sizeof(u64),   0, 128, (256 + 25),  0 );
        EncodeRegister( pWrite, pEnd, "V26.B",      CAT_VEC_BYTE,       (34 + 52) * sizeof(u64),   0, 128, (256 + 26),  0 );
        EncodeRegister( pWrite, pEnd, "V27.B",      CAT_VEC_BYTE,       (34 + 54) * sizeof(u64),   0, 128, (256 + 27),  0 );
        EncodeRegister( pWrite, pEnd, "V28.B",      CAT_VEC_BYTE,       (34 + 56) * sizeof(u64),   0, 128, (256 + 28),  0 );
        EncodeRegister( pWrite, pEnd, "V29.B",      CAT_VEC_BYTE,       (34 + 58) * sizeof(u64),   0, 128, (256 + 29),  0 );
        EncodeRegister( pWrite, pEnd, "V30.B",      CAT_VEC_BYTE,       (34 + 60) * sizeof(u64),   0, 128, (256 + 30),  0 );
        EncodeRegister( pWrite, pEnd, "V31.B",      CAT_VEC_BYTE,       (34 + 62) * sizeof(u64),   0, 128, (256 + 31),  0 );
    }
    else // 32-bit
    {
        EncodeRegister( pWrite, pEnd, "D0",         CAT_FPR_DOUBLE,     (34 +  0) * sizeof(u64),   0,  64, (256 +  0),  0 );
        EncodeRegister( pWrite, pEnd, "D1",         CAT_FPR_DOUBLE,     (34 +  1) * sizeof(u64),   0,  64, (256 +  1),  0 );
        EncodeRegister( pWrite, pEnd, "D2",         CAT_FPR_DOUBLE,     (34 +  2) * sizeof(u64),   0,  64, (256 +  2),  0 );
        EncodeRegister( pWrite, pEnd, "D3",         CAT_FPR_DOUBLE,     (34 +  3) * sizeof(u64),   0,  64, (256 +  3),  0 );
        EncodeRegister( pWrite, pEnd, "D4",         CAT_FPR_DOUBLE,     (34 +  4) * sizeof(u64),   0,  64, (256 +  4),  0 );
        EncodeRegister( pWrite, pEnd, "D5",         CAT_FPR_DOUBLE,     (34 +  5) * sizeof(u64),   0,  64, (256 +  5),  0 );
        EncodeRegister( pWrite, pEnd, "D6",         CAT_FPR_DOUBLE,     (34 +  6) * sizeof(u64),   0,  64, (256 +  6),  0 );
        EncodeRegister( pWrite, pEnd, "D7",         CAT_FPR_DOUBLE,     (34 +  7) * sizeof(u64),   0,  64, (256 +  7),  0 );
        EncodeRegister( pWrite, pEnd, "D8",         CAT_FPR_DOUBLE,     (34 +  8) * sizeof(u64),   0,  64, (256 +  8),  0 );
        EncodeRegister( pWrite, pEnd, "D9",         CAT_FPR_DOUBLE,     (34 +  9) * sizeof(u64),   0,  64, (256 +  9),  0 );
        EncodeRegister( pWrite, pEnd, "D10",        CAT_FPR_DOUBLE,     (34 + 10) * sizeof(u64),   0,  64, (256 + 10),  0 );
        EncodeRegister( pWrite, pEnd, "D11",        CAT_FPR_DOUBLE,     (34 + 11) * sizeof(u64),   0,  64, (256 + 11),  0 );
        EncodeRegister( pWrite, pEnd, "D12",        CAT_FPR_DOUBLE,     (34 + 12) * sizeof(u64),   0,  64, (256 + 12),  0 );
        EncodeRegister( pWrite, pEnd, "D13",        CAT_FPR_DOUBLE,     (34 + 13) * sizeof(u64),   0,  64, (256 + 13),  0 );
        EncodeRegister( pWrite, pEnd, "D14",        CAT_FPR_DOUBLE,     (34 + 14) * sizeof(u64),   0,  64, (256 + 14),  0 );
        EncodeRegister( pWrite, pEnd, "D15",        CAT_FPR_DOUBLE,     (34 + 15) * sizeof(u64),   0,  64, (256 + 15),  0 );
        EncodeRegister( pWrite, pEnd, "D16",        CAT_FPR_DOUBLE,     (34 + 16) * sizeof(u64),   0,  64, (256 + 16),  0 );
        EncodeRegister( pWrite, pEnd, "D17",        CAT_FPR_DOUBLE,     (34 + 17) * sizeof(u64),   0,  64, (256 + 17),  0 );
        EncodeRegister( pWrite, pEnd, "D18",        CAT_FPR_DOUBLE,     (34 + 18) * sizeof(u64),   0,  64, (256 + 18),  0 );
        EncodeRegister( pWrite, pEnd, "D19",        CAT_FPR_DOUBLE,     (34 + 19) * sizeof(u64),   0,  64, (256 + 19),  0 );
        EncodeRegister( pWrite, pEnd, "D20",        CAT_FPR_DOUBLE,     (34 + 20) * sizeof(u64),   0,  64, (256 + 20),  0 );
        EncodeRegister( pWrite, pEnd, "D21",        CAT_FPR_DOUBLE,     (34 + 21) * sizeof(u64),   0,  64, (256 + 21),  0 );
        EncodeRegister( pWrite, pEnd, "D22",        CAT_FPR_DOUBLE,     (34 + 22) * sizeof(u64),   0,  64, (256 + 22),  0 );
        EncodeRegister( pWrite, pEnd, "D23",        CAT_FPR_DOUBLE,     (34 + 23) * sizeof(u64),   0,  64, (256 + 23),  0 );
        EncodeRegister( pWrite, pEnd, "D24",        CAT_FPR_DOUBLE,     (34 + 24) * sizeof(u64),   0,  64, (256 + 24),  0 );
        EncodeRegister( pWrite, pEnd, "D25",        CAT_FPR_DOUBLE,     (34 + 25) * sizeof(u64),   0,  64, (256 + 25),  0 );
        EncodeRegister( pWrite, pEnd, "D26",        CAT_FPR_DOUBLE,     (34 + 26) * sizeof(u64),   0,  64, (256 + 26),  0 );
        EncodeRegister( pWrite, pEnd, "D27",        CAT_FPR_DOUBLE,     (34 + 27) * sizeof(u64),   0,  64, (256 + 27),  0 );
        EncodeRegister( pWrite, pEnd, "D28",        CAT_FPR_DOUBLE,     (34 + 28) * sizeof(u64),   0,  64, (256 + 28),  0 );
        EncodeRegister( pWrite, pEnd, "D29",        CAT_FPR_DOUBLE,     (34 + 29) * sizeof(u64),   0,  64, (256 + 29),  0 );
        EncodeRegister( pWrite, pEnd, "D30",        CAT_FPR_DOUBLE,     (34 + 30) * sizeof(u64),   0,  64, (256 + 30),  0 );
        EncodeRegister( pWrite, pEnd, "D31",        CAT_FPR_DOUBLE,     (34 + 31) * sizeof(u64),   0,  64, (256 + 31),  0 );

        EncodeRegister( pWrite, pEnd, "S0",         CAT_FPR_SINGLE,     (68 +  0) * sizeof(u32),   0,  32, (256 +  0),  0 );
        EncodeRegister( pWrite, pEnd, "S1",         CAT_FPR_SINGLE,     (68 +  1) * sizeof(u32),   0,  32, (256 +  0),  0 );
        EncodeRegister( pWrite, pEnd, "S2",         CAT_FPR_SINGLE,     (68 +  2) * sizeof(u32),   0,  32, (256 +  1),  0 );
        EncodeRegister( pWrite, pEnd, "S3",         CAT_FPR_SINGLE,     (68 +  3) * sizeof(u32),   0,  32, (256 +  1),  0 );
        EncodeRegister( pWrite, pEnd, "S4",         CAT_FPR_SINGLE,     (68 +  4) * sizeof(u32),   0,  32, (256 +  2),  0 );
        EncodeRegister( pWrite, pEnd, "S5",         CAT_FPR_SINGLE,     (68 +  5) * sizeof(u32),   0,  32, (256 +  2),  0 );
        EncodeRegister( pWrite, pEnd, "S6",         CAT_FPR_SINGLE,     (68 +  6) * sizeof(u32),   0,  32, (256 +  3),  0 );
        EncodeRegister( pWrite, pEnd, "S7",         CAT_FPR_SINGLE,     (68 +  7) * sizeof(u32),   0,  32, (256 +  3),  0 );
        EncodeRegister( pWrite, pEnd, "S8",         CAT_FPR_SINGLE,     (68 +  8) * sizeof(u32),   0,  32, (256 +  4),  0 );
        EncodeRegister( pWrite, pEnd, "S9",         CAT_FPR_SINGLE,     (68 +  9) * sizeof(u32),   0,  32, (256 +  4),  0 );
        EncodeRegister( pWrite, pEnd, "S10",        CAT_FPR_SINGLE,     (68 + 10) * sizeof(u32),   0,  32, (256 +  5),  0 );
        EncodeRegister( pWrite, pEnd, "S11",        CAT_FPR_SINGLE,     (68 + 11) * sizeof(u32),   0,  32, (256 +  5),  0 );
        EncodeRegister( pWrite, pEnd, "S12",        CAT_FPR_SINGLE,     (68 + 12) * sizeof(u32),   0,  32, (256 +  6),  0 );
        EncodeRegister( pWrite, pEnd, "S13",        CAT_FPR_SINGLE,     (68 + 13) * sizeof(u32),   0,  32, (256 +  6),  0 );
        EncodeRegister( pWrite, pEnd, "S14",        CAT_FPR_SINGLE,     (68 + 14) * sizeof(u32),   0,  32, (256 +  7),  0 );
        EncodeRegister( pWrite, pEnd, "S15",        CAT_FPR_SINGLE,     (68 + 15) * sizeof(u32),   0,  32, (256 +  7),  0 );
        EncodeRegister( pWrite, pEnd, "S16",        CAT_FPR_SINGLE,     (68 + 16) * sizeof(u32),   0,  32, (256 +  8),  0 );
        EncodeRegister( pWrite, pEnd, "S17",        CAT_FPR_SINGLE,     (68 + 17) * sizeof(u32),   0,  32, (256 +  8),  0 );
        EncodeRegister( pWrite, pEnd, "S18",        CAT_FPR_SINGLE,     (68 + 18) * sizeof(u32),   0,  32, (256 +  9),  0 );
        EncodeRegister( pWrite, pEnd, "S19",        CAT_FPR_SINGLE,     (68 + 19) * sizeof(u32),   0,  32, (256 +  9),  0 );
        EncodeRegister( pWrite, pEnd, "S20",        CAT_FPR_SINGLE,     (68 + 20) * sizeof(u32),   0,  32, (256 + 10),  0 );
        EncodeRegister( pWrite, pEnd, "S21",        CAT_FPR_SINGLE,     (68 + 21) * sizeof(u32),   0,  32, (256 + 10),  0 );
        EncodeRegister( pWrite, pEnd, "S22",        CAT_FPR_SINGLE,     (68 + 22) * sizeof(u32),   0,  32, (256 + 11),  0 );
        EncodeRegister( pWrite, pEnd, "S23",        CAT_FPR_SINGLE,     (68 + 23) * sizeof(u32),   0,  32, (256 + 11),  0 );
        EncodeRegister( pWrite, pEnd, "S24",        CAT_FPR_SINGLE,     (68 + 24) * sizeof(u32),   0,  32, (256 + 12),  0 );
        EncodeRegister( pWrite, pEnd, "S25",        CAT_FPR_SINGLE,     (68 + 25) * sizeof(u32),   0,  32, (256 + 12),  0 );
        EncodeRegister( pWrite, pEnd, "S26",        CAT_FPR_SINGLE,     (68 + 26) * sizeof(u32),   0,  32, (256 + 13),  0 );
        EncodeRegister( pWrite, pEnd, "S27",        CAT_FPR_SINGLE,     (68 + 27) * sizeof(u32),   0,  32, (256 + 13),  0 );
        EncodeRegister( pWrite, pEnd, "S28",        CAT_FPR_SINGLE,     (68 + 28) * sizeof(u32),   0,  32, (256 + 14),  0 );
        EncodeRegister( pWrite, pEnd, "S29",        CAT_FPR_SINGLE,     (68 + 29) * sizeof(u32),   0,  32, (256 + 14),  0 );
        EncodeRegister( pWrite, pEnd, "S30",        CAT_FPR_SINGLE,     (68 + 30) * sizeof(u32),   0,  32, (256 + 15),  0 );
        EncodeRegister( pWrite, pEnd, "S31",        CAT_FPR_SINGLE,     (68 + 31) * sizeof(u32),   0,  32, (256 + 15),  0 );
    }

    EncodeULEB(     pWrite, pEnd, 0 ); // End of Registers

    SizeRegisterDefs      = (u32)(pWrite - pBuf);
    tmapi::result Result = tmapi::result::RESULT_OK;

    if( pReturnBuffer == NULL )
    {
        Result = tmapi::result::RESULT_INVALID_PARAMETER;
    }
    else if( BufferSize < SizeRegisterDefs )
    {
        Result = tmapi::result::RESULT_BUFFER_TOO_SMALL;
        *((u32*)pReturnBuffer) = SizeRegisterDefs;
    }
    else
    {
        memcpy( pReturnBuffer, pBuf, SizeRegisterDefs );
    }
    BufferSize = SizeRegisterDefs;
    delete [] pBuf;
    return( Result );
} // NOLINT(readability/fn_size)

//==============================================================================

//==============================================================================
}} // namespace
//==============================================================================
