﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\stdafx.h"
#include <nn.h>
#include <nn/os.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs_Base.h>
#include <nn/svc/svc_Types.h>
#include <cstdlib>
#include <cinttypes>
#include "dbg_CoreDump.h"
#include "..\..\..\TargetTools\SnapShotDumper\coredump\coredump_Data.h"
#include "../dbghlp/dbghlp.h"

//==============================================================================
namespace tma { namespace dbg { namespace CoreDump {
//==============================================================================

static void compileDetails( Process* pProcess, bool QuickDump, coredump::coredump_compiled_data* pDetails )
{
    pDetails->Init( pProcess->GetProcessId(), pProcess->GetCurrentThreadId(), pProcess->Is64Bit(), pProcess->Is64BitAddressSpace(), pProcess->GetProgramId(), QuickDump );
    pProcess->GetName( pDetails->m_ProcessName, sizeof(pDetails->m_ProcessName) );
    pProcess->GetArgs( pDetails->m_Args, sizeof(pDetails->m_Args) );
    pDetails->SetException( pProcess->GetLastExceptionId(), pProcess->GetLastExceptionAddress() );

    s32 NumberOfThreads = pProcess->GetNumberOfThreads();
    for( int Index = 0; Index < NumberOfThreads; Index +=1 )
    {
        ThreadDefinition* pThread;
        if( pProcess->GetThreadDetails( Index, &pThread ) == tmapi::RESULT_OK )
        {
            pDetails->AddThread( pThread->GetThreadId(), pThread->GetName(), pThread->GetStackSize(), pThread->GetStackAddr() );
        }
    }
    s32 NumberOfModules = 0;
    pProcess->GetNumberOfModules( NumberOfModules );
    for( int Index = 0; Index < NumberOfModules; Index +=1 )
    {
        ModuleDefinition Module;
        if( pProcess->GetModuleInfo( Index, &Module ) == tmapi::RESULT_OK )
        {
            pDetails->AddModule( Module.GetName(), Module.GetID(), Module.GetAddress(), Module.GetSize() );
        }
    }
}

//==============================================================================

nn::Result Generate( Process* pProcess, char* pOutputName, bool QuickDump )
{
    TMA_PRINTF( "CoreDump::Generate %s\n", pOutputName );
    //NN_SDK_LOG( "CoreDump::Generate %s\n", pOutputName );

    coredump::coredump_compiled_data Details;
    compileDetails( pProcess, QuickDump, &Details );

    nn::Result result = dbghlp::CreateCoreDump( pOutputName, &Details, pProcess->GetHandle() );

    //NN_SDK_LOG( "dbghlp::CreateCoreDump returned %d\n", result.GetInnerValueForDebug() );
    return result;
}


//==============================================================================

static nn::os::MutexType sCoredumpMutex = NN_OS_MUTEX_INITIALIZER( false );

nn::Result Initiate( Process* pProcess, char* pOutputName, bool QuickDump )
{
    u64 SessionId = 0;
    coredump::coredump_compiled_data Details;

    compileDetails( pProcess, QuickDump, &Details );

    nn::os::LockMutex( &sCoredumpMutex );

    nn::Result result = dbghlp::InitiateCoreDump( pOutputName, &Details, pProcess->GetHandle(), &SessionId );

    if( result.IsSuccess() )
    {
        pProcess->SetCoredumpSessionId( SessionId );
    }

    nn::os::UnlockMutex( &sCoredumpMutex );

    return result;
}

//==============================================================================

nn::Result Continue( Process* pProcess, s32* pPercentDone )
{
    nn::os::LockMutex( &sCoredumpMutex );

    nn::Result result = dbghlp::ContinueCoreDump( pProcess->GetCoredumpSessionId(), pPercentDone );

    nn::os::UnlockMutex( &sCoredumpMutex );

    return result;
}

//==============================================================================

nn::Result AddTTY( Process* pProcess, char* pData, u32 SizeOfData )
{
    nn::os::LockMutex( &sCoredumpMutex );

    nn::Result result = dbghlp::AddTTYToCoreDump( pData, SizeOfData, pProcess->GetCoredumpSessionId() );

    nn::os::UnlockMutex( &sCoredumpMutex );

    return result;
}

//==============================================================================

nn::Result AddImage( Process* pProcess, char* pData, u32 SizeOfData )
{
    nn::os::LockMutex( &sCoredumpMutex );

    nn::Result result = dbghlp::AddImageToCoreDump( pData, SizeOfData, pProcess->GetCoredumpSessionId() );

    nn::os::UnlockMutex( &sCoredumpMutex );

    return result;
}

//==============================================================================

nn::Result Close( Process* pProcess )
{
    nn::os::LockMutex( &sCoredumpMutex );

    nn::Result result = dbghlp::CloseCoreDump( pProcess->GetCoredumpSessionId() );

    pProcess->SetCoredumpSessionId( 0 );

    nn::os::UnlockMutex( &sCoredumpMutex );

    return result;
}

//==============================================================================
}}} // namespace
//==============================================================================
