﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "benchmark_report_service.h"
#if ENABLE_BENCHMARK_TESTING
#include "benchmark_report_task.h"
#include "benchmark_service.h"
#include "../tmipc/tmipc_node_tics_plug.h"
#include "../tmipc/tmipc_node_usb_interface.h"

//==============================================================================
namespace tma { namespace benchmark {
//==============================================================================

s64 BenchmarkReportService::m_PingTimesStart[NumberOfPings];
s64 BenchmarkReportService::m_PingTimesFinish[NumberOfPings];
s32 BenchmarkReportService::m_PingTimesIndex;
u32 BenchmarkReportService::m_PacketTaskId;
u32 BenchmarkReportService::m_BenchmarkServiceId;


BenchmarkReportService::BenchmarkReportService()
{
//NN_SDK_LOG( "[BenchmarkReportService::BenchmarkReportService( %p )]\n", this );
    m_ServiceId = HashString( "BenchmarkReportService" );
//NN_SDK_LOG( "[BenchmarkReportService::BenchmarkReportService( %p )] - HashString('BenchmarkReportService') == 0x%08x\n", this, GetId() );
}

//==============================================================================

BenchmarkReportService::~BenchmarkReportService()
{
//NN_SDK_LOG( "[BenchmarkReportService::~BenchmarkReportService( %p )]\n", this );
}

//==============================================================================

void BenchmarkReportService::Init()
{
//NN_SDK_LOG( "[BenchmarkReportService::Init( %p )]\n", this );
    Create();
}

//==============================================================================

void BenchmarkReportService::Kill()
{
//NN_SDK_LOG( "[BenchmarkReportService::Kill( %p )]\n", this );
    Destroy();
}

//==============================================================================

tmipc::Task* BenchmarkReportService::OnNewTask( tmipc::Packet* pPacket )
{
//NN_SDK_LOG( "[BenchmarkReportService::OnNewTask( 0x%p )] - Packet: 0x%p\n", this, pPacket );
    void* pMem = s_Allocate( sizeof( BenchmarkReportTask ) );
    BenchmarkReportTask* pTask = new (pMem) BenchmarkReportTask;
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( pPacket->GetTaskId() );
    pTask->OnInitiate( pPacket );
    return( pTask );
}

//==============================================================================
// Called when a packet was received.
void BenchmarkReportService::OnPacketReceivedCallback( const tmipc::Packet& Packet )
{
    // Only process "BenchmarkService" packets.
    if( Packet.GetServiceId() == m_BenchmarkServiceId )
    {
//float Temp = { 0 };
//char Buffer[1000] = "(";
        // Make sure the array does not overflow.
        if( m_PingTimesIndex < NumberOfPings )
        {
//TMA_STRCAT(Buffer, 100, "A");
            // On the tmagent, a received packet is the timing's *start*.
            m_PacketTaskId = Packet.GetTaskId();
            m_PingTimesStart[m_PingTimesIndex] = nn::os::GetSystemTick().GetInt64Value();
//Temp = (float)(m_PingTimesStart[m_PingTimesIndex]);
        }
//TMA_STRCAT(Buffer, 100, ")");
//NN_SDK_LOG( "%d) R: %f '%s'\n", m_PingTimesIndex, Temp, Buffer);
    }
}

//==============================================================================
// Called when a packet is about to be sent.
void BenchmarkReportService::OnPacketSendCallback( const tmipc::Packet& Packet )
{
    // Only process "BenchmarkService" packets.
    if( Packet.GetServiceId() == m_BenchmarkServiceId )
    {
//float Temp = { 0 };
//char Buffer[1000] = "(";
        // Make sure the array does not overflow.
        if( m_PingTimesIndex < NumberOfPings )
        {
//TMA_STRCAT(Buffer, 100, "A");
            // Make sure the Packet's Task Id matches.
            if( m_PacketTaskId == Packet.GetTaskId() )
            {
//TMA_STRCAT(Buffer, 100, "B");
                // On the tmagent, a send packet is the timing's *finish*.
                m_PingTimesFinish[m_PingTimesIndex] = nn::os::GetSystemTick().GetInt64Value();
//Temp = (float)(m_PingTimesFinish[m_PingTimesIndex]);
                // A completed cycle.  Increment to the next index.
                m_PingTimesIndex++;
            }
        }
//TMA_STRCAT(Buffer, 100, ")");
//NN_SDK_LOG( "%d) S: %f '%s'\n", m_PingTimesIndex - 1, Temp, Buffer);
    }
}

//==============================================================================
// Start recording Packet received/send timing information.
void BenchmarkReportService::Start()
{
    // Record the BenchmarkService Id.
    // (NOT the BenchmarkREPORTService Id.)
    m_BenchmarkServiceId = GetBenchmarkService().GetId();

    for( s32 Index = 0; Index < NumberOfPings; Index++ )
    {
        m_PingTimesStart[Index] = 0;
        m_PingTimesFinish[Index] = 0;
    }
    m_PingTimesIndex = 0;
    m_PacketTaskId = -1;

    // Set the callback functions for:
    // 1. Host Bridge.
    tmipc::TICSPlug::SetOnPacketReceivedCallback( OnPacketReceivedCallback );
    tmipc::TICSPlug::SetOnPacketSendCallback( OnPacketSendCallback );
    // 2. USB
    tmipc::USBInterface::SetOnPacketReceivedCallback( OnPacketReceivedCallback );
    tmipc::USBInterface::SetOnPacketSendCallback( OnPacketSendCallback );

    // Set these back to defaults *again* - to [help] ensure a packet does not
    // get received between the Receive/Send.
    m_PingTimesIndex = 0;
    m_PacketTaskId = -1;
}

//==============================================================================
// Stop recording Packet received/send timing information.
void BenchmarkReportService::Stop()
{
    // Clear the callback functions.

    // 1. Host Bridge.
    tmipc::TICSPlug::SetOnPacketReceivedCallback( nullptr );
    tmipc::TICSPlug::SetOnPacketSendCallback( nullptr );
    // 2. USB
    tmipc::USBInterface::SetOnPacketReceivedCallback( nullptr );
    tmipc::USBInterface::SetOnPacketSendCallback( nullptr );
}

//==============================================================================
// Generate a report and add it to the given packet.  This is only safe to
// call outside of the Start()/Stop() period.
void BenchmarkReportService::GenerateReport( tmipc::Packet& Packet )
{
//NN_SDK_LOG( "\n--------------------------\n" );
//NN_SDK_LOG( "\nRaw Report recorded: %d values.\n", m_PingTimesIndex );
//
    // Gather some statistical numbers: total time, highest and shortest time.
    s32 NumberOfValidTimes  = { 0 };
    {
        s32 InternalNumberOfValidTimes  = { 0 };
        for( s32 Index = 0; Index < m_PingTimesIndex; Index++ )
        {
//NN_SDK_LOG( "%d) Start: %f; Finish: %f\n", Index + 1, (float)m_PingTimesStart[Index], (float)m_PingTimesFinish[Index] );
            if( (m_PingTimesStart[Index] != 0) && (m_PingTimesFinish[Index] != 0) )
            {
                // Increment total time.
                InternalNumberOfValidTimes++;
            }
        }

        if( InternalNumberOfValidTimes > 0 )
        {
            NumberOfValidTimes  = InternalNumberOfValidTimes;
        }
    }

    // Write the report to the packet:
    // - "Number of report timings" (s64)*.
    // - Report timings             (s64). See "Number of report timings (above)" (ticks).
    // - Tick frequency             (s64).

    // Compute the size of the table to return.
    s64 ActualElementCount          = { m_PingTimesIndex };
    const s64 SystemTickFrequency   = { nn::os::GetSystemTickFrequency() };
    const s32 NonArrayLength  =
    {
        static_cast<s32>
        (
                sizeof(SystemTickFrequency)
            +   sizeof(ActualElementCount)
        )
    };
    const s64 RequiredTimingArrayBytes = { static_cast<s64>(m_PingTimesIndex * sizeof(*m_PingTimesStart)) };
    const s64 BytesLeft = { static_cast<s32>(tmipc::MAX_PACKET_DATA) - (Packet.GetSendLen() + NonArrayLength) };
    if( BytesLeft < RequiredTimingArrayBytes )
    {
        ActualElementCount = BytesLeft / static_cast<s64>(sizeof(*m_PingTimesStart));
    }
    Packet.WriteS64( ActualElementCount );
    for( s32 Index = 0; Index < ActualElementCount; Index++ )
    {
        s64 Ticks = { -1 };
        if( (m_PingTimesFinish[Index] != 0) && (m_PingTimesStart[Index] != 0) )
        {
            Ticks = m_PingTimesFinish[Index] - m_PingTimesStart[Index];
        }
        Packet.WriteS64( Ticks );
    }
    Packet.WriteS64( SystemTickFrequency );
}

//==============================================================================
}} // namespace
//==============================================================================

#endif // ENABLE_BENCHMARK_TESTING
