﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/timesrv/detail/service/timesrv_ServiceProvider.h>

#include <nn/timesrv/detail/settings/timesrv_ClockSettings.h>
#include <nn/timesrv/detail/fs/timesrv_TimeZoneBinaryManager.h>
#include <nn/fs/fs_PosixTime.h>

#if NN_DETAIL_TIME_CONFIG_SERVICE_FRAMEWORK_MODEL == NN_DETAIL_TIME_CONFIG_SERVICE_FRAMEWORK_MODEL_HIPC
#include <nn/psc.h>
#include <nn/timesrv/detail/core/timesrv_PscHandler-spec.nx.h>
#endif
#if NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL == NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL_NX_PMIC
#include <nn/timesrv/detail/core/timesrv_PmicApi-hardware.nx.h>
#endif

namespace nn
{
namespace timesrv
{
namespace detail
{
namespace service
{

namespace
{
#if NN_DETAIL_TIME_CONFIG_SERVICE_FRAMEWORK_MODEL == NN_DETAIL_TIME_CONFIG_SERVICE_FRAMEWORK_MODEL_HIPC

    class MyPmStateChangeCallback : public core::PmStateChangeCallback
    {
    public:
        explicit MyPmStateChangeCallback(core::StandardSteadyClockCore* pStandardSteadyClockCore) NN_NOEXCEPT:
            m_pStandardSteadyClockCore(pStandardSteadyClockCore)
        {
            NN_SDK_ASSERT_NOT_NULL(pStandardSteadyClockCore);

            NN_UNUSED(m_pStandardSteadyClockCore);
        }

        virtual void ShutdownReadyCallback() NN_NOEXCEPT NN_OVERRIDE
        {
            // シャットダウン時の処理が必要ならここに記述
        }

    private:
        core::StandardSteadyClockCore* m_pStandardSteadyClockCore;
    };
#endif
}

// ---------------------------------------------------

PosixTimeNotifierToFs::PosixTimeNotifierToFs(
    core::StandardUserSystemClockCore* pStandardUserSystemClockCore,
    core::TimeZoneServiceCore* pTimeZoneServiceCore) NN_NOEXCEPT:
        m_IsAvailable(false),
        m_pStandardUserSystemClockCore(pStandardUserSystemClockCore),
        m_pTimeZoneServiceCore(pTimeZoneServiceCore)
{
    NN_SDK_ASSERT_NOT_NULL(m_pStandardUserSystemClockCore);
    NN_SDK_ASSERT_NOT_NULL(m_pTimeZoneServiceCore);
}

void PosixTimeNotifierToFs::Activate() NN_NOEXCEPT
{
    m_IsAvailable = true;
}

void PosixTimeNotifierToFs::Notify() NN_NOEXCEPT
{
    auto result = NotifyImpl();
    if (result.IsFailure())
    {
        NN_DETAIL_TIME_ERROR("Failed to PosixTimeNotifierToFs::NotifyImpl(). (%08x, %03d-%04d)\n",
            result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription());
    }
    NN_UNUSED(result);
}

nn::Result PosixTimeNotifierToFs::NotifyImpl() NN_NOEXCEPT
{
    if(!m_IsAvailable)
    {
        NN_RESULT_SUCCESS;
    }

    nn::time::PosixTime currentPosix;
    NN_RESULT_DO(m_pStandardUserSystemClockCore->GetCurrentTime(&currentPosix));

    nn::time::CalendarTime c;
    nn::time::sf::CalendarAdditionalInfo info;
    NN_RESULT_DO(m_pTimeZoneServiceCore->ToCalendarTimeWithMyRule(&c, &info, currentPosix));

    // fs プロセスへユーザー時計の時刻を通知
    NN_RESULT_DO(nn::fs::SetCurrentPosixTime(currentPosix, info.timeZone.utcOffsetSeconds));
    //NN_DETAIL_TIME_TRACE("nn::fs::SetCurrentPosixTime %lld, utcOffsetSeconds:%d (%04d/%02d/%02d %02d:%02d:%02d[%s])\n",
    //    currentPosix.value, info.timeZone.utcOffsetSeconds, c.year, c.month, c.day, c.hour, c.minute, c.second, info.timeZone.standardTimeName);

    NN_RESULT_SUCCESS;
}

// ---------------------------------------------------

LocalSystemClockContextWriter::LocalSystemClockContextWriter(
    PosixTimeNotifierToFs* pPosixTimeNotifierToFs,
    core::WorkerThread* pWorkerThread,
    core::SharedMemoryManager* pSharedMemoryManager) NN_NOEXCEPT:
        m_pPosixTimeNotifierToFs(pPosixTimeNotifierToFs),
        m_pWorkerThread(pWorkerThread),
        m_pSharedMemoryManager(pSharedMemoryManager)
{
    NN_SDK_ASSERT_NOT_NULL(pPosixTimeNotifierToFs);
    NN_SDK_ASSERT_NOT_NULL(pWorkerThread);
    NN_SDK_ASSERT_NOT_NULL(pSharedMemoryManager);
}

nn::Result LocalSystemClockContextWriter::Set(const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT
{
    // 変更なければ何もしない
    if (m_PreviousSystemClockContext && systemClockContext == m_PreviousSystemClockContext)
    {
        NN_RESULT_SUCCESS;
    }
    m_PreviousSystemClockContext = systemClockContext;

    m_pSharedMemoryManager->SetStandardLocalSystemClockContext(systemClockContext);

    // IPC スレッドを長時間ブロックしないよう非同期書き込み
    m_pWorkerThread->WriteLocalSystemClockContextAsync(systemClockContext);

    m_pPosixTimeNotifierToFs->Notify();

    NN_RESULT_SUCCESS;
}

// ---------------------------------------------------

NetworkSystemClockContextWriter::NetworkSystemClockContextWriter(
    PosixTimeNotifierToFs* pPosixTimeNotifierToFs,
    core::WorkerThread* pWorkerThread,
    core::SharedMemoryManager* pSharedMemoryManager) NN_NOEXCEPT:
        m_pPosixTimeNotifierToFs(pPosixTimeNotifierToFs),
        m_pWorkerThread(pWorkerThread),
        m_pSharedMemoryManager(pSharedMemoryManager)
{
    NN_SDK_ASSERT_NOT_NULL(pPosixTimeNotifierToFs);
    NN_SDK_ASSERT_NOT_NULL(pWorkerThread);
    NN_SDK_ASSERT_NOT_NULL(pSharedMemoryManager);
}

nn::Result NetworkSystemClockContextWriter::Set(const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT
{
    // 変更なければ何もしない
    if (m_PreviousSystemClockContext && systemClockContext == m_PreviousSystemClockContext)
    {
        NN_RESULT_SUCCESS;
    }
    m_PreviousSystemClockContext = systemClockContext;

    m_pSharedMemoryManager->SetStandardNetworkSystemClockContext(systemClockContext);

    // IPC スレッドを長時間ブロックしないよう非同期書き込み
    m_pWorkerThread->WriteNetworkSystemClockContextAsync(systemClockContext);

    m_pPosixTimeNotifierToFs->Notify();

    NN_RESULT_SUCCESS;
}

// ---------------------------------------------------

ServiceProvider::GeneralArranger::GeneralArranger() NN_NOEXCEPT
{
    // 各コンポーネントの初期化前に必要な処理をここに書く

    fs::TimeZoneBinaryManager::MountFs();
}

ServiceProvider::GeneralArranger::~GeneralArranger() NN_NOEXCEPT
{
}

ServiceProvider::ServiceProvider() NN_NOEXCEPT
    : m_GeneralArranger(),
      m_StandardLocalSystemClockCore(&m_StandardSteadyClockCore),
      m_StandardNetworkSystemClockCore(&m_StandardSteadyClockCore),
      m_StandardUserSystemClockCore(&m_StandardLocalSystemClockCore, &m_StandardNetworkSystemClockCore),
      m_TimeZoneServiceCore(&m_PosixTimeNotifierToFs),
      m_EphemeralNetworkSystemClockCore(&m_TickBasedSteadyClockCore),
      m_WorkerThread(&m_StandardSteadyClockCore, &m_SharedMemoryManager, &m_PosixTimeNotifierToFs),
      m_PosixTimeNotifierToFs(&m_StandardUserSystemClockCore, &m_TimeZoneServiceCore),
      m_LocalSystemClockContextWriter(&m_PosixTimeNotifierToFs, &m_WorkerThread, &m_SharedMemoryManager),
      m_NetworkSystemClockContextWriter(&m_PosixTimeNotifierToFs, &m_WorkerThread, &m_SharedMemoryManager),
      m_HipcServerManager(this)
{
    // 以下、時計のセットアップ順序の変更は気を付けること
    SetupStandardSteadyClockCore(); // 他のすべての時計で利用されるので、最初にセットアップすること.
    SetupStandardLocalSystemClockCore();
    SetupStandardNetworkSystemClockCore();
    SetupStandardUserSystemClockCore(); // StandardLocalSystemClockCore, StandardNetworkSystemClockCore のセットアップ完了が前提.
    SetupEphemeralNetworkSystemClockCore();

    // 時計以外のセットアップ
    SetupTimeZoneServiceCore();

    // m_PosixTimeNotifierToFs は StandardUserSystemClock のセットアップ完了後に有効化すること
    m_PosixTimeNotifierToFs.Activate();
    // サービス起動時に一回通知しとく
    m_PosixTimeNotifierToFs.Notify();
}

ServiceProvider::~ServiceProvider() NN_NOEXCEPT
{
}

nn::Result ServiceProvider::Start() NN_NOEXCEPT
{
#if NN_DETAIL_TIME_CONFIG_SERVICE_FRAMEWORK_MODEL == NN_DETAIL_TIME_CONFIG_SERVICE_FRAMEWORK_MODEL_HIPC
    LoopService();

    // TODO:終了処理
    // ...
#endif

    NN_RESULT_SUCCESS;
}

nn::Result ServiceProvider::RequestStopServer() NN_NOEXCEPT
{
#if NN_DETAIL_TIME_CONFIG_SERVICE_FRAMEWORK_MODEL == NN_DETAIL_TIME_CONFIG_SERVICE_FRAMEWORK_MODEL_HIPC
    m_HipcServerManager.RequestStop();
#endif

    NN_RESULT_SUCCESS;
}

void ServiceProvider::SetupStandardSteadyClockCore() NN_NOEXCEPT
{
    nn::time::SourceId clockSourceId = settings::ReadSteadyClockSourceId();

#if NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL == NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL_NX_PMIC
    // RTC リセットハンドリング
    if(core::IsPmicRtcResetDetected())
    {
        // RTC リセットしているのにハンドリングされないケースを極力なくすため、すぐに InvalidUuid を永続化しておく.
        // 現在の SourceId を InvlaidUuId にしておくと、以降の処理で SourceId が再生成される.
        settings::WriteSteadyClockSourceId(nn::util::InvalidUuid);
        core::ClearPmicRtcResetDetected();
        clockSourceId = nn::util::InvalidUuid;
        m_StandardSteadyClockCore.DetectRtcReset();
    }
#endif

    // 永続データ注入、初期化
    m_StandardSteadyClockCore.SetInternalOffset(settings::ReadSteadyClockInternalOffset()); // SetupSourceId で使われる可能性があるので先に注入.

    nn::time::SourceId latestClockSourceId;
    m_StandardSteadyClockCore.SetupSourceId(&latestClockSourceId, clockSourceId);

    // 更新されていれば最新値を永続化
    if (latestClockSourceId != clockSourceId)
    {
        settings::WriteSteadyClockSourceId(latestClockSourceId);
    }

    // テストオフセット
    m_StandardSteadyClockCore.SetTestOffset(settings::ReadStandardSteadyClockTestOffset());

    // 共有メモリへ反映
    m_SharedMemoryManager.SetStandardSteadyClockTimePoint(latestClockSourceId, m_StandardSteadyClockCore.GetCurrentTimePointValue());
}
void ServiceProvider::SetupStandardLocalSystemClockCore() NN_NOEXCEPT
{
    // 永続化用の Writer 注入
    m_StandardLocalSystemClockCore.SetSystemClockContextUpdateCallback(&m_LocalSystemClockContextWriter);

    // 永続データ注入、初期化
    nn::time::SystemClockContext systemClockContext = settings::ReadLocalSystemClockContext();
    m_StandardLocalSystemClockCore.Initialize(systemClockContext);
}
void ServiceProvider::SetupStandardNetworkSystemClockCore() NN_NOEXCEPT
{
    // 永続化用の Writer 注入
    m_StandardNetworkSystemClockCore.SetSystemClockContextUpdateCallback(&m_NetworkSystemClockContextWriter);

    // 永続データ注入
    const auto currentContext = settings::ReadNetworkSystemClockContext();
    NN_ABORT_UNLESS_RESULT_SUCCESS( m_StandardNetworkSystemClockCore.SetSystemClockContext(currentContext) );

    m_StandardNetworkSystemClockCore.SetSufficientAccuracyThreshold(settings::ReadNetworkClockSufficientAccuracyThreshold());

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // Windows で未補正の場合、ネットワーク時計の時刻をPC時計で自動補正する

    nn::time::SteadyClockTimePoint timePoint;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_StandardSteadyClockCore.GetCurrentTimePoint(&timePoint));
    if (timePoint.sourceId != currentContext.timeStamp.sourceId) // 未補正
    {
        auto tp = std::chrono::system_clock::now();
        time_t t = std::chrono::system_clock::to_time_t(tp);
        nn::time::PosixTime networkTime = { static_cast<int64_t>(t) };
        m_StandardNetworkSystemClockCore.SetCurrentTime(networkTime);
    }
#endif
}
void ServiceProvider::SetupStandardUserSystemClockCore() NN_NOEXCEPT
{
    // 永続データ注入
    bool enabled = settings::ReadUserSystemClockAutomaticCorrectionEnabled();
    NN_ABORT_UNLESS_RESULT_SUCCESS( m_StandardUserSystemClockCore.SetAutomaticCorrectionEnabled(enabled) );

    m_StandardUserSystemClockCore.SetAutomaticCorrectionUpdatedTime(settings::ReadUserSystemClockAutomaticCorrectionUpdatedTime());

    // 共有メモリへ反映
    m_SharedMemoryManager.SetStandardUserSystemClockAutomaticCorrectionEnabled(enabled);
}
void ServiceProvider::SetupTimeZoneServiceCore() NN_NOEXCEPT
{
    // 永続データ注入
    m_TimeZoneServiceCore.SetDeviceLocationName(settings::ReadDeviceLocationName());
    m_TimeZoneServiceCore.SetDeviceLocationUpdatedTime(settings::ReadDeviceLocationUpdatedTime());

    // 起動中に変更されないシステムデータに伴うパラメータを計算、注入

    m_TimeZoneServiceCore.SetTotalLocationNameCount(fs::TimeZoneBinaryManager::GetLocationNameCount());

    nn::time::TimeZoneRuleVersion version;
    auto result = fs::TimeZoneBinaryManager::GetTimeZoneRuleVersion(&version);
    if (result.IsSuccess())
    {
        m_TimeZoneServiceCore.SetTimeZoneRuleVersion(version);
    }
}
void ServiceProvider::SetupEphemeralNetworkSystemClockCore() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // Windows だと必ず使えるようPC時計で補正
    auto tp = std::chrono::system_clock::now();
    time_t t = std::chrono::system_clock::to_time_t(tp);
    nn::time::PosixTime networkTime = { static_cast<int64_t>(t) };
    m_EphemeralNetworkSystemClockCore.SetCurrentTime(networkTime);
#endif
}

#if NN_DETAIL_TIME_CONFIG_SERVICE_FRAMEWORK_MODEL == NN_DETAIL_TIME_CONFIG_SERVICE_FRAMEWORK_MODEL_HIPC
void ServiceProvider::LoopService() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(nn::psc::PmModule, s_PmModule);
    NN_FUNCTION_LOCAL_STATIC(nn::os::MultiWaitHolderType, s_PmModuleEventHolder);

    MyPmStateChangeCallback callback(&m_StandardSteadyClockCore);
    auto resultOfInitializePmModule = core::InitializePmModule(&s_PmModule, &callback);

    if(resultOfInitializePmModule.IsSuccess())
    {
        nn::os::InitializeMultiWaitHolder(
            &s_PmModuleEventHolder, s_PmModule.GetEventPointer()->GetBase());

        nn::os::SetMultiWaitHolderUserData(
            &s_PmModuleEventHolder, nn::psc::PmModuleId_Time);

        m_HipcServerManager.AddUserWaitHolder(&s_PmModuleEventHolder);
    }

    while(NN_STATIC_CONDITION(true))
    {
        auto p = m_HipcServerManager.Wait();
        if(p == nullptr)
        {
            break;
        }
        switch (nn::os::GetMultiWaitHolderUserData(p))
        {
        case HipcServerManager::HipcSimpleAllInOneServerManager::AcceptTag:
        case HipcServerManager::HipcSimpleAllInOneServerManager::InvokeTag:
            {
                m_HipcServerManager.ProcessAuto(p);
            }
            break;
        case nn::psc::PmModuleId_Time:
            {
                s_PmModule.GetEventPointer()->Clear();
                m_HipcServerManager.AddUserWaitHolder(&s_PmModuleEventHolder);
                core::HandlePmModule(&s_PmModule);
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    if(resultOfInitializePmModule.IsSuccess())
    {
        core::FinalizePmModule(&s_PmModule);
    }
}
#endif

}
}
}
}
