﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/time/detail/time_SharedMemoryClient.h>

namespace nn { namespace time { namespace detail {

SharedMemoryClient::SharedMemoryClient(nn::sf::SharedPointer<nn::timesrv::detail::service::IStaticService> pIStaticService) NN_NOEXCEPT
    : m_pSharedMemoryClockSettings(nullptr)
{
    NN_SDK_ASSERT_NOT_NULL(pIStaticService);

    // Attach
    nn::sf::NativeHandle nativeHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pIStaticService->GetSharedMemoryNativeHandle(&nativeHandle));
    m_SharedMemory.Attach(nn::time::detail::SharedMemoryMappingSize, nativeHandle.GetOsHandle(), nativeHandle.IsManaged());
    nativeHandle.Detach();

    // Map
    void* p = m_SharedMemory.Map(nn::os::MemoryPermission_ReadOnly);
    NN_ABORT_UNLESS_NOT_NULL(p);

    m_pSharedMemoryClockSettings = static_cast<nn::time::detail::SharedMemoryClockSettings*>(p);
}

void SharedMemoryClient::GetStandardSteadyClockCurrentTimePoint(nn::time::SteadyClockTimePoint* pOut) NN_NOEXCEPT
{
    auto data = nn::util::LoadFromLockFreeAtomicType(&m_pSharedMemoryClockSettings->standardSteadyClockTimePoint);

    pOut->value = (nn::TimeSpan(data.baseTime) + nn::os::GetSystemTick().ToTimeSpan()).GetSeconds();
    pOut->sourceId = data.sourceId;
}

void SharedMemoryClient::GetStandardUserSystemClockCurrentTime(nn::time::PosixTime* pOut) NN_NOEXCEPT
{
    // standardLocalSystemClockContext が自動補正ONを考慮した値になっているのでそのまま利用する
    auto localClockContext = nn::util::LoadFromLockFreeAtomicType(&m_pSharedMemoryClockSettings->standardLocalSystemClockContext);

    NN_ABORT_UNLESS_RESULT_SUCCESS(CalculateSystemClockCurrentTime(pOut, localClockContext));
}

void SharedMemoryClient::GetStandardUserSystemClockContext(nn::time::SystemClockContext* pOut) NN_NOEXCEPT
{
    // standardLocalSystemClockContext が自動補正ONを考慮した値になっているのでそのまま利用する
    *pOut = nn::util::LoadFromLockFreeAtomicType(&m_pSharedMemoryClockSettings->standardLocalSystemClockContext);
}

bool SharedMemoryClient::IsStandardUserSystemClockAutomaticCorrectionEnabled() NN_NOEXCEPT
{
    return nn::util::LoadFromLockFreeAtomicType(&m_pSharedMemoryClockSettings->isStandardUserSystemClockAutomaticCorrectionEnabled);
}

nn::Result SharedMemoryClient::GetStandardNetworkSystemClockCurrentTime(nn::time::PosixTime* pOut) NN_NOEXCEPT
{
    auto networkClockContext = nn::util::LoadFromLockFreeAtomicType(&m_pSharedMemoryClockSettings->standardNetworkSystemClockContext);

    return CalculateSystemClockCurrentTime(pOut, networkClockContext);
}

void SharedMemoryClient::GetStandardNetworkSystemClockContext(nn::time::SystemClockContext* pOut) NN_NOEXCEPT
{
    *pOut = nn::util::LoadFromLockFreeAtomicType(&m_pSharedMemoryClockSettings->standardNetworkSystemClockContext);
}

nn::Result SharedMemoryClient::CalculateSystemClockCurrentTime(nn::time::PosixTime* pOut, const nn::time::SystemClockContext& context) NN_NOEXCEPT
{
    auto steadyClockTimePoint = nn::util::LoadFromLockFreeAtomicType(&m_pSharedMemoryClockSettings->standardSteadyClockTimePoint);

    NN_RESULT_THROW_UNLESS(context.timeStamp.sourceId == steadyClockTimePoint.sourceId, nn::time::ResultOffsetInvalid());

    pOut->value = context.offset +
        (nn::TimeSpan(steadyClockTimePoint.baseTime) + nn::os::GetSystemTick().ToTimeSpan()).GetSeconds();

    NN_RESULT_SUCCESS;
}

}}}
