﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>

#include <nn/sf/sf_Types.h> // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h> // for nn::sf::NativeHandle
#include <nn/sf/sf_ShimLibraryUtility.h>

#include <nn/tc/tc_IManager.sfdl.h> // for IManager
#include <nn/tc/tc_ServiceName.h>
#include <nn/tc/tc_Types.h>
#include <nn/tc/tc_VirtualTemperature.h>

namespace nn { namespace tc {

namespace {

const int NumberOfClients = 1;

sf::SimpleAllInOneHipcClientManager<NumberOfClients> g_ClientManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;
sf::ShimLibraryObjectHolder<IManager> g_Holder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

} // namespace

void Initialize() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ClientManager.InitializeShimLibraryHolder(&g_Holder, nn::tc::TcServiceName));
}

void InitializeWith(sf::SharedPointer<IManager>&& manager) NN_NOEXCEPT
{
    g_Holder.InitializeHolderDirectly(std::move(manager));
}

nn::sf::SharedPointer<IManager> GetInternalManager() NN_NOEXCEPT
{
    return g_Holder.GetObject();
}

void Finalize() NN_NOEXCEPT
{
    g_Holder.FinalizeHolder();
}

nn::Result SetOperatingMode(OperatingMode operatingMode) NN_NOEXCEPT
{
    NN_RESULT_DO(g_Holder.GetObject()->SetOperatingMode(operatingMode));
    NN_RESULT_SUCCESS;
}

nn::Result GetThermalEvent(nn::os::SystemEventType* pOutEvent, EventTarget target, nn::os::EventClearMode mode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutEvent);
    nn::sf::NativeHandle sfHandle;
    NN_RESULT_DO(g_Holder.GetObject()->GetThermalEvent(&sfHandle, target));
    nn::os::AttachReadableHandleToSystemEvent(pOutEvent, sfHandle.GetOsHandle(), sfHandle.IsManaged(), mode);
    sfHandle.Detach();
    NN_RESULT_SUCCESS;
}

nn::Result IsActionAllowed(bool* pOutPossible, Action action) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutPossible);
    NN_RESULT_DO(g_Holder.GetObject()->IsActionAllowed(pOutPossible, action));
    NN_RESULT_SUCCESS;
}

nn::Result EnableVirtualTemperature(Location location) NN_NOEXCEPT
{
    NN_RESULT_DO(g_Holder.GetObject()->EnableVirtualTemperature(location));
    NN_RESULT_SUCCESS;
}

nn::Result DisableVirtualTemperature(Location location) NN_NOEXCEPT
{
    NN_RESULT_DO(g_Holder.GetObject()->DisableVirtualTemperature(location));
    NN_RESULT_SUCCESS;
}

nn::Result SetVirtualTemperature(Location location, TemperatureMilliC temperature) NN_NOEXCEPT
{
    NN_RESULT_DO(g_Holder.GetObject()->SetVirtualTemperature(location, temperature));
    NN_RESULT_SUCCESS;
}

nn::Result EnableFanControl() NN_NOEXCEPT
{
    NN_RESULT_DO(g_Holder.GetObject()->EnableFanControl());
    NN_RESULT_SUCCESS;
}

nn::Result DisableFanControl() NN_NOEXCEPT
{
    NN_RESULT_DO(g_Holder.GetObject()->DisableFanControl());
    NN_RESULT_SUCCESS;
}

nn::Result IsFanControlEnabled(bool* pOutEnabled) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutEnabled);
    NN_RESULT_DO(g_Holder.GetObject()->IsFanControlEnabled(pOutEnabled));
    NN_RESULT_SUCCESS;
}

TemperatureMilliC GetSkinTemperatureMilliC() NN_NOEXCEPT
{
    TemperatureMilliC skinTemperatureMilliC;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Holder.GetObject()->GetSkinTemperatureMilliC(&skinTemperatureMilliC));
    return skinTemperatureMilliC;
}

}} // namespace nn::tc
