﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdio>
#include <cstdarg>
#include <string>

#include <nn/nn_SdkAssert.h>

#include <nn/erpt.h>
#include <nn/err.h>
#include <nn/err/detail/err_ErrorCodeConvert.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/tc/detail/tc_Log.h>
#include <nn/tc/tc_Result.h>

#include "../tc_Constants.h"
#include "tc_ThermalCoordinator.h"
#include "tc_SettingsHolder.h"

namespace nn { namespace tc { namespace impl { namespace detail {

namespace {

#define NN_DETAIL_TC_WARN_UNLESS_RESULT_SUCCESS(expression) \
    do \
    { \
        auto result_ = (expression); \
        if ( result_.IsFailure() ) \
        { \
            NN_DETAIL_TC_WARN("Failed: %s\n  Module: %d\n  Description: %d\n  InnerValue: 0x%08x\n", \
                NN_MACRO_STRINGIZE(expression), result_.GetModule(), result_.GetDescription(), result_.GetInnerValueForDebug()); \
        } \
    } while ( NN_STATIC_CONDITION(false) )

const TemperatureMilliC SleepSkinThreshold = 63000;
const TemperatureMilliC SleepRawThreshold = 84000; // 85C を上限として誤差 1C のマージンを確保して 84C とする。温度センサの誤差は LocationAccessor で考慮済み。
const TemperatureMilliC HandheldSkinTemperatureMiddleThreshold = 58000;
const TemperatureMilliC HandheldSkinTemperatureHighThreshold = 61000;
const int64_t HandheldMiddleThresholdSeconds = 60;
const int64_t HandheldHighThresholdSeconds = 10;

const TemperatureMilliC TemperatureEstimationError = 500;

// この温度以下の場合 SkinTemperature を計算せずセンサ温度を SkinTemperature とみなす。
const TemperatureMilliC SkinTemperatureEstimationThreshold = 38000;

// 表面温度導出式の記述ミスを防ぐために本関数に式を集約します。
TemperatureMilliC CalcSkinTemperature(TemperatureMilliC junctionTemperature, const TskinCoefficients& tskinCoefficients) NN_NOEXCEPT
{
    if ( junctionTemperature < SkinTemperatureEstimationThreshold )
    {
        return junctionTemperature;
    }

    auto skinTemperature = static_cast<TemperatureMilliC>((
        tskinCoefficients.a * static_cast<int64_t>(junctionTemperature) + tskinCoefficients.b * 1000LL) / 10000LL) + TemperatureEstimationError;

    // SIGLO-85208
    return std::min(junctionTemperature, skinTemperature);
}

// 閾値表示の為の関数です。
TemperatureMilliC CalcJunctionTemperature(TemperatureMilliC skinTemperature, const TskinCoefficients& tskinCoefficients) NN_NOEXCEPT
{
    return static_cast<TemperatureMilliC>(((
        static_cast<int64_t>(skinTemperature) - TemperatureEstimationError) * 10000LL - tskinCoefficients.b * 1000LL) / tskinCoefficients.a);
}

} // namespace

bool ThermalCounter::Update(TemperatureMilliC temperature) NN_NOEXCEPT
{
    bool overThreshold = false;

    if ( temperature < m_ThermalThreshold )
    {
        m_StartTime = nn::TimeSpan(0);
    }
    else
    {
        if ( m_StartTime == nn::TimeSpan(0) )
        {
            m_StartTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        }
        else
        {
            nn::TimeSpan span = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()) - m_StartTime;
            if ( span >= m_TimeThreshold )
            {
                NN_DETAIL_TC_TRACE("Over temperature threshold %d MilliC for %d seconds.\n", m_ThermalThreshold, span.GetSeconds());
                overThreshold = true;
            }
        }
    }

    return overThreshold;
};

void ThermalCounter::Clear() NN_NOEXCEPT
{
    m_StartTime = nn::TimeSpan(0);
};

ThermalCoordinator::ThermalCoordinator() NN_NOEXCEPT
    : m_LocationPcb(nn::ts::Location_ThermalSensorInternal)
    , m_LocationSoc(nn::ts::Location_ThermalSensorExternal)
    , m_IirFilterPcb()
    , m_IirFilterSoc()
    , m_ThermalHandler()
    , m_SettingHandler()
    , m_OperatingMode(OperatingMode_Handheld)
    , m_AbnormalTemperatureEvent()
    , m_FailureCount(0)
    , m_CounterHandheldMiddle(HandheldSkinTemperatureMiddleThreshold, nn::TimeSpan::FromSeconds(HandheldMiddleThresholdSeconds))
    , m_CounterHandheldHigh(HandheldSkinTemperatureHighThreshold, nn::TimeSpan::FromSeconds(HandheldHighThresholdSeconds))
    , m_HasHighTemperatureBeenDetected(false)
    , m_SettingsHolder()
    , m_SkinTemperatureMilliC(DefaultSkinTemperatureMilliC)
{
    ;
}

void ThermalCoordinator::DumpThresholds() NN_NOEXCEPT
{
    if ( m_SettingsHolder.IsSleepEnabled() )
    {
        // スリープの Tsoc / Tpcb の閾値をログ出力します。
        // 温度が高い方 Tsoc を先に表示します。情報が多い handheld を先に表示します。
        // handheld, console 各状態の整合性を見るために handheld, console を大分類にします。
        NN_DETAIL_TC_TRACE("|               | Traw sleep  | Tskin sleep | Tskin 10sec | Tskin 60sec |\n");
        NN_DETAIL_TC_TRACE("+---------------+-------------+-------------+-------------+-------------+\n");
        NN_DETAIL_TC_TRACE("| handheld Tsoc | %8d mC | %8d mC | %8d mC | %8d mC |\n",
            SleepRawThreshold,
            CalcJunctionTemperature(SleepSkinThreshold, m_SettingsHolder.GetTskinCoefficientsSocHandheld()),
            CalcJunctionTemperature(HandheldSkinTemperatureHighThreshold, m_SettingsHolder.GetTskinCoefficientsSocHandheld()),
            CalcJunctionTemperature(HandheldSkinTemperatureMiddleThreshold, m_SettingsHolder.GetTskinCoefficientsSocHandheld())
        );
        NN_DETAIL_TC_TRACE("|          Tpcb | %8d mC | %8d mC | %8d mC | %8d mC |\n",
            SleepRawThreshold,
            CalcJunctionTemperature(SleepSkinThreshold, m_SettingsHolder.GetTskinCoefficientsPcbHandheld()),
            CalcJunctionTemperature(HandheldSkinTemperatureHighThreshold, m_SettingsHolder.GetTskinCoefficientsPcbHandheld()),
            CalcJunctionTemperature(HandheldSkinTemperatureMiddleThreshold, m_SettingsHolder.GetTskinCoefficientsPcbHandheld())
        );

        // Release ビルドでの警告防止
        auto thresholdSoc = CalcJunctionTemperature(SleepSkinThreshold, m_SettingsHolder.GetTskinCoefficientsSocConsole());
        NN_DETAIL_TC_TRACE("| console  Tsoc | %8d mC | %8d mC |        - mC |        - mC |\n",
            SleepRawThreshold,
            thresholdSoc
        );
        NN_UNUSED(thresholdSoc);

        // Release ビルドでの警告防止
        auto thresholdPcb = CalcJunctionTemperature(SleepSkinThreshold, m_SettingsHolder.GetTskinCoefficientsPcbConsole());
        NN_DETAIL_TC_TRACE("|          Tpcb | %8d mC | %8d mC |        - mC |        - mC |\n",
            SleepRawThreshold,
            thresholdPcb
        );
        NN_UNUSED(thresholdPcb);
    }
}

void ThermalCoordinator::Initialize() NN_NOEXCEPT
{
    nn::os::CreateSystemEvent(&m_AbnormalTemperatureEvent, nn::os::EventClearMode_ManualClear, true);

    // fwdbg settings の設定値の取得。
    m_SettingsHolder.LoadSettings();

    // 包含するオブジェクトの初期化。
    m_LocationPcb.Initialize();
    m_LocationSoc.Initialize();

    m_ThermalHandler.Initialize(&m_SettingsHolder);
    m_SettingHandler.Initialize();

    m_IirFilterSoc.SetGain(m_SettingsHolder.GetIirFilterGainSoc() / 100.0);
    m_IirFilterPcb.SetGain(m_SettingsHolder.GetIirFilterGainPcb() / 100.0);

    DumpThresholds();
}

void ThermalCoordinator::Finalize() NN_NOEXCEPT
{
    m_SettingHandler.Finalize();
    m_ThermalHandler.Finalize();

    m_LocationPcb.Finalize();
    m_LocationSoc.Finalize();

    nn::os::DestroySystemEvent(&m_AbnormalTemperatureEvent);
}

void ThermalCoordinator::EnterSleep(const char* pFormat, ...) NN_NOEXCEPT
{
    const size_t bufferSize = 0x100;
    char buffer[bufferSize];
    va_list arg;

    std::memset(buffer, '\0', bufferSize);

    va_start(arg, pFormat);
    size_t size = std::vsnprintf(buffer, bufferSize, pFormat, arg);
    va_end(arg);

    NN_SDK_ASSERT(size < bufferSize);

    if ( size > 0 && size < bufferSize )
    {
        NN_DETAIL_TC_WARN_V1("%s\n", buffer);
    }

    if ( m_SettingsHolder.IsSleepEnabled() )
    {
        NN_DETAIL_TC_WARN("Sleep system because of high temperature!\n");
        nn::os::SignalSystemEvent(&m_AbnormalTemperatureEvent);
    }
}

// ThermalInfo カテゴリのコンテキスト更新
void ThermalCoordinator::UpdateErrorReportContext(TemperatureMilliC pcbTemperature, TemperatureMilliC socTemperature) NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::ThermalInfo);
    NN_DETAIL_TC_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::TemperaturePcb, pcbTemperature));
    NN_DETAIL_TC_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::TemperatureSoc, socTemperature));
    NN_DETAIL_TC_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::CurrentFanDuty, m_ThermalHandler.GetFanActualRotationSpeedLevel()));
    NN_DETAIL_TC_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::LastDvfsThresholdTripped, m_SettingHandler.GetLastDvfsTemperature()));
    NN_DETAIL_TC_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
}

// ThermalInfo カテゴリのコンテキストを更新してから、エラーレポートを作成する
void ThermalCoordinator::SubmitErrorReport(Result result, TemperatureMilliC pcbTemperature, TemperatureMilliC socTemperature) NN_NOEXCEPT
{
    UpdateErrorReportContext(pcbTemperature, socTemperature);

    nn::erpt::Context context(nn::erpt::ErrorInfo);

    // エラーレポートの作成にエラーコードの指定が必要で、エラーコードは文字列で指定する必要がある
    auto errorCode = nn::err::detail::ConvertResultToErrorCode(result);
    char errorCodeString[nn::err::ErrorCode::StringLengthMax];
    util::SNPrintf(errorCodeString, sizeof(errorCodeString), "%04d-%04d", errorCode.category, errorCode.number);
    NN_DETAIL_TC_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::ErrorCode, errorCodeString, static_cast<uint32_t>(sizeof(errorCodeString))));

    NN_DETAIL_TC_WARN_UNLESS_RESULT_SUCCESS(context.CreateReport(nn::erpt::ReportType_Invisible));
}

// 定期的に実行されることを想定した主要な処理。
void ThermalCoordinator::Update() NN_NOEXCEPT
{
    // 最大のセンサ温度を基準値とする。
    TemperatureMilliC pcbRawTemperature = 0;
    TemperatureMilliC socRawTemperature = 0;
    TemperatureMilliC socRealTemperature = 0;

    // 何らかの原因で温度が取得できなかった場合、3回まで取得を試みて、駄目なら強制シャットダウンする。
    auto resultPcb = m_LocationPcb.GetTemperature(&pcbRawTemperature);
    auto resultSoc = m_LocationSoc.GetTemperature(&socRawTemperature);
    auto resultSocReal = m_LocationSoc.GetRealTemperature(&socRealTemperature);
    if ( resultPcb.IsFailure() || resultSoc.IsFailure() || resultSocReal.IsFailure() )
    {
        m_FailureCount++;
        if ( m_FailureCount <= 3 )
        {
            NN_DETAIL_TC_WARN("Could not get temperature. count = %d\n", m_FailureCount);
            return;
        }
        else
        {
            NN_DETAIL_TC_ERROR_V1("Could not get temperature 4 times\n");
            NN_DETAIL_TC_ERROR("Shutdown system because of thermal sensor error!\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(ResultTemperatureUnavailable());
        }
    }
    m_FailureCount = 0;

    // 温度センサによる強制シャットダウンがかかる前にスリープさせる為に Raw でスレッショルドを確認する。
    if ( socRawTemperature >= SleepRawThreshold && !m_HasHighTemperatureBeenDetected )
    {
        SubmitErrorReport(ResultSocTemperatureHigh(), pcbRawTemperature, socRawTemperature);
        EnterSleep("socTemperature is %d MilliC (threshold %d MilliC)!", socRawTemperature, SleepRawThreshold);
        m_HasHighTemperatureBeenDetected = true;
    }

    // 温度センサによる強制シャットダウンがかかる前にスリープさせる為に Raw でスレッショルドを確認する。
    if ( pcbRawTemperature >= SleepRawThreshold && !m_HasHighTemperatureBeenDetected )
    {
        SubmitErrorReport(ResultPcbTemperatureHigh(), pcbRawTemperature, socRawTemperature);
        EnterSleep("pcbTemperature is %d MilliC (threshold %d MilliC)!", pcbRawTemperature, SleepRawThreshold);
        m_HasHighTemperatureBeenDetected = true;
    }

    // 温度変化を鈍らせる。
    TemperatureMilliC pcbJunctionTemperature = m_IirFilterPcb.Update(pcbRawTemperature);
    TemperatureMilliC socJunctionTemperature = m_IirFilterSoc.Update(socRawTemperature);

    TemperatureMilliC pcbSkinTemperature;
    TemperatureMilliC socSkinTemperature;

    // 表面温度を計算する。現在は予測式は同一。
    if ( m_OperatingMode == OperatingMode_Handheld )
    {
        socSkinTemperature = CalcSkinTemperature(socJunctionTemperature, m_SettingsHolder.GetTskinCoefficientsSocHandheld());
        pcbSkinTemperature = CalcSkinTemperature(pcbJunctionTemperature, m_SettingsHolder.GetTskinCoefficientsPcbHandheld());
    }
    else
    {
        socSkinTemperature = CalcSkinTemperature(socJunctionTemperature, m_SettingsHolder.GetTskinCoefficientsSocConsole());
        pcbSkinTemperature = CalcSkinTemperature(pcbJunctionTemperature, m_SettingsHolder.GetTskinCoefficientsPcbConsole());
    }

    // 表面温度を計算する。
    TemperatureMilliC skinTemperature = std::max(pcbSkinTemperature, socSkinTemperature);

    switch ( m_SettingsHolder.GetTskinSelect() )
    {
    case TskinSelect::Soc:
        skinTemperature = socSkinTemperature;
        break;
    case TskinSelect::Pcb:
        skinTemperature = pcbSkinTemperature;
        break;
    case TskinSelect::Both:
        skinTemperature = std::max(pcbSkinTemperature, socSkinTemperature);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_SkinTemperatureMilliC = skinTemperature;

    if ( m_OperatingMode == OperatingMode_Handheld )
    {
        // 表面温度 58C 以上 60 秒間維持でスリープに遷移する。
        if ( m_CounterHandheldMiddle.Update(skinTemperature) && !m_HasHighTemperatureBeenDetected )
        {
            SubmitErrorReport(ResultHandheldSkinTemperatureHigh(), pcbRawTemperature, socRawTemperature);
            EnterSleep("skinTemperature is %d MilliC (threshold %d MilliC)!", skinTemperature, HandheldSkinTemperatureMiddleThreshold);
            m_HasHighTemperatureBeenDetected = true;
        }

        // 表面温度 61C 以上 10 秒間維持でスリープに遷移する。
        if ( m_CounterHandheldHigh.Update(skinTemperature) && !m_HasHighTemperatureBeenDetected )
        {
            SubmitErrorReport(ResultHandheldSkinTemperatureVeryHigh(), pcbRawTemperature, socRawTemperature);
            EnterSleep("skinTemperature is %d MilliC (threshold %d MilliC)!", skinTemperature, HandheldSkinTemperatureHighThreshold);
            m_HasHighTemperatureBeenDetected = true;
        }
    }
    else
    {
        m_CounterHandheldMiddle.Clear();
        m_CounterHandheldHigh.Clear();
    }

    if ( skinTemperature >= SleepSkinThreshold && !m_HasHighTemperatureBeenDetected )
    {
        // 表面温度 63C 以上でスリープに遷移する。
        SubmitErrorReport(ResultSkinTemperatureHigh(), pcbRawTemperature, socRawTemperature);
        EnterSleep("skinTemperature is %d MilliC (threshold %d MilliC)!", skinTemperature, SleepSkinThreshold);
        m_HasHighTemperatureBeenDetected = true;
    }

    m_ThermalHandler.SetTemperature(skinTemperature, pcbRawTemperature, socRawTemperature);
    m_SettingHandler.SetTemperature(socRealTemperature);

    if ( m_SettingsHolder.IsLogEnabled() )
    {
        NN_DETAIL_TC_INFO("[evaluation], ms:%lld, PCB:%d,%d,%d, SoC:%d,%d,%d, Skin:%d, Tgt:%d,%d,%d, Cur:%d\n",
            nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMilliSeconds(),
            pcbRawTemperature,
            pcbJunctionTemperature,
            pcbSkinTemperature,
            socRawTemperature,
            socJunctionTemperature,
            socSkinTemperature,
            skinTemperature,
            m_ThermalHandler.GetRotationSpeedLevel(RateSelect::Table),
            m_ThermalHandler.GetRotationSpeedLevel(RateSelect::Prev),
            m_ThermalHandler.GetFanRotationSpeedLevel(),
            m_ThermalHandler.GetFanActualRotationSpeedLevel());
    }

    UpdateErrorReportContext(pcbRawTemperature, socRawTemperature);
} // NOLINT(impl/function_size)

nn::Result ThermalCoordinator::SetOperatingMode(OperatingMode operatingMode) NN_NOEXCEPT
{
    nn::Result result;

    switch ( operatingMode )
    {
    case OperatingMode_Console:
    case OperatingMode_Handheld:
        m_OperatingMode = operatingMode;
        m_ThermalHandler.SetOperatingMode(operatingMode);
        result = ResultSuccess();
        break;
    default:
        result = ResultInvalidOperatingModeId();
        break;
    }

    return result;
}

// 本関数は実装されているが機能としては未実装であり SystemEvent は発生しない。
nn::Result ThermalCoordinator::GetEventPtr(nn::os::SystemEventType** pOutEventPtr, EventTarget target) NN_NOEXCEPT
{
    nn::Result result;

    switch ( target )
    {
    case EventTarget_AbnormalTemperature:
        *pOutEventPtr = &m_AbnormalTemperatureEvent;
        result = ResultSuccess();
        break;
    default:
        result = ResultInvalidEventTargetId();
        break;
    }

    return result;
}

// 本関数は未実装であり常に true を返す。
nn::Result ThermalCoordinator::IsActionAllowed(bool* pOutPossible, Action action) NN_NOEXCEPT
{
    nn::Result result;

    switch ( action )
    {
    case Action_Startup:
        *pOutPossible = true;
        result = ResultSuccess();
        break;
    default:
        result = ResultInvalidActionId();
        break;
    }

    return result;
}

nn::Result ThermalCoordinator::SetVirtualTemperatureEnabled(Location location, bool enabled) NN_NOEXCEPT
{
    if ( location == Location_ThermalSensorInternal )
    {
        m_LocationPcb.SetVirtualTemperatureEnabled(enabled);
        NN_RESULT_SUCCESS;
    }
    else if ( location == Location_ThermalSensorExternal )
    {
        m_LocationSoc.SetVirtualTemperatureEnabled(enabled);
        NN_RESULT_SUCCESS;
    }
    else
    {
        return ResultInvalidLocationId();
    }
}

nn::Result ThermalCoordinator::SetVirtualTemperature(Location location, TemperatureMilliC temperature) NN_NOEXCEPT
{
    if ( location == Location_ThermalSensorInternal )
    {
        m_LocationPcb.SetVirtualTemperature(temperature);
        NN_RESULT_SUCCESS;
    }
    else if ( location == Location_ThermalSensorExternal )
    {
        m_LocationSoc.SetVirtualTemperature(temperature);
        NN_RESULT_SUCCESS;
    }
    else
    {
        return ResultInvalidLocationId();
    }
}

void ThermalCoordinator::SetPowerMode(PowerMode powerMode) NN_NOEXCEPT
{
    if ( powerMode == PowerMode_SleepReady )
    {
        m_CounterHandheldMiddle.Clear();
        m_CounterHandheldHigh.Clear();
        m_HasHighTemperatureBeenDetected = false;
    }

    m_ThermalHandler.SetPowerMode(powerMode);
    m_SettingHandler.SetPowerMode(powerMode);

    m_LocationPcb.SetPowerMode(powerMode);
    m_LocationSoc.SetPowerMode(powerMode);
}

nn::Result ThermalCoordinator::SetFanControlEnabled(bool enabled) NN_NOEXCEPT
{
    m_ThermalHandler.SetFanControlEnabled(enabled);
    NN_RESULT_SUCCESS;
}

nn::Result ThermalCoordinator::GetFanControlEnabled(bool* pOutEnabled) NN_NOEXCEPT
{
    *pOutEnabled = m_ThermalHandler.GetFanControlEnabled();
    NN_RESULT_SUCCESS;
}

TemperatureMilliC ThermalCoordinator::GetSkinTemperatureMilliC() NN_NOEXCEPT
{
    return m_SkinTemperatureMilliC;
}

ThermalCoordinator& GetThermalCoordinator() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(ThermalCoordinator, s_ThermalCoordinator);
    return s_ThermalCoordinator;
}

}}}} // namespace nn::tc::impl::detail
