﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Common.h>

#include <nn/fan/fan.h>
#include <nn/tc/tc_Types.h>
#include <nn/tc/impl/tc_PowerModeTypes.h>

#include <nn/os.h>
#include <nn/nn_SdkAssert.h>

#include "tc_AdvancedThermalHandler.h"

namespace nn { namespace tc { namespace impl { namespace detail {

namespace
{
const int Duty0Rate = 0;
const int Duty10Rate = 25; // 10 * 255 / 100
const int Duty20Rate = 51; // 20 * 255 / 100
const int Duty60Rate = 153; // 60 * 255 / 100
const int Duty100Rate = 255; // 100 * 255 / 100

// 表面温度予想は、センサの取得温度に 1 度のばらつき想定、
// 表面温度予想に 0.5 度のばらつき想定をしている
// http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=164712508
// そのため、この目標値にはバッファをあまり設けたくはないが、
// 48000 は直接的な影響はないが、48 度がギリギリすぎるということでひとまず 47500
const TemperatureMilliC HoldableThermal = 47500;
const TemperatureMilliC TouchableThermal = 60000;

struct ThermalLog
{
    int64_t           millisec;
    TemperatureMilliC skin;
    TemperatureMilliC pcb;
    TemperatureMilliC soc;
};

struct ObjectThermalMap
{
    PowerMode powerMode;
    OperatingMode operatingMode;
    TemperatureMilliC objectThermal;
    int minRate;
    int maxRate;
};

// PowerMode_FullAwake かつ OperatingMode_Console 時は別ポリシーにより、20% 以上の制御となる。
// そのポリシーが外されるときには別途検討が要る。
const ObjectThermalMap ObjectThermals[] = {
    {PowerMode_FullAwake,    nn::tc::OperatingMode_Handheld, HoldableThermal, Duty0Rate, Duty100Rate},
    {PowerMode_FullAwake,    nn::tc::OperatingMode_Console,  TouchableThermal, Duty0Rate, Duty100Rate},
    {PowerMode_MinimumAwake, nn::tc::OperatingMode_Handheld, HoldableThermal, Duty0Rate, Duty60Rate},
    {PowerMode_MinimumAwake, nn::tc::OperatingMode_Console,  HoldableThermal, Duty0Rate, Duty60Rate},
    {PowerMode_SleepReady,   nn::tc::OperatingMode_Handheld, HoldableThermal, Duty0Rate, Duty60Rate},
    {PowerMode_SleepReady,   nn::tc::OperatingMode_Console,  HoldableThermal, Duty0Rate, Duty60Rate},
};

class ThermalLogs
{
    NN_DISALLOW_COPY(ThermalLogs);

public:
    ThermalLogs() NN_NOEXCEPT {}

    void Initialize() NN_NOEXCEPT
    {
        Reset();
    }
    void Reset() NN_NOEXCEPT
    {
        m_Index = 0;
        m_IsFilled = false;
    }

    void AppendData(const ThermalLog& log) NN_NOEXCEPT
    {
        m_Logs[m_Index] = log;
        m_Index++;

        if (m_Index >= LogSize)
        {
            m_Index = 0;
            m_IsFilled = true;
        }
    }
    bool CheckDataAvailable(int64_t currentMillisec) NN_NOEXCEPT
    {
        if (m_IsFilled == false)
        {
            return false;
        }
        // データが古いものになっていたら、一度測定値を破棄する
        // ひとまず、10 秒以上測定間隔があいていたら破棄
        // あるいは、時計が戻っていても破棄
        auto latest = GetLatest();
        if (currentMillisec < latest.millisec || latest.millisec + 10000 < currentMillisec)
        {
            Reset();
        }
        return m_IsFilled;
    }
    const ThermalLog& GetLatest() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsFilled, "Thermal data not filled");

        auto latestIndex = (m_Index == 0) ? (LogSize - 1) : (m_Index - 1);
        return m_Logs[latestIndex];
    }
    const ThermalLog& GetOldest() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsFilled, "Thermal data not filled");

        return m_Logs[m_Index];
    }

private:
    static const int LogSize = 10;
    ThermalLog m_Logs[LogSize];
    int m_Index; // 次にデータを入れる場所 = 一番古いデータ
    bool m_IsFilled;
};

class AdvancedPolicy
{
    NN_DISALLOW_COPY(AdvancedPolicy);
public:
    AdvancedPolicy() NN_NOEXCEPT {}
    void Initialize() NN_NOEXCEPT
    {
        m_Logs.Initialize();
        m_CurrentRotationRate = 0;
        m_BufferedRotationDelta = 0;
        m_CurrentOperatingMode = tc::OperatingMode_Undefined;
        m_CurrentPowerMode = tc::impl::PowerMode_FullAwake;
        m_InitialTick = nn::os::GetSystemTick();
        m_FanStopped = true;
    }
    int GetRotationRate() NN_NOEXCEPT
    {
        // ファンが止まっていたら、止まった扱いにする
        if (m_CurrentRotationRate == 0)
        {
            m_FanStopped = true;
            return 0;
        }

        // 一度ファンが止まったら、10% にたどり着くまでファンは回さない
        if (m_FanStopped && m_CurrentRotationRate < Duty10Rate)
        {
            return 0;
        }
        m_FanStopped = false;

        // 1% - 19% は 20% に切り上げる
        return (m_CurrentRotationRate < Duty20Rate) ? Duty20Rate : m_CurrentRotationRate;
    }
    int64_t GetCurrentMillisec() NN_NOEXCEPT
    {
        return nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - m_InitialTick).GetMilliSeconds();
    }
    void SetOperatingMode(OperatingMode mode) NN_NOEXCEPT
    {
        m_CurrentOperatingMode = mode;
    }
    void SetPowerMode(PowerMode mode) NN_NOEXCEPT
    {
        // PowerMode_SleepReady になったら、duty は 0 にする
        if (mode == PowerMode_SleepReady)
        {
            m_CurrentRotationRate = 0;
        }
        m_CurrentPowerMode = mode;
    }
    void AppendData(const ThermalLog& log) NN_NOEXCEPT
    {
        UpdateRotationRate(log);
        m_Logs.AppendData(log);
    }

private:
    int GetDeltaByDerivative(const ThermalLog& log, const ThermalLog& latest, const ThermalLog& oldest) NN_NOEXCEPT
    {
        // 500mC でフリップ
        int latestDeltaFlipped = (log.skin / 500) - (latest.skin / 500);
        int oldestDelta = log.skin - oldest.skin;

        // 温度変化による制御
        if (latestDeltaFlipped > 0)
        {
            // 温度が上がった瞬間は、多めにあげる
            m_BufferedRotationDelta += 8;
        }
        else if (latestDeltaFlipped < 0)
        {
            // 温度が下がった瞬間は、多めに下げる
            m_BufferedRotationDelta -= 8;
        }
        else if (100 <= oldestDelta && oldestDelta < 500)
        {
            // 10sec 前と比べて温度が上がる傾向 (500mC) であれば、ちょっとずつあげる
            m_BufferedRotationDelta += 1;
        }
        else if (oldestDelta >= 500)
        {
            // 10sec 前と比べて温度が上がる傾向 (500mC 超) であれば、もう少しあげる
            m_BufferedRotationDelta += 3;
        }
        else if (oldestDelta <= -100)
        {
            // 温度が下がる傾向であれば、ちょっとずつさげる
            m_BufferedRotationDelta -= 1;
        }

        // 1sec 間の遷移量の上限を設ける
        int delta = 0;
        if (m_BufferedRotationDelta < 0)
        {
            delta = std::max(-3, m_BufferedRotationDelta);
        }
        else
        {
            delta = std::min(3, m_BufferedRotationDelta);
        }
        m_BufferedRotationDelta -= delta;

        return delta;
    }
    // 0 を返す場合、適温なので追加の処理をすることを期待している
    int GetDeltaByAbsoluteThermal(const ThermalLog& log, TemperatureMilliC objectThermal) NN_NOEXCEPT
    {
        if (log.skin < objectThermal - 5000)
        {
            // 目標温度よりも 5 度以上差がある場合、ファンを遅くする
            return -2;
        }
        else if (log.skin < objectThermal - 1000)
        {
            // 目標温度 - 1 よりも下の場合は、ちょっとずつファンを遅くする
            return -1;
        }
        else if (log.skin <= objectThermal)
        {
            // 目標温度近辺の場合、0 を返す
            return 0;
        }
        else if (log.skin <= objectThermal + 5000)
        {
            // 目標温度よりも上の場合は、ちょっとずつファンを速くする
            return 1;
        }
        else
        {
            return 2;
        }
    }

    TemperatureMilliC GetObjectThermal() NN_NOEXCEPT
    {
        for (auto& o : ObjectThermals)
        {
            if (m_CurrentPowerMode == o.powerMode && m_CurrentOperatingMode == o.operatingMode)
            {
                return o.objectThermal;
            }
        }
        return HoldableThermal;
    }

    int GetMinRate() NN_NOEXCEPT
    {
        for (auto& o : ObjectThermals)
        {
            if (m_CurrentPowerMode == o.powerMode && m_CurrentOperatingMode == o.operatingMode)
            {
                return o.minRate;
            }
        }
        return Duty0Rate;
    }

    int GetMaxRate() NN_NOEXCEPT
    {
        for (auto& o : ObjectThermals)
        {
            if (m_CurrentPowerMode == o.powerMode && m_CurrentOperatingMode == o.operatingMode)
            {
                return o.maxRate;
            }
        }
        return Duty60Rate;
    }

    void TidyRotationRate() NN_NOEXCEPT
    {
        // 0% からの立ち上がりの遅さ対策
        if (m_FanStopped && m_CurrentRotationRate >= Duty10Rate)
        {
            switch (m_CurrentOperatingMode)
            {
            case nn::tc::OperatingMode_Handheld:
            case nn::tc::OperatingMode_Console:
                m_CurrentRotationRate = Duty20Rate;
                break;
            default:
                // nothing to do
                break;
            }
        }

        m_CurrentRotationRate = std::max(m_CurrentRotationRate, GetMinRate());
        m_CurrentRotationRate = std::min(m_CurrentRotationRate, GetMaxRate());
    }

    void UpdateRotationRate(const ThermalLog& log) NN_NOEXCEPT
    {
        // 測定結果が利用可能か確認
        auto millisec = GetCurrentMillisec();
        if (m_Logs.CheckDataAvailable(millisec) == false)
        {
            return;
        }
        auto latest = m_Logs.GetLatest();
        auto oldest = m_Logs.GetOldest();

        int delta = GetDeltaByDerivative(log, latest, oldest);
        auto objectThermal = GetObjectThermal();
        int delta2 = GetDeltaByAbsoluteThermal(log, objectThermal);
        if (delta2 == 0)
        {
            // 目標温度近辺の場合、温度変化を緩くする
            delta /= 2;
        }
        else
        {
            delta += delta2;
        }
        m_CurrentRotationRate += delta;
        TidyRotationRate();
    }

    ThermalLogs m_Logs;
    int m_CurrentRotationRate;
    int m_BufferedRotationDelta;
    OperatingMode m_CurrentOperatingMode;
    PowerMode m_CurrentPowerMode;
    nn::os::Tick m_InitialTick;
    bool         m_FanStopped; // ファンが一度止まった = しばらく回さない
};

AdvancedPolicy Advanced;

}

void InitializeAdvancedPolicy() NN_NOEXCEPT
{
    Advanced.Initialize();
}

void FinalizeAdvancedPolicy() NN_NOEXCEPT
{
}

nn::fan::RotationSpeedLevel GetRotationSpeedLevelFromAdvancedPolicy(TemperatureMilliC skinTemperature, TemperatureMilliC pcbTemperature, TemperatureMilliC socTemperature) NN_NOEXCEPT
{
    ThermalLog history = {Advanced.GetCurrentMillisec(), skinTemperature, pcbTemperature, socTemperature};
    Advanced.AppendData(history);

    auto rate = Advanced.GetRotationRate();

    return static_cast<nn::fan::RotationSpeedLevel>(rate);
}

void SetOperatingModeToAdvancedPolicy(OperatingMode operatingMode) NN_NOEXCEPT
{
    Advanced.SetOperatingMode(operatingMode);
}

void SetPowerModeToAdvancedPolicy(PowerMode powerMode) NN_NOEXCEPT
{
    Advanced.SetPowerMode(powerMode);
}

}}}} // namespace nn::tc::impl::detail
