﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ソフトウェアキーボード呼び出しに関する公開ヘッダ
 */

#include <nn/swkbd/detail/swkbd_InlineKeyboardApiImpl.h>
#include <nn/la/la_Api.h>

#include <nn/swkbd/swkbd_InlinePrivateTypes.h>
#include <nn/swkbd/swkbd_Result.h>
#include <algorithm>
#include <cmath>

#include <nn/vi/vi_Result.h>
#include <nn/util/util_StringUtil.h>
#include <nn/nn_Log.h>

namespace nn { namespace swkbd {

namespace detail {

namespace {
    template <typename T>
    const T& clamp( const T& v, const T& min, const T& max )
    {
        return std::max( min, std::min( max, v ) );
    }

    //---------------------------------------------------------------------------
    //! @brief   Rect のユーティリティ
    //---------------------------------------------------------------------------
    struct RectUtil : public Rect
    {
        //---------------------------------------------------------------------------
        //! @brief   Rect の矩形に値があるかどうかを判定します。
        //---------------------------------------------------------------------------
        bool contains( int x, int y )
        {
            return ( left <= x && x <= (left + width) && top <= y && y <= (top + height) );
        }
    };
}


static const int c_MaxHeightNormalPredict = 450;
static const int c_MaxHeightNormalNoPredict = 400;
static const int c_MaxHeightNumericPredict = 400;
static const int c_MaxHeightNumericNoPredict = 350;

InlineKeyboardImpl::InlineKeyboardImpl() NN_NOEXCEPT
    : m_State( State_None )
    , m_Handle( nn::applet::InvalidLibraryAppletHandle )
    , m_FinishedInitializeCallback( nullptr )
    , m_FinishedKeyboardCallback( nullptr )
    , m_ChangedStringCallback( nullptr )
    , m_ChangedStringCallbackUtf8( nullptr )
    , m_MovedCursorCallback( nullptr )
    , m_MovedCursorCallbackUtf8( nullptr )
    , m_MovedTabCallback( nullptr )
    , m_DecidedEnterCallback( nullptr )
    , m_DecidedEnterCallbackUtf8( nullptr )
    , m_DecidedCancelCallback( nullptr )
    , m_ReleasedUserWordInfoCallback( nullptr )
    , m_KeyboardStorageInfo()
    , m_UserWordInfo()
    , m_CustomizeDicBuffer( nullptr )
    , m_CustomizeDicBufferSize( 0 )
    , m_CustomizeDicInfo()
    , m_IsUseTransferStorage( false )
    , m_IsInvalidDirectionalButtons( false )
{
}

InlineKeyboardImpl::~InlineKeyboardImpl() NN_NOEXCEPT
{
}

void
InlineKeyboardImpl::NormalizeParams() NN_NOEXCEPT
{
    if ( m_KeyboardStorageInfo.isScaleWithFooter )
    {
        this->SetFooterBgAlpha( GetKeytopBgAlpha() );
        this->SetInputModeFadeType( GetKeytopBgAlpha() == 1.f ? InputModeFadeType_None : InputModeFadeType_FadeAndInvisible );
    }
    else
    {
        this->SetInputModeFadeType( InputModeFadeType_InvisibleAndFade );
    }
}

bool
InlineKeyboardImpl::Initialize( const InitializeArg& arg ) NN_NOEXCEPT
{
    // 非同期 swkbd が起動していれば何もしない
    if( m_Handle != nn::applet::InvalidLibraryAppletHandle )
    {
        return false;
    }

    // 起動準備を開始する
    // デバイス情報など、こちらでキャッシュしたほうが良いものはキャッシュしておく
    m_KeyboardStorageInfo.initializeArg = arg;
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_InitializeArg );

    // 起動
    Launch();

    // ここまで来たら、起動開始まではいけている

    // ここまで問題なければ Success
    // TODO: Success 以外返せないのでは？
    return true;
}

nn::Result
InlineKeyboardImpl::Finalize() NN_NOEXCEPT
{
    // 非同期 swkbd が起動していなければ何もしない
    // TODO: 正しい返り値が必要では？
    //NN_SDK_REQUIRES( m_Handle != nn::applet::InvalidLibraryAppletHandle );
    if( m_Handle == nn::applet::InvalidLibraryAppletHandle )
    {
        return nn::swkbd::ResultCanceled();
    }

    // 終了コマンドを送る
    RequestCommand command = RequestCommand::RequestCommand_Finalize;
    SendRequest_( command, nullptr, 0 );

    // カスタマイズ辞書がつけっぱなら解放完了Replyを受け取って処理するまで待機
    if( m_IsUseTransferStorage )
    {
        WaitForUnsetCustomizeDic_();
    }

    // swkbd を終了する
    // TODO: 処理をブロックしないべき？
    //        disappear のみブロックしない、で良いかもしれない
    // ライブラリアプレットの終了確認
    nn::applet::JoinLibraryApplet( m_Handle );
    nn::applet::LibraryAppletExitReason exit_reason = nn::applet::GetLibraryAppletExitReason( m_Handle );

    // swkbd 終了理由を格納する変数
    // ただし、この変数は「LAが正しく終了したか」の判定にのみ使う
    // swkbd 内でキャンセル入力があったかどうかは、別で判定する
    nn::Result la_result;

    // 終了値のハンドリング
    // SAがLAを強制終了させたとき LibraryAppletExitReason_Canceledになる
    // それは許容する必要がある。
    // 異常終了のときはABORTでいいはず。
    // ポリシー：http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=164702960
    NN_ABORT_UNLESS( exit_reason != nn::applet::LibraryAppletExitReason_Abnormal );
    // Unexpectedはデバッグで撲滅する必要がある
    NN_ABORT_UNLESS( exit_reason != nn::applet::LibraryAppletExitReason_Unexpected );
    if( exit_reason == nn::applet::LibraryAppletExitReason_Canceled )
    {
        // 異常終了として扱われた場合は、念のためキャンセルとしておく
        la_result = nn::swkbd::ResultCanceled();
    }
    else
    {
        // ここまでで問題なければ、成功とする
        la_result = nn::ResultSuccess();
    }

    // これ以上 swkbd からデータを受け取る必要はない
    // swkbd を終了する
    nn::applet::CloseLibraryApplet( m_Handle );

    // MEMO: CloseLibraryApplet で無効なハンドルになっているが、中身は変わっていないため、
    //       意図的に Invalid なものにする
    m_Handle = nn::applet::InvalidLibraryAppletHandle;

    // ストレージ用キャッシュも解放する
    m_KeyboardStorageInfo.SetDefault();

    // MEMO: この時は TransferStorage も解放されているはず
    m_IsUseTransferStorage = false;

    // 終了結果を返す
    return la_result;
}



bool
InlineKeyboardImpl::Launch() NN_NOEXCEPT
{
    // 非同期 swkbd が起動していれば何もしない
    if( m_Handle != nn::applet::InvalidLibraryAppletHandle )
    {
        return false;
    }

    // LA 共通パラメータを設定する
    // 共通パラメータは独立したストレージとして扱う
    // 引数には swkbd shim のバージョンを埋め込む
    nn::la::CommonArgumentsWriter commonArg( c_CurrentMajorVersion, c_CurrentMinorVersion );

    nn::applet::LibraryAppletMode la_mode = nn::applet::LibraryAppletMode_PartialForeground;
    // 画面に表示しない場合の設定
    if( !m_KeyboardStorageInfo.initializeArg.isOverlayLayer )
    {
        la_mode = nn::applet::LibraryAppletMode_PartialForegroundWithIndirectDisplay;
    }

    // swkbd 起動準備を行う
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateLibraryApplet(
        &m_Handle,
        nn::applet::AppletId_LibraryAppletSwkbd,
        la_mode ) );

    // まず初めに LA 共通パラメータ用ストレージを送信する
    commonArg.PushToInChannel( m_Handle );

    // 次に swkbd 固有パラメータを送信する
    nn::la::PushToInChannel( m_Handle,
        &m_KeyboardStorageInfo.initializeArg, sizeof( InitializeArg ) );

    // 呼び出し元 LA に渡すパラメータの準備
    nn::la::LibraryAppletStartHookUserArg userArg;
    userArg.isExtremity = true;

    // ライブラリアプレットの起動
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::StartLibraryApplet( m_Handle ) );

    // ここまで問題なければ Success
    // TODO: Success 以外返せないのでは？
    return true;
}


void
InlineKeyboardImpl::SetVolume( float volume ) NN_NOEXCEPT
{
    if(m_KeyboardStorageInfo.volume == volume) { return; }

    // キャッシュする
    m_KeyboardStorageInfo.volume = volume;
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_Volume );
}


void
InlineKeyboardImpl::Appear( const AppearArg& arg ) NN_NOEXCEPT
{
    // キャッシュする
    m_KeyboardStorageInfo.appearArg = arg;

    if(m_IsInvalidDirectionalButtons)
    {
        m_KeyboardStorageInfo.appearArg.invalidButtonFlag |= detail::InvalidButton_DirectionalButton;
    }
    else
    {
        m_KeyboardStorageInfo.appearArg.invalidButtonFlag &= ~detail::InvalidButton_DirectionalButton;
    }
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_AppearArg );
    // Appear と Disappear は排他なので、後勝ちにする
    m_KeyboardStorageInfo.updateFlags.Reset( UpdateFlags_RequestDisappear );
}

void
InlineKeyboardImpl::Disappear() NN_NOEXCEPT
{
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_RequestDisappear );
    // Appear と Disappear は排他なので、後勝ちにする
    m_KeyboardStorageInfo.updateFlags.Reset( UpdateFlags_AppearArg );
}

void
InlineKeyboardImpl::SetInputText( const InputText& info ) NN_NOEXCEPT
{
    // Cruiserが同じ文字列を設定してコールバックが返るのを期待したコードがあるのでコメントアウト
    //if( (m_KeyboardStorageInfo.inputText.textLength == info.textLength)
    //    && (nn::util::Strncmp<char16_t>( m_KeyboardStorageInfo.inputText.text, info.text, info.textLength) == 0) )
    //{
    //    return;
    //}
    // キャッシュする
    m_KeyboardStorageInfo.inputText = info;
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_StringInfo );
}

void
InlineKeyboardImpl::SetCursorPos( int32_t cursorPos ) NN_NOEXCEPT
{
    // Cruiserが同じカーソル位置を設定してコールバックが返るのを期待したコードがあるのでコメントアウト
    //if( m_KeyboardStorageInfo.cursorPos == cursorPos ) { return; }
    // キャッシュする
    m_KeyboardStorageInfo.cursorPos = cursorPos;
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_CursorPos );
}


//---------------------------------------------------------------------------
//
bool
InlineKeyboardImpl::SetUserWordInfo( const UserWordInfo& info ) NN_NOEXCEPT
{
    // 非同期 swkbd が起動していなければ何もしない
    NN_SDK_REQUIRES( m_Handle != nn::applet::InvalidLibraryAppletHandle );
    if( m_Handle == nn::applet::InvalidLibraryAppletHandle )
    {
        return false;
    }

    // バッファが不正な場合、なにもしない
    NN_SDK_REQUIRES( info.pWorkBuf != nullptr );
    if( info.pWorkBuf == nullptr )
    {
        return false;
    }

    // 表示中は設定不可
    if((m_State != State_Disappear) && (m_State != State_None))
    {
        return false;
    }

    // キャッシュする
    m_UserWordInfo = info;

    // ユーザ単語の登録が追加なのか破棄なのかを判定する
    if( info.userWordNum == 0 || info.pUserWord == nullptr )
    {
        // ユーザ単語情報がない場合は、ユーザー辞書情報を破棄させる
        UnsetUserWordInfo();
    }
    else
    {
        RequestCommand command = RequestCommand::RequestCommand_SetUserWordInfo;

        size_t storage_size = UserWordInfo::GetRequiredWorkBufferSize( info.userWordNum );

        size_t offset = 0;

        // ラージストレージとして送信する
        nn::applet::StorageHandle large_storage_handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateLargeStorage( &large_storage_handle,
            info.pWorkBuf, storage_size, true ) );

        // 書き込む順番：コマンド → 単語数 → ユーザー辞書

        // コマンド書き込み
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage( large_storage_handle,
            offset, &command, sizeof( RequestCommand ) ) );
        offset += sizeof( RequestCommand );

        // 単語数書き込み
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage( large_storage_handle,
            offset, &( info.userWordNum ), sizeof( int32_t ) ) );
        offset += sizeof( int32_t );

        // ユーザー辞書書き込み
        // ユーザ単語が 0 だった場合は、ユーザー辞書の中身を使わなくさせる
        if( info.userWordNum != 0 )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage( large_storage_handle,
                offset, info.pUserWord, info.userWordNum * sizeof( UserWord ) ) );
        }

        // swkbd へ送信
        nn::applet::PushToInteractiveInChannel( m_Handle, large_storage_handle );

        // 辞書を使うなら、ユーザー辞書の破棄処理は棄却する
        m_KeyboardStorageInfo.updateFlags.Reset( UpdateFlags_UnsetUserWordInfo );
    }

    return true;
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetUtf8Mode( bool isUseUtf8 ) NN_NOEXCEPT
{
    if( m_KeyboardStorageInfo.isUseUtf8 == isUseUtf8 ) { return; }
    // キャッシュする
    m_KeyboardStorageInfo.isUseUtf8 = isUseUtf8;
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_Utf8Mode );
}


//---------------------------------------------------------------------------
//
bool
InlineKeyboardImpl::SetCustomizeDic( const void* pBuffer, size_t size, const CustomizeDicInfo& info ) NN_NOEXCEPT
{
    // 非同期 swkbd が起動していなければ何もしない
    NN_SDK_REQUIRES( m_Handle != nn::applet::InvalidLibraryAppletHandle );
    if( m_Handle == nn::applet::InvalidLibraryAppletHandle )
    {
        return false;
    }

    // バッファが不正な場合も何もしない
    NN_SDK_REQUIRES(pBuffer != nullptr );
    if(pBuffer == nullptr )
    {
        return false;
    }

    // カスタマイズ辞書を使用中の場合は何もしない
    if( m_IsUseTransferStorage )
    {
        return false;
    }

    // 表示中は設定不可
    if((m_State != State_Disappear) && (m_State != State_None))
    {
        return false;
    }

    // キャッシュする
    m_CustomizeDicBuffer = const_cast< void* >(pBuffer);
    m_CustomizeDicBufferSize = size;
    m_CustomizeDicInfo = info;

    // 共有ストレージとして送信する
    nn::applet::StorageHandle transfer_handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateTransferStorage(
        &transfer_handle, m_CustomizeDicBuffer, size ) );

    // 共有メモリなので追加書き込みはなし

    // swkbd へ送信
    nn::applet::PushToInteractiveInChannel( m_Handle, transfer_handle );

    // 加えて、オフセット情報も送り出す
    RequestCommand command = RequestCommand::RequestCommand_SetCustomizeDic;
    SendRequest_( command, &info, sizeof( CustomizeDicInfo ) );

    // 送信を終えたら、カスタマイズ辞書を使っているものとしてロックする
    m_IsUseTransferStorage = true;
    return true;
}


//---------------------------------------------------------------------------
//
bool
InlineKeyboardImpl::UnsetCustomizeDic() NN_NOEXCEPT
{
    if( (m_State != State_Disappear) && (m_State != State_None) ) { return false; }
    if(m_CustomizeDicBuffer == nullptr) { return false; }

    // キャッシュを破棄する
    m_CustomizeDicBuffer = nullptr;
    m_CustomizeDicBufferSize = 0;
    //m_CustomizeDicInfo = info;
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_UnsetCustomizeDic );
    return true;
}


//---------------------------------------------------------------------------
//
bool
InlineKeyboardImpl::UnsetUserWordInfo() NN_NOEXCEPT
{
    if((m_State != State_Disappear) && (m_State != State_None)) { return false; }
    if(m_UserWordInfo.userWordNum == 0) { return false; }

    m_UserWordInfo.userWordNum = 0; // GetMaxHeight で参照するため、先にクリアしておく
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_UnsetUserWordInfo );
    return true;
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetKeytopBgAlpha( float alpha ) NN_NOEXCEPT
{
    float normalizeAlpha = clamp( alpha, 0.f, 1.f );
    if ( m_KeyboardStorageInfo.keytopBgAlpha != normalizeAlpha )
    {
        m_KeyboardStorageInfo.keytopBgAlpha = normalizeAlpha;
        m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_BgAlpha );
    }
}

//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetFooterBgAlpha( float alpha ) NN_NOEXCEPT
{
    float normalizeAlpha = clamp( alpha, 0.f, 1.f );
    if ( m_KeyboardStorageInfo.footerBgAlpha != normalizeAlpha )
    {
        m_KeyboardStorageInfo.footerBgAlpha = normalizeAlpha;
        m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_BgAlpha );
    }
}

//---------------------------------------------------------------------------
void
InlineKeyboardImpl::SetKeytopScale( float scaleX, float scaleY ) NN_NOEXCEPT
{
    if(    (m_KeyboardStorageInfo.keytopScaleX == scaleX)
        && (m_KeyboardStorageInfo.keytopScaleY == scaleY) ) { return; }
    m_KeyboardStorageInfo.keytopScaleX = scaleX;
    m_KeyboardStorageInfo.keytopScaleY = scaleY;

    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_ScaleTrans );
}

//---------------------------------------------------------------------------
void
InlineKeyboardImpl::SetKeytopTranslate( float transX, float transY ) NN_NOEXCEPT
{
    if(    (m_KeyboardStorageInfo.keytopTranslateX == transX)
        && (m_KeyboardStorageInfo.keytopTranslateY == transY) ) { return; }
    m_KeyboardStorageInfo.keytopTranslateX = transX;
    m_KeyboardStorageInfo.keytopTranslateY = transY;

    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_ScaleTrans );
}

//---------------------------------------------------------------------------
void
InlineKeyboardImpl::SetKeytopAsFloating( bool isFloating ) NN_NOEXCEPT
{
    if( m_KeyboardStorageInfo.isFloatingKeytop == isFloating ) { return; }
    m_KeyboardStorageInfo.isFloatingKeytop = isFloating;

    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_ScaleTrans );
}

//---------------------------------------------------------------------------
void
InlineKeyboardImpl::SetBalloonScale( float scale ) NN_NOEXCEPT
{
    if( m_KeyboardStorageInfo.baloonScale == scale ) { return; }
    m_KeyboardStorageInfo.baloonScale = scale;
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_ScaleTrans );
}

//---------------------------------------------------------------------------
void
InlineKeyboardImpl::SetFooterScalable( bool footerScalable ) NN_NOEXCEPT
{
    if( m_KeyboardStorageInfo.isScaleWithFooter == footerScalable ) { return; }
    m_KeyboardStorageInfo.isScaleWithFooter = footerScalable;
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_ScaleTrans );
}

//---------------------------------------------------------------------------
void
InlineKeyboardImpl::SetAlphaEnabledInInputMode( bool enabled ) NN_NOEXCEPT
{
    if( m_KeyboardStorageInfo.enableAlphaInInputMode == enabled ) { return; }
    m_KeyboardStorageInfo.enableAlphaInInputMode = enabled;
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_BgAlpha );
}

//---------------------------------------------------------------------------
void
InlineKeyboardImpl::SetInputModeFadeType( uint8_t type ) NN_NOEXCEPT
{
    if( m_KeyboardStorageInfo.inputModeFadeType != type )
    {
        m_KeyboardStorageInfo.inputModeFadeType = type;
        m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_BgAlpha );
    }
}

//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetDisableTouch( bool disable ) NN_NOEXCEPT
{
    if( m_KeyboardStorageInfo.disableTouch == disable ) { return; }
    m_KeyboardStorageInfo.disableTouch = disable;
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_ScaleTrans );
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetDisableUSBKeyboard( bool disable ) NN_NOEXCEPT
{
    if( m_KeyboardStorageInfo.disableHwkbd == disable ) { return; }
    m_KeyboardStorageInfo.disableHwkbd = disable;
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_EnableHWKBD );
}


//---------------------------------------------------------------------------
//
int
InlineKeyboardImpl::GetTouchRectangles( Rect* primary, Rect* secondary ) const NN_NOEXCEPT
{
    const int c_ScreenHeight   = 720;
    const int c_ScreenWidth    = 1280;
    const int c_FooterHeight   = 72;
    const int c_FloatingFrameHeight = 14;

    float mainHeight = GetMaxHeight();
    float mainWidth  = c_ScreenWidth;

    int scaleCenterX = c_ScreenWidth / 2;
    int scaleCenterY = c_ScreenHeight / 2;
    float mainTopOffset = mainHeight - scaleCenterY;

    if ( ! m_KeyboardStorageInfo.isScaleWithFooter )
    {
        mainHeight -= c_FooterHeight;

        if ( m_KeyboardStorageInfo.isFloatingKeytop )
        {
            mainHeight += c_FloatingFrameHeight;
        }
    }

    mainWidth  *= m_KeyboardStorageInfo.keytopScaleX;
    mainHeight *= m_KeyboardStorageInfo.keytopScaleY;
    mainTopOffset *= m_KeyboardStorageInfo.keytopScaleY;

    if ( primary )
    {
        float left = scaleCenterX - mainWidth / 2 + c_ScreenWidth * m_KeyboardStorageInfo.keytopTranslateX;
        float top  = scaleCenterY - mainTopOffset - c_ScreenHeight * m_KeyboardStorageInfo.keytopTranslateY;

        primary->left = static_cast<int16_t>( std::lroundf(left) );
        primary->top  = static_cast<int16_t>( std::lroundf(top) );
        primary->width  = static_cast<int16_t>( std::lroundf(mainWidth) );
        primary->height = static_cast<int16_t>( std::lroundf(mainHeight) );
    }

    if ( secondary )
    {
        if ( m_KeyboardStorageInfo.isScaleWithFooter )
        {
            secondary->left   = 0;
            secondary->top    = 0;
            secondary->width  = 0;
            secondary->height = 0;
        }
        else
        {
            secondary->left = 0;
            secondary->top = c_ScreenHeight - c_FooterHeight;
            secondary->width = c_ScreenWidth;
            secondary->height = c_FooterHeight;
        }
    }

    return m_KeyboardStorageInfo.isScaleWithFooter ? 1 : 2;
}

//---------------------------------------------------------------------------
//
State
InlineKeyboardImpl::Calc() NN_NOEXCEPT
{
    // ハンドルが不正だったら状態は未初期化なものを返す
    if( m_Handle == nn::applet::InvalidLibraryAppletHandle )
    {
        return State_None;
    }

    // swkbd が終了していたら強制的に終了パスを呼ぶ
    if( nn::os::TryWaitSystemEvent( nn::applet::GetLibraryAppletExitEvent( m_Handle ) ) )
    {
        FinishApplet_();

        // 終了状態を返す
        return State_None;
    }

    // ストレージ送信要求が来ていたら、送信する
    if( m_KeyboardStorageInfo.IsChanged() )
    {
        // ストレージの送信
        // この時点でハンドルの存在は保障されている
        RequestCommand command = RequestCommand::RequestCommand_SetKeyboardStrorageInfo;
        SendRequest_( command, &m_KeyboardStorageInfo, sizeof( KeyboardStorageInfo ) );

        // 送信後はフラグのみデフォルト値に戻す
        m_KeyboardStorageInfo.updateFlags.Reset();
    }

    // 以下、swkbd からのデータ受け取り対応
    // - nn::applet::TryPopFromInteractiveOutChannel を呼んで、ストレージがあるかどうか確認
    // - ストレージがあった場合は中身を解析
    // - swkbd の状態を取得して変数に格納
    // - 先頭にある shim に対する要求コマンドを解析
    // - 要求結果に合わせ、必要なコールバック関数を呼ぶ
    nn::applet::StorageHandle handle;

    while( nn::applet::TryPopFromInteractiveOutChannel( &handle, m_Handle ) )
    {
        // ここに来た場合は、インタラクティブ出力チャンネルが存在したことになる

        // 返信パラメータ格納用変数の準備
        nn::swkbd::ReplyType reply_type;

        // ヘッダを読み込む
        size_t offset = 0;
        // 状態の更新
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage(
            handle, offset, &( m_State ), sizeof( nn::swkbd::State ) ) );
        offset += sizeof( nn::swkbd::State );
        // 返答タイプの更新
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage(
            handle, offset, &( reply_type ), sizeof( nn::swkbd::ReplyType ) ) );
        offset += sizeof( nn::swkbd::ReplyType );

        // ReplyType によって受け取る内容の変更
        CalcFromReplyType_( reply_type, handle, offset );
    }

    return m_State;
}

//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::WaitForUnsetCustomizeDic_()  NN_NOEXCEPT
{
    nn::swkbd::ReplyType reply_type;
    nn::applet::StorageHandle handle;
    int retry = 0;
    while(retry < 10)// 10回リトライ（１回に付き100ms待機）
    {
        if( ! nn::applet::TryPopFromInteractiveOutChannel(&handle, m_Handle) )
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
            retry++;
            continue;
        }
        // ここに来た場合は、インタラクティブ出力チャンネルが存在したことになる

        size_t offset = 0;
        // 状態の更新
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::ReadFromStorage(
            handle, offset, &(m_State), sizeof(nn::swkbd::State)));
        offset += sizeof(nn::swkbd::State);
        // 返答タイプの更新
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::ReadFromStorage(
            handle, offset, &(reply_type), sizeof(nn::swkbd::ReplyType)));
        offset += sizeof(nn::swkbd::ReplyType);

        if( reply_type == ReplyType_UnsetCustomizeDic )
        {// カスタマイズ辞書解放イベントがきたので終了
            CalcUnsetCustomizeDic_(handle, offset);
            break;
        }
        else
        {
            CalcFromReplyType_(reply_type, handle, offset);
        }
    }
    return;
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::CalcFromReplyType_( nn::swkbd::ReplyType replyType,
    nn::applet::StorageHandle handle,
    size_t offset ) NN_NOEXCEPT
{
    switch( replyType )
    {
    case ReplyType_FinishedInitialize:
    {
        // 初期化終了イベント
        CalcFinishedInitialize_( handle, offset );
        break;
    }
    case ReplyType_ChangedState:
    {
        // 状態変更イベント
        CalcChangedState_( handle, offset );
        break;
    }
    case ReplyType_ChangedString:
    {
        // 文字列変更イベント
        CalcChangedString_( handle, offset );
        break;
    }
    case ReplyType_MovedCursor:
    {
        // カーソル移動イベント
        CalcMovedCursor_( handle, offset );
        break;
    }
    case ReplyType_MovedTab:
    {
        // タブ移動イベント(2.0.0NUPでは非対応)
        CalcMovedTab_( handle, offset );
        break;
    }
    case ReplyType_DecidedEnter:
    {
        // 確定閉じイベント
        CalcDecidedEnter_( handle, offset );
        break;
    }
    case ReplyType_DecidedCancel:
    {
        // キャンセル閉じイベント
        CalcDecidedCancel_( handle, offset );
        break;
    }
    case ReplyType_ChangedStringUtf8:
    {
        // 文字列変更イベント(UTF-8)
        CalcChangedStringUtf8_( handle, offset );
        break;
    }
    case ReplyType_MovedCursorUtf8:
    {
        // カーソル移動イベント(UTF-8)
        CalcMovedCursorUtf8_( handle, offset );
        break;
    }
    case ReplyType_DecidedEnterUtf8:
    {
        // 確定閉じイベント(UTF-8)
        CalcDecidedEnterUtf8_( handle, offset );
        break;
    }
    case ReplyType_UnsetCustomizeDic:
    {
        // カスタマイズ辞書解放イベント
        CalcUnsetCustomizeDic_( handle, offset );
        break;
    }
    case ReplyType_ReleasedUserWordInfo:
    {
        // ユーザー辞書解放イベント
        CalcUnsetUserWordInfo_( handle, offset );
        break;
    }
    default:
    {
        // 想定外のメッセージ
        // 特に何もせず、ストレージを解放する
        nn::applet::ReleaseStorage( handle );
        break;
    }
    }
}


//---------------------------------------------------------------------------
//
int
InlineKeyboardImpl::GetMaxHeight() const NN_NOEXCEPT
{
    if( m_KeyboardStorageInfo.appearArg.keyboardMode != KeyboardMode_Numeric )
    {
        if( m_KeyboardStorageInfo.appearArg.isPredictionEnabled ||
            m_UserWordInfo.userWordNum != 0 ||
            m_CustomizeDicBufferSize != 0 )
        {
            // 非Numeric + 予測変換有効時
            return c_MaxHeightNormalPredict;
        }
        else
        {
            return c_MaxHeightNormalNoPredict;
        }
    }
    else
    {
        // Numeric
        if( m_KeyboardStorageInfo.appearArg.isPredictionEnabled ||
            m_UserWordInfo.userWordNum != 0 ||
            m_CustomizeDicBufferSize != 0)
        {
            // Numeric + 予測変換有効時
            return c_MaxHeightNumericPredict;
        }
        else
        {
            return c_MaxHeightNumericNoPredict;
        }
    }
}


//---------------------------------------------------------------------------
//
bool
InlineKeyboardImpl::IsUsedTouchPointByKeyboard( int32_t x, int32_t y ) NN_NOEXCEPT
{
    RectUtil primary, secondary;

    int touch_num = this->GetTouchRectangles( &primary, &secondary );

    if ( primary.contains( x, y ) ) { return true; }
    else if ( touch_num >= 2 && secondary.contains( x, y ) ) { return true; }

    return false;
}

//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetDirectionalButtonAssignEnabled( bool enabled ) NN_NOEXCEPT
{
    m_IsInvalidDirectionalButtons = ! enabled; // まだAppearされていないときはこのフラグを見て invalidButtonFlagを調整する
    if(enabled)
    {
        m_KeyboardStorageInfo.appearArg.invalidButtonFlag &= ~detail::InvalidButton_DirectionalButton;
    }
    else
    {
        m_KeyboardStorageInfo.appearArg.invalidButtonFlag |= detail::InvalidButton_DirectionalButton;
    }

    // 動的に変更を通知
    m_KeyboardStorageInfo.updateFlags.Set(UpdateFlags_InvalidButton);
}

//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::EnableSe( SeGroup group, bool enable ) NN_NOEXCEPT
{
    m_KeyboardStorageInfo.targetSe = group;
    m_KeyboardStorageInfo.updateFlags.Set( enable ? UpdateFlags_EnableSe : UpdateFlags_DisableSe );
}

//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::EnableBackspace( bool enable ) NN_NOEXCEPT
{
    m_KeyboardStorageInfo.isValidBackspace = enable;
    m_KeyboardStorageInfo.updateFlags.Set( UpdateFlags_EnableBackspace );
}

//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetFinishedInitializeCallback( FinishedInitializeCallback pCallback ) NN_NOEXCEPT
{
    if( pCallback == nullptr )
    {
        m_FinishedInitializeCallback = nullptr;
        return;
    }

    m_FinishedInitializeCallback = pCallback;
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetFinishedKeyboardCallback( FinishedKeyboardCallback pCallback ) NN_NOEXCEPT
{
    if( pCallback == nullptr )
    {
        m_FinishedKeyboardCallback = nullptr;
        return;
    }

    m_FinishedKeyboardCallback = pCallback;
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetChangedStringCallback( ChangedStringCallback pCallback ) NN_NOEXCEPT
{
    if( pCallback == nullptr )
    {
        m_ChangedStringCallback = nullptr;
        return;
    }

    m_ChangedStringCallback = pCallback;
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetChangedStringCallbackUtf8( ChangedStringCallbackUtf8 pCallback ) NN_NOEXCEPT
{
    if( pCallback == nullptr )
    {
        m_ChangedStringCallbackUtf8 = nullptr;
        return;
    }

    m_ChangedStringCallbackUtf8 = pCallback;
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetMovedCursorCallback( MovedCursorCallback pCallback ) NN_NOEXCEPT
{
    if( pCallback == nullptr )
    {
        m_MovedCursorCallback = nullptr;
        return;
    }

    m_MovedCursorCallback = pCallback;
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetMovedCursorCallbackUtf8( MovedCursorCallbackUtf8 pCallback ) NN_NOEXCEPT
{
    if( pCallback == nullptr )
    {
        m_MovedCursorCallbackUtf8 = nullptr;
        return;
    }

    m_MovedCursorCallbackUtf8 = pCallback;
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetMovedTabCallback( MovedTabCallback pCallback ) NN_NOEXCEPT
{
    if( pCallback == nullptr )
    {
        m_MovedTabCallback = nullptr;
        return;
    }

    m_MovedTabCallback = pCallback;
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetDecidedEnterCallback( DecidedEnterCallback pCallback ) NN_NOEXCEPT
{
    if( pCallback == nullptr )
    {
        m_DecidedEnterCallback = nullptr;
        return;
    }

    m_DecidedEnterCallback = pCallback;
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetDecidedEnterCallbackUtf8( DecidedEnterCallbackUtf8 pCallback ) NN_NOEXCEPT
{
    if( pCallback == nullptr )
    {
        m_DecidedEnterCallbackUtf8 = nullptr;
        return;
    }

    m_DecidedEnterCallbackUtf8 = pCallback;
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetDecidedCancelCallback( DecidedCancelCallback pCallback ) NN_NOEXCEPT
{
    if( pCallback == nullptr )
    {
        m_DecidedCancelCallback = nullptr;
        return;
    }

    m_DecidedCancelCallback = pCallback;
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::SetReleasedUserWordInfoCallback( ReleasedUserWordInfoCallback pCallback ) NN_NOEXCEPT
{
    if( pCallback == nullptr )
    {
        m_ReleasedUserWordInfoCallback = nullptr;
        return;
    }

    m_ReleasedUserWordInfoCallback = pCallback;
}


//---------------------------------------------------------------------------
//
nn::Result
InlineKeyboardImpl::SendRequest_( RequestCommand command, const void* pRequestCommandBuffer, size_t requestCommandSize ) NN_NOEXCEPT
{
    // 非同期 swkbd が起動していなければ何もしない
    // TODO: 正しい返り値が必要では？
    NN_SDK_REQUIRES( m_Handle != nn::applet::InvalidLibraryAppletHandle );
    if( m_Handle == nn::applet::InvalidLibraryAppletHandle )
    {
        return nn::swkbd::ResultCanceled();
    }

    // ライブラリアプレットに対して動的にメッセージを送る
    // IPC コマンドを送信する
    {
        nn::applet::StorageHandle storage_handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateStorage( &storage_handle, sizeof( RequestCommand ) + requestCommandSize ) );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage( storage_handle, 0, &command, sizeof( RequestCommand ) ) );
        if( pRequestCommandBuffer != nullptr )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage( storage_handle, sizeof( RequestCommand ), pRequestCommandBuffer, requestCommandSize ) );
        }
        nn::applet::PushToInteractiveInChannel( m_Handle, storage_handle );
    }

    // ここまで問題なければ成功とする
    return nn::ResultSuccess();
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::CalcFinishedInitialize_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT
{
    nn::swkbd::FinishedInitializeArg arg;

    // データがあるなら読み込む
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage(
        handle, offset, &( arg ), sizeof( nn::swkbd::FinishedInitializeArg ) ) );

    // 取り込み終わったのでストレージを解放する
    nn::applet::ReleaseStorage( handle );

    // コールバック関数を投げる
    if( m_FinishedInitializeCallback )
    {
        m_FinishedInitializeCallback();
    }
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::CalcChangedState_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT
{
    // 状態は共通ヘッダに格納されているので、
    // 基本的に追加データの読み込みは行わない

    // 取り込み終わったのでストレージを解放する
    nn::applet::ReleaseStorage( handle );

    // 状態は return で返すので、
    // 特にコールバックも行わない
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::CalcChangedString_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT
{
    nn::swkbd::ChangedStringArg arg;

    // データがあるなら読み込む
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage(
        handle, offset, &( arg ), sizeof( nn::swkbd::ChangedStringArg ) ) );

    // 取り込み終わったのでストレージを解放する
    nn::applet::ReleaseStorage( handle );

    // コールバック関数を投げる
    if( m_ChangedStringCallback )
    {
        m_ChangedStringCallback( &arg );
    }
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::CalcChangedStringUtf8_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT
{
    nn::swkbd::ChangedStringArgUtf8 arg;

    // データがあるなら読み込む
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage(
        handle, offset, &( arg ), sizeof( nn::swkbd::ChangedStringArgUtf8 ) ) );

    // 取り込み終わったのでストレージを解放する
    nn::applet::ReleaseStorage( handle );

    // コールバック関数を投げる
    if( m_ChangedStringCallbackUtf8 )
    {
        m_ChangedStringCallbackUtf8( &arg );
    }
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::CalcMovedCursor_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT
{
    nn::swkbd::MovedCursorArg arg;

    // データがあるなら読み込む
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage(
        handle, offset, &( arg ), sizeof( nn::swkbd::MovedCursorArg ) ) );

    // 取り込み終わったのでストレージを解放する
    nn::applet::ReleaseStorage( handle );

    // コールバック関数を投げる
    if( m_MovedCursorCallback )
    {
        m_MovedCursorCallback( &arg );
    }
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::CalcMovedCursorUtf8_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT
{
    nn::swkbd::MovedCursorArgUtf8 arg;

    // データがあるなら読み込む
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage(
        handle, offset, &( arg ), sizeof( nn::swkbd::MovedCursorArgUtf8 ) ) );

    // 取り込み終わったのでストレージを解放する
    nn::applet::ReleaseStorage( handle );

    // コールバック関数を投げる
    if( m_MovedCursorCallbackUtf8 )
    {
        m_MovedCursorCallbackUtf8( &arg );
    }
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::CalcMovedTab_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT
{
    nn::swkbd::MovedTabArg arg;

    // データがあるなら読み込む
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage(
        handle, offset, &( arg ), sizeof( nn::swkbd::MovedTabArg ) ) );

    // 取り込み終わったのでストレージを解放する
    nn::applet::ReleaseStorage( handle );

    // コールバック関数を投げる
    if( m_MovedTabCallback )
    {
        m_MovedTabCallback( &arg );
    }
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::CalcDecidedEnter_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT
{
    nn::swkbd::DecidedEnterArg arg;

    // データがあるなら読み込む
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage(
        handle, offset, &( arg ), sizeof( nn::swkbd::DecidedEnterArg ) ) );

    // 取り込み終わったのでストレージを解放する
    nn::applet::ReleaseStorage( handle );

    // コールバック関数を投げる
    if( m_DecidedEnterCallback )
    {
        m_DecidedEnterCallback( &arg );
    }
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::CalcDecidedEnterUtf8_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT
{
    nn::swkbd::DecidedEnterArgUtf8 arg;

    // データがあるなら読み込む
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage(
        handle, offset, &( arg ), sizeof( nn::swkbd::DecidedEnterArgUtf8 ) ) );

    // 取り込み終わったのでストレージを解放する
    nn::applet::ReleaseStorage( handle );

    // コールバック関数を投げる
    if( m_DecidedEnterCallbackUtf8 )
    {
        m_DecidedEnterCallbackUtf8( &arg );
    }
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::CalcDecidedCancel_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT
{
    // 取り込み終わったのでストレージを解放する
    nn::applet::ReleaseStorage( handle );

    // コールバック関数を投げる
    if( m_DecidedCancelCallback )
    {
        m_DecidedCancelCallback();
    }
}



//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::CalcUnsetCustomizeDic_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT
{
    // カスタマイズ辞書のセットを可能にする
    m_IsUseTransferStorage = false;

    // 取り込み終わったのでストレージを解放する
    nn::applet::ReleaseStorage( handle );
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::CalcUnsetUserWordInfo_( nn::applet::StorageHandle handle, size_t offset ) NN_NOEXCEPT
{
    // 取り込み終わったのでストレージを解放する
    nn::applet::ReleaseStorage( handle );

    // コールバック関数を投げる
    if( m_ReleasedUserWordInfoCallback )
    {
        m_ReleasedUserWordInfoCallback();
    }
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::GetWindowSize( int* pWidth, int* pHeight ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pWidth != nullptr );
    NN_SDK_REQUIRES( pHeight != nullptr );

    //  画面サイズ指定
    // TODO:   画面サイズに応じたバッファサイズとアラインをあらかじめ計算しておくのが良い
    *pWidth = 1280;
    *pHeight = 720;
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::GetImageMemoryRequirement( size_t* pOutRequiredSize,
    size_t* pOutRequiredAlignment ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pOutRequiredSize != nullptr );
    NN_SDK_REQUIRES( pOutRequiredAlignment != nullptr );

    // 画面サイズ取得
    int width;
    int height;
    GetWindowSize( &width, &height );

    size_t size;
    size_t alignment;
    // 必要なバッファサイズとアライメントの制約を取得
    auto r = nn::vi::GetIndirectImageMemoryRequirement( &size, &alignment, width, height );

    // 取得結果の反映
    if( r.IsSuccess() )
    {
        *pOutRequiredSize = size;
        *pOutRequiredAlignment = alignment;
    }
    else
    {
        *pOutRequiredSize = 0;
        *pOutRequiredAlignment = 0;
    }
}


//---------------------------------------------------------------------------
//
bool
InlineKeyboardImpl::GetImage( void* pBuffer, size_t bufferSize ) NN_NOEXCEPT
{
    // 非同期 swkbd が起動していなければ何もしない
    if( m_Handle == nn::applet::InvalidLibraryAppletHandle )
    {
        return false;
    }

    // バッファに何も指定されていなければ無視する
    NN_SDK_REQUIRES( pBuffer != nullptr );
    if( pBuffer == nullptr )
    {
        return false;
    }

    // swkbd が終了していたら強制的に終了パスを呼ぶ
    if( nn::os::TryWaitSystemEvent( nn::applet::GetLibraryAppletExitEvent( m_Handle ) ) )
    {
        FinishApplet_();

        // 終了状態を返す
        return false;
    }

    // 画面サイズ取得
    int width;
    int height;
    GetWindowSize( &width, &height );

    // 要求サイズを内部でも取得しておく
    size_t size;
    size_t alignment;
    GetImageMemoryRequirement( &size, &alignment );

    // バッファサイズが正しいものを指定していなければ、終了させる
    if( size != bufferSize )
    {
        return false;
    }

    // バッファが正しくアライメントされているかどうかチェックする
    if ( ! nn::util::is_aligned( reinterpret_cast<uintptr_t>(pBuffer), alignment ) )
    {
        return false;
    }

    // IndirectLayer の ConsumerHandle を取得
    auto indirectLayerConsumerHandle = nn::applet::GetIndirectLayerConsumerHandle( m_Handle );

    size_t outSize;
    size_t stride;

    // 画像取得
    auto r = nn::vi::GetIndirectImage( &outSize, &stride,
        pBuffer,
        size, indirectLayerConsumerHandle,
        width, height );

    if( r.IsSuccess() )
    {
        return true;
    }
    else if( nn::vi::ResultNotReady::Includes( r ) )
    {
        // アプリが渡したバッファも壊れないため、ABORT にはせず、false で返す
        return false;
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( r );
        return false;
    }
}


//---------------------------------------------------------------------------
//
void
InlineKeyboardImpl::FinishApplet_() NN_NOEXCEPT
{
    // 念のため仮待機してみる
    nn::applet::JoinLibraryApplet( m_Handle );

    // swkbd を終了する
    nn::applet::CloseLibraryApplet( m_Handle );

    // ハンドルを無効化しておく
    m_Handle = nn::applet::InvalidLibraryAppletHandle;

    // ストレージ用キャッシュもフラグのみ解放する
    m_KeyboardStorageInfo.updateFlags.Reset();

    // この時は TransferStorage も使用可能になっているはず
    m_IsUseTransferStorage = false;

    // swkbd 終了コールバックを投げる
    if( m_FinishedKeyboardCallback )
    {
        m_FinishedKeyboardCallback();
    }
}


} // namespace detail

}} // namespace nn::swkbd

