﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/usb/usb_Host.h>
#include <nn/usb/usb_Result.public.h>
#include "stdfu_DfuFile.h"

namespace nn {
namespace stdfu {

enum DfuAttributes
{
    DfuAttributes_DownloadCapableBit        = 0x01,
    DfuAttributes_UploadCapableBit          = 0x02,
    DfuAttributes_ManifestationToerantBit   = 0x04,
};

struct DfuDescriptor
{
    uint8_t     bLength;
    uint8_t     bDescriptorType;
    uint8_t     bmAttributes;
    uint16_t    wDetachTimeout;
    uint16_t    wTransferSize;
    uint16_t    bcdDfuVersion;
} __attribute__((__packed__));


enum DfuStateCode
{
    DfuStateCode_Idle                       = 0x00,
    DfuStateCode_Detach                     = 0x01,
    DfuStateCode_DfuIdle                    = 0x02,
    DfuStateCode_DfuDownloadSync            = 0x03,
    DfuStateCode_DfuDownloadBusy            = 0x04,
    DfuStateCode_DfuDownloadIdle            = 0x05,
    DfuStateCode_DfuManifestSync            = 0x06,
    DfuStateCode_DfuManifest                = 0x07,
    DfuStateCode_DfuManifestWaitReset       = 0x08,
    DfuStateCode_DfuUploadIdle              = 0x09,
    DfuStateCode_DfuError                   = 0x0a,
    DfuStateCode_DfuUploadSync              = 0x91,
    DfuStateCode_DfuUploadBusy              = 0x92,
};


enum DfuStatusCode
{
    DfuStatusCode_Ok                        = 0x00,
    DfuStatusCode_ErrTarget                 = 0x01,
    DfuStatusCode_ErrFile                   = 0x02,
    DfuStatusCode_ErrWrite                  = 0x03,
    DfuStatusCode_ErrErase                  = 0x04,
    DfuStatusCode_ErrCheckErase             = 0x05,
    DfuStatusCode_ErrProg                   = 0x06,
    DfuStatusCode_ErrVerify                 = 0x07,
    DfuStatusCode_ErrAddress                = 0x08,
    DfuStatusCode_ErrNotDone                = 0x09,
    DfuStatusCode_ErrFirmware               = 0x0a,
    DfuStatusCode_ErrVendor                 = 0x0b,
    DfuStatusCode_ErrUsbr                   = 0x0c,
    DfuStatusCode_ErrPor                    = 0x0d,
    DfuStatusCode_ErrUnknown                = 0x0e,
    DfuStatusCode_ErrStalledPkt             = 0x0f,
};


struct DfuStatus
{
    uint8_t     bStatus;
    uint8_t     bwPollTieout[3];
    uint8_t     bState;
    uint8_t     iString;
} __attribute__((__packed__));


class DfuDevice
{
public:

    Result PerformUpdate(uint16_t vid, uint16_t pid, uint16_t bcd, DfuFile *pDfuFile, uint32_t imageCount, DfuUpdateStatus *pDfuUpdateStatus) NN_NOEXCEPT;

private:

    nn::usb::Host               m_UsbHost;
    nn::usb::HostInterface      m_UsbHostInterface;
    uint32_t                    m_Handle;
    DfuDescriptor               m_DfuDescriptor;

    DfuTargetPrefix            *m_pDfuTargetPrefix;
    uint8_t                     m_AlternateSetting;
    uint8_t                    *m_pImageName;
    uint32_t                    m_ElementsCount;

    NN_ALIGNAS(4096) uint8_t    m_TransferBuffer[4096];

    Result GetDfuDevice(uint16_t vid, uint16_t pid, uint16_t bcd) NN_NOEXCEPT;

    // For update processing
    Result GetDfuDescriptor() NN_NOEXCEPT;
    Result DownloadElementAddresses(DfuFile *pDfuFile, DfuUpdateStatus *pDfuUpdateStatus) NN_NOEXCEPT;
    Result DownloadElements(DfuFile *pDfuFile, DfuUpdateStatus *pDfuUpdateStatus) NN_NOEXCEPT;
    Result VerifyElements(DfuFile *pDfuFile, DfuUpdateStatus *pDfuUpdateStatus) NN_NOEXCEPT;
    Result EndDfuDownload() NN_NOEXCEPT;
    Result PollForStatus(uint8_t bStatus, uint8_t bState, uint32_t retries) NN_NOEXCEPT;

    // DFU commands
    Result GetStaus(DfuStatus *pDfuStatus) NN_NOEXCEPT;
    Result GetState(uint8_t *pState) NN_NOEXCEPT;
    Result Download(uint8_t *pData, uint32_t bytes, uint16_t blockNumber) NN_NOEXCEPT;
    Result Upload(uint8_t *pData, uint32_t bytes, uint16_t blockNumber) NN_NOEXCEPT;
    Result Abort() NN_NOEXCEPT;

    Result ControlRequest(
                            void      *pData,
                            uint8_t    bmRequestType,
                            uint8_t    bRequest,
                            uint16_t   wValue,
                            uint16_t   wIndex,
                            uint16_t   wLength
                            ) NN_NOEXCEPT;
};

} // namespace stdfu
} // namespace nn
