﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/stdfu/stdfu_Result.public.h>
#include "../detail/stdfu_Log.h"
#include "stdfu_AhidDevice.h"

namespace nn {
namespace stdfu {
//////////////////////////////////////////////////////////////////////////////
// Public
//////////////////////////////////////////////////////////////////////////////
Result AhidDevice::ChangeToDfuDevice(nn::ahid::Ahid *pAhid) NN_NOEXCEPT
{
    Result result;
    uint32_t bytesTransferred;
    uint8_t data[DfuIntrruptDataSize];

    memset(data, 0, sizeof(data));
    data[0] = 0x80;   // report id
    data[1] = 0x08;   // Which side of KUINA does the controller be connecting to

    result = pAhid->Write(&bytesTransferred, data, sizeof(data));

    if (result.IsSuccess())
    {
        result = pAhid->Read(&bytesTransferred, data, sizeof(data));

        if (result.IsSuccess())
        {
            memset(data, 0, sizeof(data));
            data[0] = 0x80;   // report id
            data[1] = 0x07;   // Get Firmware version

            result = pAhid->Write(&bytesTransferred, data, sizeof(data));

            if (result.IsSuccess())
            {
                result = pAhid->Read(&bytesTransferred, data, sizeof(data));

                if (result.IsSuccess())
                {
                    memset(data, 0, sizeof(data));
                    data[0] = 0x82;   // report id
                    data[1] = 0x01;   // Go to Dfu Mode

                    result = pAhid->Write(&bytesTransferred, data, sizeof(data));
                    // The device should re-enumerate as DFU device afer this
                }
            }
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result AhidDevice::ChangeToDfuDevice(uint16_t vid, uint16_t pid, uint16_t bcd) NN_NOEXCEPT
{
    Result result;
    nn::ahid::hdr::DeviceParameters hdrDeviceParameters;

    result = GetHdrDevice(&hdrDeviceParameters, vid, pid, bcd);

    if (result.IsSuccess())
    {
        nn::ahid::Ahid ahid;

        result = ahid.Initialize(&hdrDeviceParameters);

        if (result.IsSuccess())
        {
            result = ChangeToDfuDevice(&ahid);
        }

        ahid.Finalize();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
// Private
//////////////////////////////////////////////////////////////////////////////
Result AhidDevice::GetHdrDevice(nn::ahid::hdr::DeviceParameters *pDeviceParameters, uint16_t vid, uint16_t pid, uint16_t bcd) NN_NOEXCEPT
{
    NN_DETAIL_STDFU_INFO("Looking for AHID Device vid: %04x pid:%04x bcd: %04x\n", vid, pid, bcd);

    Result result;
    nn::ahid::hdr::Hdr hdr;

    result = hdr.Initialize();

    if (result.IsSuccess())
    {
        size_t outEntries;
        result = hdr.GetDeviceEntries(&outEntries);

        if (result.IsSuccess())
        {
            size_t inEntries = nn::ahid::hdr::AhidDevicesCountMax;
            nn::ahid::hdr::AttachFilter attachFilter;
            nn::ahid::hdr::DeviceHandle deviceHandle[inEntries];

            memset(&attachFilter, 0, sizeof(nn::ahid::hdr::AttachFilter));

            result = hdr.GetDeviceList(&outEntries, inEntries, deviceHandle, &attachFilter);

            if (result.IsSuccess())
            {
                result = ResultAhidDeviceNotFound();

                // for loop does not like to compare size_t in this build system :0
                int entries = static_cast<int>(outEntries);

                for (int i = 0; i < entries; i++)
                {
                    Result result1 = hdr.GetDeviceParameters(pDeviceParameters, deviceHandle[i]);

                    if (result1.IsSuccess())
                    {
                        if (
                                (pDeviceParameters->busId == nn::ahid::hdr::AhidBusIdUsb)   &&
                                (pDeviceParameters->vendorId == vid)                        &&
                                (pDeviceParameters->productId == pid)                       &&
                                (pDeviceParameters->versionNumber == bcd)
                            )
                        {
                            /*
                            NN_SDK_LOG("HDR DeviceHandle:               %08x\n",    deviceHandle[i]);
                            NN_SDK_LOG("    deviceHandle:               %08x\n",    hdrDeviceParameters.deviceHandle);
                            NN_SDK_LOG("    servicePath:                %s\n",      hdrDeviceParameters.servicePath);
                            NN_SDK_LOG("    usagePage:                  %04x\n",    hdrDeviceParameters.usagePage);
                            NN_SDK_LOG("    usageId:                    %04x\n",    hdrDeviceParameters.usageId);
                            NN_SDK_LOG("    busId:                      %d\n",      hdrDeviceParameters.busId);
                            NN_SDK_LOG("    vendorId:                   %08x\n",    hdrDeviceParameters.vendorId);
                            NN_SDK_LOG("    productId:                  %08x\n",    hdrDeviceParameters.productId);
                            NN_SDK_LOG("    versionNumber:              %08x\n",    hdrDeviceParameters.versionNumber);
                                */
                            result = ResultSuccess();

                            break;
                        }
                    }
                }
            }
        }

        hdr.Finalize();
    }

    if (!result.IsSuccess())
    {
        NN_DETAIL_STDFU_WARN("AHID device not found\n");
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
} // namespace stdfu
} // namespace nn
