﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/ssl/ssl_SessionCache.h>
#include <nn/ssl/ssl_Types.h>
#include <nn/ssl/detail/ssl_Build.h>

#include "detail/ssl_ServiceSession.h"
#include "server/ssl_Util.h"

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

using namespace nn::ssl::detail;

namespace nn { namespace ssl {


    nn::Result FlushSessionCache(
        uint32_t* pOutEntriesDeletedCount,
        const char * pInHostName,
        uint32_t hostNameBufLen,
        FlushSessionCacheOptionType optionValue
        ) NN_NOEXCEPT
    {
        nn::Result ret = ResultSuccess();
        uint32_t tempEntriesDeletedCount = 0;
        nn::ssl::sf::FlushSessionCacheOptionType flushOption;
        flushOption.option = optionValue;

        do
        {
            if (!ServiceSession::IsInitialized())
            {
                ret = ResultLibraryNotInitialized();
                break;
            }

            SharedPointer<ISslService>  *controlIf = ServiceSession::GetServiceSession();
            NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(controlIf, ret, ResultLibraryNotInitialized());

            if(optionValue == FlushSessionCacheOptionType_SingleHost)
            {
                NN_SDK_REQUIRES_NOT_NULL(pInHostName);
                if(pInHostName == nullptr)
                {
                    ret = ResultInvalidPointer();
                    break;
                }

                size_t strLen = strnlen(pInHostName, hostNameBufLen);
                if(strLen == 0)
                {
                    ret = ResultInvalidArgument();
                    break;
                }

                InBuffer buf(pInHostName, strLen + 1);
                ret = (*controlIf)->FlushSessionCache(&tempEntriesDeletedCount, buf, flushOption);
            }
            else if(optionValue == FlushSessionCacheOptionType_AllHosts)
            {
                if(pInHostName != nullptr)
                {
                    ret = ResultInvalidArgument();
                    break;
                }

                if(hostNameBufLen != 0)
                {
                    ret = ResultInvalidArgument();
                    break;
                }

                InBuffer buf(nullptr, 0);
                ret = (*controlIf)->FlushSessionCache(&tempEntriesDeletedCount, buf, flushOption);
            }
            else
            {
                ret = ResultInvalidOptionType();
                break;
            }

        } while (NN_STATIC_CONDITION(false));

        if(pOutEntriesDeletedCount != nullptr)
        {
            if(ret.IsSuccess())
            {
                *pOutEntriesDeletedCount = tempEntriesDeletedCount;
            }
            else
            {
                *pOutEntriesDeletedCount = 0;
            }
        }

        return ret;
    }

} }    //  nn::ssl
