﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>
#include <nn/os.h>

#include <nn/ssl/detail/ssl_Common.h>

#include "server/ssl_NssConfigFsManager.win32.h"
#include "server/ssl_MemoryManager.h"
#include "server/ssl_NssCore.h"
#include "detail/ssl_ServiceSession.h"
#include "server/ssl_Util.h"
#include "detail/ssl_ApiBase.h"
#include "server/ssl_Crl.h"
#include "server/ssl_EvCertUtilManager.h"

using namespace nn::ssl::detail;

namespace nn { namespace ssl {


namespace
{
std::aligned_storage<BuildParameter_MemoryManagerHeapSizeWindows>::type  g_HeapBuf;

// ------------------------------------------------------------------------------------------------
// class CoreStack
// ------------------------------------------------------------------------------------------------
class CoreStack
{
    bool m_IsInitialized;

public:
    CoreStack() NN_NOEXCEPT;
    ~CoreStack() NN_NOEXCEPT;
    bool IsInitialized() NN_NOEXCEPT;
    nn::Result Initialize() NN_NOEXCEPT;

    static void Finalize() NN_NOEXCEPT;
};

CoreStack::CoreStack() NN_NOEXCEPT : m_IsInitialized(false)
{
}

CoreStack::~CoreStack() NN_NOEXCEPT
{
}

bool CoreStack::IsInitialized() NN_NOEXCEPT
{
    return m_IsInitialized;
}

nn::Result CoreStack::Initialize() NN_NOEXCEPT
{
    nn::Result                  ret = nn::ResultSuccess();

    do
    {
        //  Init memory management
        SslMemoryManager::Initialize(
            reinterpret_cast<char*>(&g_HeapBuf),
            BuildParameter_MemoryManagerHeapSizeWindows);

        //  Mount the config filesystem area
        ret = NssConfigFsManagerWindows::Initialize();
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to mount file system\n");
            break;
        }

        //  Win32 builds are different than Horizon because they are direct link
        //  only.  There is no separate process for SSL like there is on Horizon,
        //  so the Initialize/Finalize must perform the same NSPR/NSS init and
        //  shutdown which would be done by the SSL server process.
        ret = NssCore::Initialize();
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to init NSS\n");
            break;
        }

        //  The SSL library will be finalized only when the program exits
        //  This is where the NSS/NSPR gets finalized
        atexit(CoreStack::Finalize);

        //  Initialize CRL manager which loads built-in CRL too
        ret = CrlManager::Initialize();
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to initialize CRL manager\n");
            break;
        }

        //  Initialize built-in policy OIDs
        ret = EvCertUtilManager::Initialize();
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to initialize built-in policy IDs\n");
        }

        //  Now that all the immediate init is done, tell NssCore to
        //  kick off any deferred init
        NssCore::StartDeferredInit();

        m_IsInitialized = true;
    } while (NN_STATIC_CONDITION(false));

    return ret;
}

void CoreStack::Finalize() NN_NOEXCEPT
{
    EvCertUtilManager::Finalize();

    if (CrlManager::Finalize().IsFailure())
    {
        NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to finalize CrlManager\n");
    }

    if (NssCore::Finalize().IsFailure())
    {
        NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to finalize NSS\n");
    }

    if (NssConfigFsManagerWindows::Finalize().IsFailure())
    {
        NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to unmount file system\n");
    }

    if (SslMemoryManager::Finalize().IsFailure())
    {
        NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to finalize memory mgmt\n");
    }
}

// ------------------------------------------------------------------------------------------------
// class SslApi
// ------------------------------------------------------------------------------------------------
class SslApi : public SslApiBase
{
private:
    CoreStack     m_Core;
    nn::os::Mutex m_Mutex;

    static nn::Result   ConvertResultToPublic(nn::Result result);

public:
    SslApi() NN_NOEXCEPT;
    ~SslApi() NN_NOEXCEPT;

    nn::Result Initialize() NN_NOEXCEPT;
    nn::Result Initialize(uint32_t concurLimit) NN_NOEXCEPT;
    nn::Result Finalize() NN_NOEXCEPT;
};

SslApi::SslApi() NN_NOEXCEPT :
    m_Mutex(false)
{
}

SslApi::~SslApi() NN_NOEXCEPT
{
}

nn::Result SslApi::ConvertResultToPublic(nn::Result result)
{
    if(ResultInternalNss::Includes(result))
    {
        // Getting internal result here means an error occurred on the lower NSS level
        NN_DETAIL_SSL_DBG_PRINT("[ssl] NSS returned internal error. Desc:%d\n",
            result.GetDescription());
        return ResultErrorLower();
    }

    return Util::ConvertResultFromInternalToExternal(result);
}

nn::Result SslApi::Initialize() NN_NOEXCEPT
{
    return SslApi::Initialize(nn::ssl::DefaultConcurrencyLimit);
}

nn::Result SslApi::Initialize(uint32_t concurLimit) NN_NOEXCEPT
{
    nn::Result                  ret = nn::ResultSuccess();

    do
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        //  Check manupilate reference counter and returns here if the library is already
        //  initialized
        ret = SslApiBase::Initialize();
        if (nn::ssl::ResultLibraryAlreadyInitialized::Includes(ret))
        {
            ret = ResultLibraryAlreadyInitialized();
            break;
        }

        //  On Windows, NSS/NSPR will be initialized only once. It will be finalized only when
        //  the program exits in CoreStack::Finalize() which is registered by atexit().
        if (m_Core.IsInitialized() == false)
        {
            ret = m_Core.Initialize();
            if (ret.IsFailure())
            {
                break;
            }
        }

        //  Init our connection to the service
        ret = ServiceSession::Initialize(concurLimit);
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to init service session\n");
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    return ConvertResultToPublic(ret);
}

nn::Result SslApi::Finalize() NN_NOEXCEPT
{
    nn::Result                  ret = nn::ResultSuccess();

    do
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        //  Manupilate a reference counter
        ret = SslApiBase::Finalize();
        if (ret.IsFailure())
        {
            if (nn::ssl::ResultLibraryNotInitialized::Includes(ret))
            {
                break;
            }

            if (nn::ssl::ResultInternalReferenceRemains::Includes(ret))
            {
                //  Someone is still referencing the library (the reference count is not 0
                //  after it got decremented), just return success
                ret = nn::ResultSuccess();
                break;
            }
        }

        if (!ServiceSession::IsInitialized())
        {
            ret = ResultLibraryNotInitialized();
            break;
        }

        ServiceSession::Finalize();
    } while (NN_STATIC_CONDITION(false));

    return ret;
}

// ------------------------------------------------------------------------------------------------
// Global object
// ------------------------------------------------------------------------------------------------
    SslApi g_SslLibrary;

} // un-named namespace

// ------------------------------------------------------------------------------------------------
// Public interface
// ------------------------------------------------------------------------------------------------
nn::Result Initialize() NN_NOEXCEPT
{
    return g_SslLibrary.Initialize();
}

nn::Result Initialize(uint32_t concurLimit) NN_NOEXCEPT
{
    return g_SslLibrary.Initialize(concurLimit);
}

nn::Result Finalize() NN_NOEXCEPT
{
    return g_SslLibrary.Finalize();
}

nn::Result GetSslResultFromValue(
    nn::Result* pOutResult,
    const char* pInValue,
    uint32_t inValueSize) NN_NOEXCEPT
{
    return SslApiBase::GetSslResultFromValue(pOutResult, pInValue, inValueSize);
}

}}
