﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ssl/detail/ssl_Common.h>
#include "server/ssl_ServiceDatabase.h"
#include "server/ssl_Util.h"


namespace nn { namespace ssl { namespace detail {

// -------------------------------------------------------------------------------------------------
// Static data used within the module
// -------------------------------------------------------------------------------------------------
nn::os::Mutex                   SslServiceDatabase::g_Lock(false);
SfObservableListType            SslServiceDatabase::g_Clients;
SfObservableListType            SslServiceDatabase::g_Contexts;
SfObservableListType            SslServiceDatabase::g_Connections;
SslDbObjObserver                SslServiceDatabase::g_ServiceObserver(&g_Lock, &g_Clients);
SslDbObjObserver                SslServiceDatabase::g_ContextObserver(&g_Lock, &g_Contexts);
SslDbObjObserver                SslServiceDatabase::g_ConnectionObserver(&g_Lock, &g_Connections);


// -------------------------------------------------------------------------------------------------
// Observer helper, used to cleanup SF managed objects via SslSfObservable.
// The observers do NOT need to verify that children of their object have been cleaned
// up since SF handles that for us.  For example, if a SslContextImpl is being destroyed,
// all SslConnectionImpl instances in g_Connections which have this SslContextImpl as the
// parent will already have been removed from g_Connections since the childrens'
// destructors will be called first so their observers are called first.
// -------------------------------------------------------------------------------------------------
SslDbObjObserver::SslDbObjObserver(nn::os::Mutex *lock, SfObservableListType *observerables) :
    m_Lock(lock),
    m_pList(observerables)
{
}


SslDbObjObserver::~SslDbObjObserver()
{
}


void SslDbObjObserver::OnDestroy(SslSfObservable *observedObj) NN_NOEXCEPT
{
    m_Lock->Lock();

    do
    {
        auto iterFound = std::find(m_pList->begin(), m_pList->end(), *observedObj);

        // If the iter got to the end of the list, the obj could not be found.
        if(iterFound == m_pList->end())
        {
            NN_DETAIL_SSL_DBG_PRINT("[SslDbObjObserver::OnDestroy] unknown obj: %p\n", observedObj);
            break;
        }

        //  Remove the tracking of this object.  No free is necessary as
        //  this method is called from the SslSfObservable's destructor.
        m_pList->erase(iterFound);
    } while (NN_STATIC_CONDITION(false));

    m_Lock->Unlock();
}


// -------------------------------------------------------------------------------------------------
// The SSL Database methods
// -------------------------------------------------------------------------------------------------
nn::Result SslServiceDatabase::FindContextsAndTakeAction(SslServiceImpl  *pOwnerService,
                                                         SslDbActionCb   pActionCb,
                                                         void            *pCbArg)
{
    nn::Result                  result = ResultSuccess();
    PRCList                     *pList = nullptr;
    NssUtil::ObjListNode        *pCurNode = nullptr;

    //  Lock everything down
    g_Lock.Lock();

    do
    {
        //  We need to construct a list which is going to contain all SSL contexts
        //  which are subordinate to the provided service.
        pList = new PRCList;
        if (pList == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SslServiceDatabase::FindContextsAndTakeAction] failed to create list\n");
            result = ResultInsufficientMemory();
            break;
        }

        PR_INIT_CLIST(pList);

        //  Walk the list of all contexts, look for ones which are from the
        //  owner service.  Each one we find needs to go into this new list.
        for (auto ctxIt = g_Contexts.begin();
             ctxIt != g_Contexts.end();
             ctxIt++)
        {
            SslContextImpl *pCtx = static_cast<SslContextImpl *>(&*ctxIt);
            if (pOwnerService != pCtx->GetParentSslService())
            {
                continue;
            }

            //  Create a new node and insert it into the list
            pCurNode = new NssUtil::ObjListNode;
            if (pCurNode == nullptr)
            {
                NN_DETAIL_SSL_DBG_PRINT("[SslServiceDatabase::FindContextsAndTakeAction] failed to create node for ctx %p\n",
                                        pCtx);
                result = ResultInsufficientMemory();
                break;
            }

            PR_INIT_CLIST(&pCurNode->links);
            pCurNode->pObj = pCtx;
            PR_APPEND_LINK(&pCurNode->links, pList);
        }

        //  Make sure the list was created successfully
        if (result.IsFailure())
        {
            break;
        }

        //  Call the action callback so it can do what it needs
        result = pActionCb(pList, pCbArg);
    } while (NN_STATIC_CONDITION(false));

    //  Unlock, we're done traversing the list and operating on it
    g_Lock.Unlock();

    //  Cleanup the list, if it exists
    if (pList != nullptr)
    {
        for (PRCList *pCurEntry = PR_LIST_HEAD(pList);
             pCurEntry != pList;
             pCurEntry = PR_LIST_HEAD(pList))
        {
            PR_REMOVE_LINK(pCurEntry);
            delete pCurEntry;
        }

        delete pList;
        pList = nullptr;
    }

    return result;
}


nn::Result SslServiceDatabase::AddSslService(SslServiceImpl *newService) NN_NOEXCEPT
{
    nn::Result                  ret = nn::ResultSuccess();

    //  Lock down the entire database for the duration of this op
    g_Lock.Lock();

    do
    {
        //  Bounds check for the max number of clients
        if (g_Clients.size() >= g_MaxSslClientCountTotal)
        {
            NN_DETAIL_SSL_DBG_PRINT("[AddSslService] too many clients\n");
            ret = ResultResourceMax();
            break;
        }

        ret = newService->AddObserver(&g_ServiceObserver);
        if(ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[AddSslService] AddObserver failed\n");
            break;
        }

        //  We have room for more clients.  Add this one to the list and
        //  attach our observer so we ca automagically clean up when it goes away.
        g_Clients.push_back(*newService);

    } while (NN_STATIC_CONDITION(false));

    g_Lock.Unlock();

    return ret;
}


nn::Result SslServiceDatabase::AddSslContext(SslContextImpl *newContext) NN_NOEXCEPT
{
    nn::Result                  ret = nn::ResultSuccess();
    uint32_t                    ctxCount = 0;

    //  Lock down the entire database for the duration of this op
    g_Lock.Lock();

    do
    {
        //  Make sure we actually have record of this client
        SslServiceImpl *parent = newContext->GetParentSslService();

        auto iterFound = std::find(g_Clients.begin(), g_Clients.end(), *parent);
        if (iterFound == g_Clients.end())
        {
            NN_DETAIL_SSL_DBG_PRINT("[AddSslContext] unknown client for context %p\n", newContext);
            ret = ResultErrorLower();
            break;
        }

        //  Verify this won't bump us over the global maximum context count
        if (g_Contexts.size() >= g_MaxSslContextCountTotal)
        {
            NN_DETAIL_SSL_DBG_PRINT("[AddSslContext] too many contexts\n");
            ret = ResultResourceMax();
            break;
        }

        //  Figure out how many existing context objects are present
        //  for this particular client and if it is within bounds.
        ret = GetContextCountLocked(parent, &ctxCount);
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[AddSslContext] failed to get context count for service %p\n", parent);
            break;
        }

        if (ctxCount >= nn::ssl::MaxContextCount)
        {
            NN_DETAIL_SSL_DBG_PRINT("[AddSslContext] too many contexts for client\n");
            ret = ResultResourceMax();
            break;
        }

        ret = newContext->AddObserver(&g_ContextObserver);
        if(ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[AddSslContext] AddObserver failed\n");
            break;
        }

        //  We have room, go ahead and track this one
        g_Contexts.push_back(*newContext);

    } while (NN_STATIC_CONDITION(false));

    g_Lock.Unlock();

    return ret;
}


nn::Result SslServiceDatabase::AddSslConnection(SslConnectionImpl *newConnection) NN_NOEXCEPT
{
    nn::Result                  ret = nn::ResultSuccess();
    uint32_t                    ctxConnectionCount = 0;
    uint32_t                    totalParentConnCount = 0;

    //  Lock down the entire database for the duration of this op
    g_Lock.Lock();

    do
    {
        //  Make sure we have a record of this connection's context
        SslContextImpl *parent = newConnection->GetParentSslContext();

        auto iterFound = std::find(g_Contexts.begin(), g_Contexts.end(), *parent);
        if (iterFound == g_Contexts.end())
        {
            NN_DETAIL_SSL_DBG_PRINT("[AddSslConnection] no context for connection %p\n", newConnection);
            ret = ResultErrorLower();
            break;
        }

        //  Check first to be sure we are not going to exceed the maximum
        //  number of global SSL connections
        if (g_Connections.size() >= g_MaxSslConnectionCountTotal)
        {
            NN_DETAIL_SSL_DBG_PRINT("[AddSslConnection] too many connections\n");
            ret = ResultResourceMax();
            break;
        }

        //  Now make sure we are not going to excced the maximum number
        //  of SSL connections for a given client.  We need to walk all
        //  contexts which have the same serivce parent to come up with
        //  the total.
        SslServiceImpl *parentService = parent->GetParentSslService();
        for (auto it = g_Contexts.begin(); it != g_Contexts.end(); ++it)
        {
            SslContextImpl *curCtx = static_cast<SslContextImpl *>(&*it);
            if (parentService == curCtx->GetParentSslService())
            {
                ret = GetConnectionCountLocked(curCtx, &ctxConnectionCount);
                if (ret.IsFailure())
                {
                    break;
                }

                totalParentConnCount += ctxConnectionCount;
            }
        }

        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[AddSslConnection] failed to get connection count for context %p\n", parent);
            break;
        }

        if (totalParentConnCount >= nn::ssl::MaxConnectionCount)
        {
            NN_DETAIL_SSL_DBG_PRINT("[AddSslConnection] too many connections for context\n");
            ret = ResultResourceMax();
            break;
        }

        ret = newConnection->AddObserver(&g_ConnectionObserver);
        if(ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[AddSslConnection] AddObserver failed\n");
            break;
        }

        //  We have the space, add the new connection and observe it
        g_Connections.push_back(*newConnection);

    } while (NN_STATIC_CONDITION(false));

    g_Lock.Unlock();

    return ret;
}


nn::Result SslServiceDatabase::GetContextCountLocked(SslServiceImpl *service, uint32_t *count) NN_NOEXCEPT
{
    *count = 0;

    for (auto ctxIt = g_Contexts.begin();
        ctxIt != g_Contexts.end();
        ++ctxIt)
    {
        SslContextImpl *ctx = static_cast<SslContextImpl *>(&*ctxIt);
        if (service == ctx->GetParentSslService())
        {
            *count = *count + 1;
        }
    }

    return nn::ResultSuccess();
}


nn::Result SslServiceDatabase::GetConnectionCountLocked(SslContextImpl *context, uint32_t *count) NN_NOEXCEPT
{
    *count = 0;

    for (auto conIt = g_Connections.begin();
        conIt != g_Connections.end();
        ++conIt)
    {
        SslConnectionImpl *conn = static_cast<SslConnectionImpl *>(&*conIt);
        if (context == conn->GetParentSslContext())
        {
            *count = *count + 1;
        }
    }

    return nn::ResultSuccess();
}

nn::Result SslServiceDatabase::GetContextCount(SslServiceImpl *service, uint32_t *count) NN_NOEXCEPT
{
    nn::Result                  ret;

    g_Lock.Lock();
    ret = GetContextCountLocked(service, count);
    g_Lock.Unlock();
    return ret;
}


nn::Result SslServiceDatabase::GetConnectionCount(SslContextImpl *context, uint32_t *count) NN_NOEXCEPT
{
    nn::Result                  ret;

    g_Lock.Lock();
    ret = GetConnectionCountLocked(context, count);
    g_Lock.Unlock();
    return ret;
}


}}}
