﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/ssl/ssl_Types.h>
#include <nn/ssl/detail/ssl_Build.h>

#include "server/ssl_Util.h"
#include "server/ssl_MemoryManager.h"

// ------------------------------------------------------------------------------------------------
// Memory manager
// TODOs
//     - Add actual allocation/free code in AllocateChunk/Free
//     - Add tracking logic to manage memory allocation in process level
// ------------------------------------------------------------------------------------------------

namespace nn { namespace ssl { namespace detail {

nn::lmem::HeapHandle                         SslMemoryManager::g_Heap;
size_t                                       SslMemoryManager::g_HeapStartFreeSize;
nn::sf::ExpHeapAllocator                     SslMemoryManager::g_SfAllocator;
char*                                        SslMemoryManager::g_pHeapHead = nullptr;
size_t                                       SslMemoryManager::g_HeapSize = 0;

Result SslMemoryManager::Initialize(char* pBuffer, size_t bufferSize)
{
    nn::Result result = nn::ResultSuccess();
    do
    {
        if (pBuffer == nullptr || bufferSize == 0)
        {
            result = ResultInternalLogicError();
            break;
        }

        if (g_Heap == nullptr)
        {
            g_Heap = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(pBuffer),
                                             bufferSize,
                                             nn::lmem::CreationOption_ThreadSafe);
            g_HeapStartFreeSize = nn::lmem::GetExpHeapTotalFreeSize(g_Heap);

            g_SfAllocator.Attach(g_Heap);

            g_pHeapHead = pBuffer;
            g_HeapSize  = bufferSize;
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}

Result SslMemoryManager::Finalize()
{
    //  Detatch the SF allocator before destroying the heap
    g_SfAllocator.Detach();

    size_t freeSpace = nn::lmem::GetExpHeapTotalFreeSize(g_Heap);
    if (freeSpace != g_HeapStartFreeSize)
    {
        NN_DETAIL_SSL_DBG_PRINT("[SslMemoryManager] leaking %d bytes\n",
                                g_HeapStartFreeSize - freeSpace);
    }

    nn::lmem::DestroyExpHeap(g_Heap);
    g_Heap = nullptr;
    memset(g_pHeapHead, 0, g_HeapSize);

    return ResultSuccess();
}

void* SslMemoryManager::AllocateChunk(size_t sizeInByte, uint64_t id)
{
    NN_UNUSED(id);

    //  TODO: Add histogram or other tracking by an ID
    void *ret = nn::lmem::AllocateFromExpHeap(g_Heap, sizeInByte);
    return ret;
}

Result SslMemoryManager::Free(void* pMemoryAddress, uint64_t pid)
{
    NN_UNUSED(pid);

    nn::lmem::FreeToExpHeap(g_Heap, pMemoryAddress);
    return ResultSuccess();
}


nn::sf::ExpHeapAllocator *SslMemoryManager::GetSfAllocator()
{
    return &g_SfAllocator;
}

}}}
