﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_SdCardForDebug.h>

#include <nn/settings/factory/settings_Ssl.h>
#include <nn/spl/spl_Api.h>

#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/detail/ssl_Common.h>
#include "ssl_InternalPki.h"
#include "ssl_Util.h"

#include "ssl_NssCommon.h"


#if defined(NN_BUILD_CONFIG_OS_HORIZON)
/*  The device label is defined within the PKCS#11 module and referenced here
    for lookup purposes.  It is just declared extern here rather than try
    to include extreaneous NSS headers.  */
extern "C" {
extern const char*              g_pDevicePkiLabel;
}
#endif    //  NN_BUILD_CONFIG_OS_HORIZON


namespace nn { namespace ssl { namespace detail {

// ------------------------------------------------------------------------------------------------
// class InternalPkiManager
// ------------------------------------------------------------------------------------------------
// ------------------------------------------------------------------------------------------------
// Constatns
// ------------------------------------------------------------------------------------------------
#if defined(NN_BUILD_CONFIG_OS_WIN)
const char* InternalPkiManager::g_MountName                        = "psuedopkivol";
const char* InternalPkiManager::g_PsuedoCertPath                   = "psuedopkivol:/device_cert.p12";
const char* InternalPkiManager::g_PsuedoPasswordPath               = "psuedopkivol:/device_cert.key";
const char* InternalPkiManager::g_pDevicePkiLabel                  = "_nxdev_ccd_";
#endif    //  NN_BUILD_CONFIG_OS_WIN


// ------------------------------------------------------------------------------------------------
// Private Data
// ------------------------------------------------------------------------------------------------
SECMODModule*                   InternalPkiManager::g_pMod = nullptr;
PK11SlotInfo*                   InternalPkiManager::g_pSlot = nullptr;


// ------------------------------------------------------------------------------------------------
// Private Methods
// ------------------------------------------------------------------------------------------------
#if defined(NN_BUILD_CONFIG_OS_WIN)
nn::Result InternalPkiManager::ReadFileContents(const char *path,
                                                char       **pOutBuf,
                                                uint32_t   *pOutLen)
{
    nn::Result                  ret = ResultSuccess();
    fs::FileHandle              file;
    int64_t                     fileLen;
    char                        *buf = nullptr;

    file.handle = nullptr;
    *pOutBuf = nullptr;
    *pOutLen = 0;

    do
    {
        //  Open the file and get its size
        ret = fs::OpenFile(&file, path, fs::OpenMode_Read);
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] failed to open file \'%s\'.\n", path);
            ret = ResultErrorLower();
            break;
        }

        ret = fs::GetFileSize(&fileLen, file);
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] Failed to get the file size for \'%s\'.\n", path);
            break;
        }

        //  Allocate a buffer large enough to hold it.
        buf = new char[static_cast<unsigned int>(fileLen + 1)];
        if (buf == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] Failed to allocate the memory for file \'%s\'.\n", path);
            ret = ResultInsufficientMemory();
            break;
        }

        //  Now read all of the data out
        memset(buf, 0, static_cast<size_t>(fileLen + 1));
        ret = fs::ReadFile(file, 0, buf, static_cast<size_t>(fileLen));
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] Failed to read file \'%s\'.\n", path);
            break;
        }

        *pOutBuf = buf;
        *pOutLen = static_cast<uint32_t>(fileLen);
    } while (NN_STATIC_CONDITION(false));

    if (file.handle != nullptr)
    {
        fs::CloseFile(file);
        file.handle = nullptr;
    }

    if (ret.IsFailure())
    {
        //  Cleanup temporary data
        if (buf != nullptr)
        {
            delete[] buf;
            buf = nullptr;
            fileLen = 0;
        }
    }

    return ret;
}


nn::Result InternalPkiManager::ReadFromExternal(char     **pOutCertData,
                                                uint32_t *pOutCertLen,
                                                char     **pOutPwData,
                                                uint32_t *pOutPwLen)
{
    nn::Result                  result = ResultSuccess();

    do
    {
        result = ReadFileContents(g_PsuedoCertPath, pOutCertData, pOutCertLen);
        if (result.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] Failed to read the psuedo device client certificate file (desc:%d)\n", result.GetDescription());
            break;
        }

        result = ReadFileContents(g_PsuedoPasswordPath, pOutPwData, pOutPwLen);
        if (result.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] Failed to read the password file for the psuedo device client certificate file.\n");
            break;
        }

        NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] Obtained the psuedo device client certificate and the password.\n");

    } while(NN_STATIC_CONDITION(false));

    return result;
}


nn::Result InternalPkiManager::ReadAndImportFromSdCard()
{
    nn::Result                  ret = ResultSuccess();
    bool                        mounted = false;
    char                        *pCertData = nullptr;
    uint32_t                    certDataLen = 0;
    char                        *pPwData = nullptr;
    uint32_t                    pwDataLen = 0;

    do
    {
        //  Mount the SD card to our own mount point
        ret = nn::fs::MountSdCardForDebug(g_MountName);
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] Failed to mount SD card.\n");
            ret = ResultErrorLower();
            break;
        }

        mounted = true;
        NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] Mounted SD card.\n");

        //  Read the data from the mounted filesystem
        ret = ReadFromExternal(&pCertData, &certDataLen, &pPwData, &pwDataLen);
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] Failed to read cert/pw data\n");
            ret = ResultErrorLower();
            break;
        }

        //  Import it into the PKCS#11 slot via PKCS#12 utility functions.  Note
        //  that we do NOT need to create a PKCS#11 slot for this, which is
        //  different than what is done in CertStore for regular client PKI.
        Pk12ImportDetails       importDetails;
        CERTCertificate         *pCert = nullptr;
        SECKEYPrivateKey        *pPrivKey = nullptr;
        char                    *finalNick;

        finalNick = PL_strdup(GetNickname(nn::ssl::Context::InternalPki_DeviceClientCertDefault));
        if (finalNick == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] Failed to create temp nickname\n");
            ret = ResultInsufficientMemory();
            break;
        }

        importDetails.pInP12Data = pCertData;
        importDetails.pInPwData  = pPwData;
        importDetails.p12DataLen = certDataLen;
        importDetails.pwDataLen  = pwDataLen;
        ret = Pk11ModuleUtil::ImportPk12(&importDetails,
                                         finalNick,
                                         PL_strlen(finalNick) + 1,
                                         &pCert,
                                         &pPrivKey);

        //  We don't need the cert/key at this level, we can discard them
        //  and the nickname immediatley.
        PL_strfree(finalNick);

        if (pCert != nullptr)
        {
            CERT_DestroyCertificate(pCert);
        }

        if (pPrivKey != nullptr)
        {
            SECKEY_DestroyPrivateKey(pPrivKey);
        }

        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] failed to import from SD\n");
            break;
        }

    } while (NN_STATIC_CONDITION(false));

    //  Unmount the SD card, we are done with it
    if (mounted)
    {
        nn::fs::Unmount(g_MountName);
    }

    //  Cleanup our buffers, no longer needed
    if (pCertData != nullptr)
    {
        memset(pCertData, 0, certDataLen);
        delete[] pCertData;
    }

    if (pPwData != nullptr)
    {
        memset(pPwData, 0, pwDataLen);
        delete[] pPwData;
    }

    return ret;
}
#endif    //  NN_BUILD_CONFIG_OS_WIN


// ------------------------------------------------------------------------------------------------
// Public Methods
// ------------------------------------------------------------------------------------------------
const char* InternalPkiManager::GetNickname(nn::ssl::Context::InternalPki pkiType)
{
    const char                  *ret = nullptr;

    switch (pkiType)
    {
        case nn::ssl::Context::InternalPki::InternalPki_DeviceClientCertDefault:
            ret = g_pDevicePkiLabel;
            break;

        default:
            NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] invalid PKI type (%d), no nickname\n", pkiType);
            break;
    }

    return ret;
}


nn::Result InternalPkiManager::CheckAndInstallIfNeeded()
{
    nn::Result                  ret = ResultErrorLower();
    CERTCertificate             *pCert = nullptr;
    SECKEYPrivateKey            *pKey = nullptr;
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    SECMODModule                *pMod = nullptr;
    PK11SlotInfo                *pSlot = nullptr;
#endif    //  NN_BUILD_CONFIG_OS_HORIZON

    do
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        //  Query the default cert/key store and see if we already have
        //  the device cert/key installed.
        ret = Pk11ModuleUtil::GetClientPki(g_pDevicePkiLabel,
                                           &pCert,
                                           &pKey);
        if (ret.IsSuccess())
        {
            //  Already imported (not our first run!), so just jump out
            //  with a success.
            NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] device PKI already imported\n");
            break;
        }

        //  The device PKI has not been imported yet, so do it now.
        //  For Windows variants, support reading credentials from PKCS#12
        //  off SD card (Windows is emulated by fs.)
        ret = ReadAndImportFromSdCard();
        if (ret.IsSuccess())
        {
            //  If this was successful, break out as this overrides
            //  built-in device credentials (non-Windows)
            NN_DETAIL_SSL_DBG_PRINT("(InternalPkiManager) successfully imported device PKI from SD\n");
            break;
        }

        NN_DETAIL_SSL_DBG_PRINT("(InternalPkiManager) failed to import from SD\n");

#else
        char                    modOpt[] = { '\0' };

        //  On NX platforms, use the custom PKCS#11 module which wraps nn::spl
        //  to do device credential related activities.
        SECStatus status =
            SECMOD_AddNewModuleEx("cknnspl",
                                  "libnn_cknnspl3.a",
                                  SECMOD_PubMechFlagstoInternal(PUBLIC_MECH_RSA_FLAG),
                                  0,
                                  modOpt,
                                  NULL);
        if (status != SECSuccess) {
            NN_DETAIL_SSL_WARN_PRINT("[InternalPkiManager] failed to add cknnspl\n");
        }

        pMod = SECMOD_FindModule("cknnspl");
        if (pMod == nullptr)
        {
            NN_DETAIL_SSL_WARN_PRINT("[InternalPkiManager] failed to load cknnspl\n");
            ret =  ResultErrorLower();
            break;
        }

        NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] getting cknnspl slot...\n");
        pSlot = PK11_FindSlotByName("Nintendo Crypto Services");
        if (pSlot == nullptr)
        {
            NN_DETAIL_SSL_WARN_PRINT("[InternalPkiManager] failed to open cknnspl slot\n");
            ret = ResultErrorLower();
            break;
        }

        NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager] loaded cknnspl mod %p, slot %p\n",
                                pMod,
                                pSlot);
        //  Done
        g_pSlot = pSlot;
        pSlot = nullptr;
        g_pMod  = pMod;
        pMod = nullptr;
        ret = ResultSuccess();

#endif    //  NN_BUILD_CONFIG_OS_WIN

    } while (NN_STATIC_CONDITION(false));

    if (pCert != nullptr)
    {
        CERT_DestroyCertificate(pCert);
        pCert = nullptr;
    }

    if (pKey != nullptr)
    {
        SECKEY_DestroyPrivateKey(pKey);
        pKey = nullptr;
    }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    if (pSlot != nullptr)
    {
        PK11_FreeSlot(pSlot);
        pSlot = nullptr;
    }

    if (pMod != nullptr)
    {
        SECMOD_DestroyModule(pMod);
        pMod = nullptr;
    }
#endif    //  NN_BUILD_CONFIG_OS_HORIZON

    return ret;
}


PK11SlotInfo* InternalPkiManager::GetPk11Slot()
{
    PK11SlotInfo*               pResult = g_pSlot;

    if (pResult == nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager::GetPk11Slot] no cknnspl mod, use internal slot\n");
        pResult = PK11_GetInternalKeySlot();
    }
    else
    {
        //  Bump the ref count on the slot before returning it as the
        //  caller needs to "free" it when it is done using it.
        PK11_ReferenceSlot(pResult);
    }

    NN_DETAIL_SSL_DBG_PRINT("[InternalPkiManager::GetPk11Slot] %p\n", pResult);
    return pResult;
}


}}} // namespace nn { namespace ssl { namespace detail {
