﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/sf_HipcSimpleClientSessionManager.h>
#include <nn/sm/sm_Result.h>

#include "detail/ssl_ISslServiceFactory.h"
#include "detail/ssl_ISslServiceDfcFactory.h"
#include "server/ssl_MemoryManager.h"
#include "server/ssl_SslServiceManager.h"
#include "server/ssl_Util.h"


namespace nn { namespace ssl { namespace detail {

static HipcSimpleClientSessionManager   g_SessionMgr;
static SharedPointer<ISslService>       g_Service = nullptr;


nn::Result ISslServiceFactory::Initialize(uint32_t maxConcurTrans) NN_NOEXCEPT
{
    nn::Result                  ret = ResultSuccess();

    do
    {
        if ((g_Service != nullptr) && (g_Service.Get() != nullptr))
        {
            NN_DETAIL_SSL_DBG_PRINT("[ISslServiceFactory::Initialize] already initialized\n");
            ret = ResultLibraryAlreadyInitialized();
            break;
        }

        ret = g_SessionMgr.InitializeByName<ISslService,
                                            nn::sf::ExpHeapAllocator::Policy>(&g_Service,
                                                                              SslMemoryManager::GetSfAllocator(),
                                                                              nn::ssl::detail::SslServiceSessionName);
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[CreateISslService] unable to init: %d-%d\n",
                                    ret.GetModule(),
                                    ret.GetDescription());
            if (nn::sm::ResultMaxSessions::Includes(ret))
            {
                ret = ResultResourceBusy();
            }
            else
            {
                ret = ResultErrorLower();
            }

            break;
        }

        //  Set the max sessions for this client to be the max Context count
        if (static_cast<uint32_t>(g_SessionMgr.GetSessionCount()) != maxConcurTrans)
        {
            g_SessionMgr.SetSessionCount(maxConcurTrans);
        }

    } while (NN_STATIC_CONDITION(false));

    return ret;
}


nn::Result ISslServiceFactory::Finalize() NN_NOEXCEPT
{
    g_Service = nullptr;
    g_SessionMgr.Finalize();
    return ResultSuccess();
}


nn::Result ISslServiceFactory::CreateISslService(SharedPointer<ISslService> *pOut) NN_NOEXCEPT
{
    nn::Result                  ret = ResultSuccess();

    do
    {
        if ((g_Service == nullptr) || (g_Service.Get() == nullptr))
        {
            NN_DETAIL_SSL_DBG_PRINT("[CreateISslService] not initialized!\n");
            ret = ResultInternalLogicError();
            break;
        }

    } while (NN_STATIC_CONDITION(false));

    if (ret.IsSuccess())
    {
        *pOut = g_Service;
    }

    return ret;
}

} } }
