﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>

#include <nn/ssl/detail/ssl_Common.h>
#include <nn/ssl/ssl_Api.debug.h>

#include "ssl_DebugUtil.h"
#include "ssl_DebugImpl.h"

#ifndef NN_DETAIL_SSL_ENABLE_PROCESS_DEBUG
#error 'This file must not be used unless the debug flag is defined'
#endif

namespace nn { namespace ssl { namespace detail {

namespace
{

nn::Result ConfigurePeriodicDump(
    Debug::Output* pOutBufferInfo,
    Debug::Input* pInBufferInfo)
{
    NN_UNUSED(pOutBufferInfo);

    nn::Result result = nn::ResultSuccess();
    do
    {
        if (pInBufferInfo->bufferSize < sizeof(Debug::PeriodicDumpConfig))
        {
            result = ResultBufferTooShort();
            break;
        }

        char* pTmpConfig = const_cast<char*>(pInBufferInfo->pBuffer);
        Debug::PeriodicDumpConfig* pConfig = reinterpret_cast<Debug::PeriodicDumpConfig*>(pTmpConfig);

        switch (pConfig->type)
        {
        case Debug::PeriodicDumpType_HeapStats:
            {
                g_DebugUtil.ConfigurePeriodicDump(pConfig->enable);
            }
            break;
        case Debug::PeriodicDumpType_SessionCacheInfo:
            {
                g_DebugUtil.GetSessionCacheDebugger()->ConfigurePeriodicDump(pConfig->enable);
            }
            break;
        default:
            {
                NN_SDK_LOG("[ConfigurePeriodicDump] Invalid type: 0x%x)\n", pConfig->type);
                result = ResultInternalLogicError();
            }
            break;
        }

    } while (NN_STATIC_CONDITION(false));

    return result;
}

// Get SSL process heap stats
nn::Result GetHeapStats(
    Debug::Output* pOutBufferInfo,
    Debug::Input* pInBufferInfo)
{
    NN_UNUSED(pInBufferInfo);

    nn::Result result = nn::ResultSuccess();
    do
    {
        if (pOutBufferInfo->bufferSize < sizeof(Debug::HeapStats))
        {
            result = ResultBufferTooShort();
            break;
        }

        Debug::HeapStats* pStats = reinterpret_cast<Debug::HeapStats*>(pOutBufferInfo->pBuffer);
        g_DebugUtil.GetHeapStats(&pStats->curSize, &pStats->maxSize, &pStats->minSize);
    } while (NN_STATIC_CONDITION(false));

    return result;
}

nn::Result ProcessHeapTrack(
    bool processStart,
    Debug::Output* pOutBufferInfo,
    Debug::Input* pInBufferInfo)
{
    nn::Result result = nn::ResultSuccess();

    do
    {
        if (pInBufferInfo->bufferSize < sizeof(Debug::HeapTrackPoint))
        {
            result = ResultBufferTooShort();
            break;
        }

        char* pTmpTracker = const_cast<char*>(pInBufferInfo->pBuffer);
        Debug::HeapTrackPoint* pTracker = reinterpret_cast<Debug::HeapTrackPoint*>(pTmpTracker);

        if (processStart == true)
        {
            g_DebugUtil.StartHeapTrack(pTracker->_name);
        }
        else
        {
            if (pOutBufferInfo->bufferSize < sizeof(Debug::TrackStats))
            {
                result = ResultBufferTooShort();
                break;
            }

            Debug::TrackStats* pOutStats = reinterpret_cast<Debug::TrackStats*>(pOutBufferInfo->pBuffer);
            bool rval = g_DebugUtil.EndHeapTrack(
                &pOutStats->elapsedMsec,
                &pOutStats->delta,
                &pOutStats->curHeapSpace,
                pTracker->_name);
            if (rval == false)
            {
                result = ResultInvalidReference();
                break;
            }
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}

} // Un-named namespace


// ------------------------------------------------------------------------------------------------
// Public interface
// ------------------------------------------------------------------------------------------------
nn::Result DebugImpl::Ioctl(
    Debug::Output* pOutBufferInfo,
    Debug::Input*  pInBufferInfo,
    Debug::IoctlCommand cmd)
{
    nn::Result result = nn::ResultSuccess();

    do
    {
        switch (cmd)
        {
        case Debug::IoctlCommand_ConfigurePeriodicDump:
            {
                result = ConfigurePeriodicDump(pOutBufferInfo, pInBufferInfo);
            }
            break;
        case Debug::IoctlCommand_GetHeapStats:
            {
                result = GetHeapStats(pOutBufferInfo, pInBufferInfo);
            }
            break;
        case Debug::IoctlCommand_StartHeapTrack:
        case Debug::IoctlCommand_EndHeapTrack:
            {
                result = ProcessHeapTrack(
                    (cmd == Debug::IoctlCommand_StartHeapTrack)?true:false,
                    pOutBufferInfo,
                    pInBufferInfo);
            }
            break;
        case Debug::IoctlCommand_Invalid:
        default:
            {
                NN_SDK_LOG("[DebugImpl::Ioctl] Received unsupported command: 0x%x)\n", cmd);
                result = ResultInternalLogicError();
            }
            break;
        }
    } while (NN_STATIC_CONDITION (false));

    return result;
}

}}}
