﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/srepo/service/srepo_Server.h>
#include <nn/srepo/detail/srepo_Log.h>
#include <nn/srepo/detail/ipc/srepo_ServiceConfig.h>
#include <nn/srepo/detail/ipc/srepo_ServiceName.h>
#include <nn/srepo/detail/service/srepo_Capability.h>
#include <nn/srepo/detail/service/srepo_Service.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os.h>
#include <nn/sf/sf_HipcServer.h>

namespace nn { namespace srepo { namespace service {

namespace
{
    const int ParallelExecutionCountMax = 1;

    nn::os::ThreadType g_Threads[ParallelExecutionCountMax] = {};
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStacks[ParallelExecutionCountMax][16 * 1024];

    struct SrepoServiceHipcSimpleAllInOneServerManagerOption
    {
        // これ以上のサイズのデータは、[BufferTransferMode(BufferTransferMode.MapAlias)] でマップ転送に切り替えること。
        static const size_t PointerTransferBufferSize = 128;

        static const bool CanDeferInvokeRequest = false;

        static const int SubDomainCountMax = detail::ipc::SessionCountMaxForUser + detail::ipc::SessionCountMaxForAdministrator;
        static const int ObjectInSubDomainCountMax = SubDomainCountMax * detail::ipc::ObjectCountPerProcess;
    };

    class SrepoServiceServerManager :
        public nn::sf::HipcSimpleAllInOneServerManager<
            SrepoServiceHipcSimpleAllInOneServerManagerOption::ObjectInSubDomainCountMax * 2,
            detail::ipc::ServiceCount,
            SrepoServiceHipcSimpleAllInOneServerManagerOption>
    {
    };

    SrepoServiceServerManager* g_ServerManager = nullptr;
    std::aligned_storage<sizeof (SrepoServiceServerManager)>::type g_ServerManagerStorage;
}

namespace
{
    void RegisterService(int index, int sessionCountMax, const char* serviceName, Bit32 capability) NN_NOEXCEPT
    {
        detail::service::InitializeService(index, serviceName, capability);

        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ServerManager->RegisterObjectForPort(detail::service::GetSrepoService(index),
            sessionCountMax, serviceName));
    }

    void IpcDispatcher(void*) NN_NOEXCEPT
    {
        g_ServerManager->LoopAuto();
    }
}

void StartServer() NN_NOEXCEPT
{
    NN_DETAIL_SREPO_INFO("[srepo] Start service...\n");

    g_ServerManager = new (&g_ServerManagerStorage) SrepoServiceServerManager;

    // srepo:u
    RegisterService(0, detail::ipc::SessionCountMaxForUser, detail::ipc::ServiceNameForUser,
        detail::service::Capability::Flag_Write);

    // srepo:a
    RegisterService(1, detail::ipc::SessionCountMaxForAdministrator, detail::ipc::ServiceNameForAdministrator,
        detail::service::Capability::Flag_All);

    g_ServerManager->Start();

    for (int i = 0; i < ParallelExecutionCountMax; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Threads[i],
            IpcDispatcher, nullptr, g_ThreadStacks[i], sizeof (g_ThreadStacks[i]), NN_SYSTEM_THREAD_PRIORITY(srepo, IpcServer)));

        nn::os::SetThreadNamePointer(&g_Threads[i], NN_SYSTEM_THREAD_NAME(srepo, IpcServer));
        nn::os::StartThread(&g_Threads[i]);
    }

    NN_DETAIL_SREPO_INFO("[srepo] Start service... done!\n");
}

void StopServer() NN_NOEXCEPT
{
    g_ServerManager->RequestStop();

    for (int i = 0; i < ParallelExecutionCountMax; i++)
    {
        nn::os::DestroyThread(&g_Threads[i]);
    }

    g_ServerManager->~SrepoServiceServerManager();
    g_ServerManager = nullptr;

    detail::service::FinalizeServices();

    NN_DETAIL_SREPO_INFO("[srepo] End service.\n");
}

}}}
