﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_Types.h>                  // for nn::sf::SharedPointer
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/spsm/spsm_Api.h>
#include <nn/spsm/spsm_Debug.h>
#include <nn/spsm/spsm_ShimInternal.h>
#include <nn/spsm/spsm_ServiceName.h>
#include <nn/spsm/detail/spsm_IPowerStateInterface.h>
#include <nn/spsm/detail/spsm_Log.h>

namespace nn { namespace spsm {

namespace {

const int NumberOfClients = 1;
sf::SimpleAllInOneHipcClientManager<NumberOfClients> g_ClientManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;
sf::ShimLibraryObjectHolder<detail::IPowerStateInterface> g_PowerStateInterfaceHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

nn::os::Mutex g_ManagerAccessMutex(false);

}

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_ManagerAccessMutex)> lock(g_ManagerAccessMutex);
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ClientManager.InitializeShimLibraryHolder(&g_PowerStateInterfaceHolder, nn::spsm::ServiceName));
}

void Finalize() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_ManagerAccessMutex)> lock(g_ManagerAccessMutex);
    g_PowerStateInterfaceHolder.FinalizeHolder();
}

void InitializeWith(nn::sf::SharedPointer<detail::IPowerStateInterface>&& manager) NN_NOEXCEPT
{
    std::lock_guard<decltype(g_ManagerAccessMutex)> lock(g_ManagerAccessMutex);
    g_PowerStateInterfaceHolder.InitializeHolderDirectly(std::move(manager));
}

nn::sf::SharedPointer<detail::IPowerStateInterface> GetPowerStateInterface() NN_NOEXCEPT
{
    return g_PowerStateInterfaceHolder.GetObject();
}

PowerState GetCurrentState() NN_NOEXCEPT
{
    int32_t outCurrentState;
    auto result = g_PowerStateInterfaceHolder.GetObject()->GetCurrentState(static_cast<std::int32_t*>(&outCurrentState));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return static_cast<PowerState>(outCurrentState);
}

Result SleepSystemAndWaitAwake() NN_NOEXCEPT
{
    // スリープをトリガーしつつ、受け取った FullAwake 通知ハンドルにシステムイベントをバインド
    os::SystemEvent fullAwakeEvent;
    sf::NativeHandle temp;
    NN_RESULT_DO(g_PowerStateInterfaceHolder.GetObject()->EnterSleep(&temp));
    nn::os::AttachReadableHandleToSystemEvent(fullAwakeEvent.GetBase(), temp.GetOsHandle(), temp.IsManaged(), os::EventClearMode_ManualClear);
    temp.Detach();

    // イベントを待つ
    fullAwakeEvent.Wait();
    fullAwakeEvent.Clear();
    NN_RESULT_SUCCESS;
}

WakeReasonFlagSet GetLastWakeReason() NN_NOEXCEPT
{
    WakeReasonFlagSet outWakeReasonFlagSet;
    auto result = g_PowerStateInterfaceHolder.GetObject()->GetLastWakeReason(static_cast<WakeReasonFlagSet*>(&outWakeReasonFlagSet));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return outWakeReasonFlagSet;
}

void StartShutdown(bool reboot) NN_NOEXCEPT
{
    auto result = g_PowerStateInterfaceHolder.GetObject()->Shutdown(reboot);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

NN_NORETURN void ShutdownSystem() NN_NOEXCEPT
{
    StartShutdown(false);
    for (;;)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(100000));
    }
    // NEVER REACHED HERE
}

NN_NORETURN void RebootSystem() NN_NOEXCEPT
{
    StartShutdown(true);
    for (;;)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(100000));
    }
    // NEVER REACHED HERE
}

void InitializeNotificationMessageEvent(nn::os::SystemEventType* pOutEvent) NN_NOEXCEPT
{
    sf::NativeHandle temp;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_PowerStateInterfaceHolder.GetObject()->GetNotificationMessageEventHandle(&temp));
    nn::os::AttachReadableHandleToSystemEvent(pOutEvent, temp.GetOsHandle(), temp.IsManaged(), os::EventClearMode_ManualClear);
    temp.Detach();
}

NotificationMessage GetNotificationMessage() NN_NOEXCEPT
{
    std::int32_t message;
    NN_RESULT_DO(g_PowerStateInterfaceHolder.GetObject()->ReceiveNotificationMessage(&message));
    return static_cast<NotificationMessage>(message);
}

void AnalyzeLogForLastSleepWakeSequence(SleepWakeSequenceAnalyzedData* pOutData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutData);
    SleepWakeSequenceAnalyzedData temp;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_PowerStateInterfaceHolder.GetObject()->AnalyzeLogForLastSleepWakeSequence(&temp));
    *pOutData = temp;
}

void AnalyzePerformanceLogForLastSleepWakeSequence(SleepWakeSequencePerformanceData* pOutData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutData);
    nn::sf::OutBuffer buffer( (char*)pOutData, sizeof(nn::spsm::SleepWakeSequencePerformanceData) );
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_PowerStateInterfaceHolder.GetObject()->AnalyzePerformanceLogForLastSleepWakeSequence(buffer));
}

void ResetEventLog() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_PowerStateInterfaceHolder.GetObject()->ResetEventLog());
}

void ChangeHomeButtonLongPressingTime(TimeSpan timeSpan) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_PowerStateInterfaceHolder.GetObject()->ChangeHomeButtonLongPressingTime(timeSpan.GetNanoSeconds()));
}

void PutErrorState() NN_NOEXCEPT
{
    auto result = g_PowerStateInterfaceHolder.GetObject()->PutErrorState();
    if ( result.IsFailure() )
    {
        // この関数を呼んだときにはすでに異常系に入っているので、
        // ベストエフォートで進めてもらうためにログは出しつつ上位には失敗扱いさせない
        NN_DETAIL_SPSM_ERROR("PutErrorState failed with 0x%08x\n", result.GetInnerValueForDebug());
    }
}

}}
