﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/hid/system/hid_SleepButton.h>

#include "spsm_PowerButton.h"

namespace nn { namespace spsm { namespace observer {

//-----------------------------------------------------------------------------
//  初期化
//  TORIAEZU:
//      初期化時は POWER ボタンが押されていないと仮定する。
//      仮に押されていたとしても長押しとして検知しない。
//
void PowerButton::Initialize(nn::spsm::server::PowerStateMessageQueue* pPowerStateMessageQueue) NN_NOEXCEPT
{
    m_pPowerStateMessageQueue = pPowerStateMessageQueue;

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    hid::system::BindSleepButtonEvent(m_PowerButtonEvent.GetBase(), os::EventClearMode_AutoClear);
    m_PowerButtonEvent.Clear();
#endif

    m_TickShortPressing  = os::ConvertToTick(GetTimeSpanShortPressing());
    m_TickMiddlePressing = os::ConvertToTick(GetTimeSpanMiddlePressing());
    m_TickLongPressing   = os::ConvertToTick(GetTimeSpanLongPressing());

    os::InitializeMultiWaitHolder(&m_TimerEventHolder,       m_TimerEvent.GetBase());
    os::InitializeMultiWaitHolder(&m_PowerButtonEventHolder, m_PowerButtonEvent.GetBase());

    m_TimerEventHolder.userData       = reinterpret_cast<uintptr_t>(this);
    m_PowerButtonEventHolder.userData = reinterpret_cast<uintptr_t>(this);
}


//-----------------------------------------------------------------------------
//  デストラクタ
//
void PowerButton::Finalize() NN_NOEXCEPT
{
    os::FinalizeMultiWaitHolder(&m_TimerEventHolder);
    os::FinalizeMultiWaitHolder(&m_PowerButtonEventHolder);
}


//-----------------------------------------------------------------------------
//  TimerEvent ハンドラ関数（多重待ち解除時に呼ばれる）
//
void PowerButton::TimerEventHandler() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    if (!m_TimerEvent.TryWait())
    {
        return;
    }

    if (m_State)
    {
        auto interval = os::GetSystemTick() - m_StartTick;

        // ホールド時間によってメッセージを変える
        if (interval >= m_TickLongPressing)
        {
            m_TimerEvent.Stop();

            // 7sec 通過時点で長押しを検知
            NotifyLongPressingPowerButton();
        }
        else if (interval >= m_TickMiddlePressing)
        {
            // 3sec 通過時点で中押しを検知
            NotifyMiddlePressingPowerButton();
        }
    }
}

//-----------------------------------------------------------------------------
//  PowerButtonEvent ハンドラ関数（多重待ち解除時に呼ばれる）
//
void PowerButton::PowerButtonEventHandler() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    if (!m_PowerButtonEvent.TryWait())
    {
        return;
    }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    hid::system::SleepButtonState   powerButtonState;
    int count = hid::system::GetSleepButtonStates(&powerButtonState, 1);
    NN_SDK_ASSERT(count == 1);
    NN_UNUSED(count);

    if ( powerButtonState.buttons.Test<hid::system::SleepButton::Active>() )
    {
        // 押下エッジ検出なら時間計測開始
        m_StartTick = os::GetSystemTick();
        m_State     = true;
        m_TimerEvent.Clear();
        // NN_DETAIL_SPSM_INFO_V1("%s: ON, start time=%d ms\n", __FUNCTION__, nn::os::ConvertToTimeSpan(m_StartTick).GetMilliSeconds());

        auto firstSpan  = GetTimeSpanMiddlePressing();
        auto secondSpan = GetTimeSpanLongPressing() - firstSpan;
        m_TimerEvent.StartPeriodic( firstSpan, secondSpan );

        NotifyStartPressingPowerButton();
    }
    else
    {
        // 開放エッジ検出なら時間計測停止
        m_TimerEvent.Stop();
        // NN_DETAIL_SPSM_INFO_V1("%s: OFF, start time=%d ms\n", __FUNCTION__, nn::os::ConvertToTimeSpan(m_StartTick).GetMilliSeconds());

        // ボタンが押されていたら、その時間によってメッセージを通知
        if (m_State)
        {
            auto interval = os::GetSystemTick() - m_StartTick;
            // NN_DETAIL_SPSM_INFO_V1("%s: OFF, interval=%d ms\n", __FUNCTION__, nn::os::ConvertToTimeSpan(interval).GetMilliSeconds());

            // ここでは 30msec～3sec 区間の時のみ短押しのメッセージを通知
            if (m_TickShortPressing <= interval &&
                                       interval < m_TickMiddlePressing)
            {
                NotifyShortPressingPowerButton();
            }
        }
        m_State = false;
    }
#endif
}

//-----------------------------------------------------------------------------
//  EnablePowerButtonCounterEvent ハンドラ関数（多重待ち解除時に呼ばれる）
//
void PowerButton::EnablePowerButtonCounterEventHandler() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if ( m_IsCountEnabled )
    {
        return;
    }
    m_IsCountEnabled = true;

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // SIGLO-49234: 電源ボタン取得が有効になった後、hid が最低一回サンプリングするのを待つ
    // nn::hid::detail::HomeButtonManager::SamplingInterval の値を使用
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));

    // 有効なボタン状態の値を使ってステートを再初期化
    hid::system::SleepButtonState   powerButtonState;
    int count = hid::system::GetSleepButtonStates(&powerButtonState, 1);
    NN_SDK_ASSERT(count == 1);
    NN_UNUSED(count);

    m_State = powerButtonState.buttons.Test<hid::system::SleepButton::Active>();
#endif

    if ( m_State )
    {
        // 押下状態なら、時間計測をやり直す（タイムスタンプもリセット）
        m_TimerEvent.Stop();
        m_TimerEvent.Clear();
        m_StartTick = os::GetSystemTick();
        // NN_DETAIL_SPSM_INFO_V1("%s: ON, start time=%d ms\n", __FUNCTION__, nn::os::ConvertToTimeSpan(m_StartTick).GetMilliSeconds());

        auto firstSpan = GetTimeSpanMiddlePressing();
        auto secondSpan = GetTimeSpanLongPressing() - firstSpan;
        m_TimerEvent.StartPeriodic(firstSpan, secondSpan);
    }
}

//-----------------------------------------------------------------------------
//  DisablePowerButtonCounterEvent ハンドラ関数（多重待ち解除時に呼ばれる）
//
void PowerButton::DisablePowerButtonCounterEventHandler() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if ( !m_IsCountEnabled )
    {
        return;
    }
    m_IsCountEnabled = false;

    // hid の状態変化を検出して呼ばれるハンドラではないので、
    // ここでは m_State を変更せず、タイマーの状態だけを操作するように注意する
    if ( m_State )
    {
        // 押下状態だったら、タイマーを止めておく
        m_TimerEvent.Stop();
        m_TimerEvent.Clear();
        NN_DETAIL_SPSM_INFO_V1("%s: ON, start time=%d ms\n", __FUNCTION__, nn::os::ConvertToTimeSpan(m_StartTick).GetMilliSeconds());
    }
}

}}} // namespace nn::spsm::observer

