﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/kpr/kpr_KeyCode.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_ISettingsServer.sfdl.h>
#include <nn/settings/settings_Keyboard.h>
#include <nn/settings/settings_Language.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_Out.h>
#include <nn/util/util_StringUtil.h>

#include "settings_KeyCodeMapDatabase.h"
#include "settings_SettingsServer.h"
#include "settings_StaticObject.h"
#include "settings_SystemSettingsDatabase.h"

namespace nn { namespace settings { namespace detail {

namespace {

//!< 互換性維持のための定義を表す構造体です。
template<size_t Major, size_t Minor, size_t Micro>
struct CompatibleDefinition;

//!< 互換性維持のための定義を表す構造体です。(4.0.0 NUP)
template<>
struct CompatibleDefinition<4, 0, 0>
{
    //!< 利用可能な言語コードの数を返します。
    static inline int32_t GetAvailableLanguageCodeCount() NN_NOEXCEPT
    {
        return 15;
    }
};

//!< キーボード配列の種別を返します。
KeyboardLayout GetKeyboardLayout(const LanguageCode& code) NN_NOEXCEPT
{
    const int count = NN_ARRAY_SIZE(code.string);

    if (::nn::util::Strncmp(code.string, "ja", count) == 0)
    {
        return KeyboardLayout_Japanese;
    }

    if (::nn::util::Strncmp(code.string, "fr", count) == 0)
    {
        return KeyboardLayout_French;
    }

    if (::nn::util::Strncmp(code.string, "de", count) == 0)
    {
        return KeyboardLayout_German;
    }

    if (::nn::util::Strncmp(code.string, "it", count) == 0)
    {
        return KeyboardLayout_Italian;
    }

    if (::nn::util::Strncmp(code.string, "es", count) == 0)
    {
        return KeyboardLayout_Spanish;
    }

    if (::nn::util::Strncmp(code.string, "ko", count) == 0)
    {
        return KeyboardLayout_Korean;
    }

    if (::nn::util::Strncmp(code.string, "nl", count) == 0)
    {
        return KeyboardLayout_EnglishUsInternational;
    }

    if (::nn::util::Strncmp(code.string, "pt", count) == 0)
    {
        return KeyboardLayout_Portuguese;
    }

    if (::nn::util::Strncmp(code.string, "ru", count) == 0)
    {
        return KeyboardLayout_Russian;
    }

    if (::nn::util::Strncmp(code.string, "en-GB", count) == 0)
    {
        return KeyboardLayout_EnglishUk;
    }

    if (::nn::util::Strncmp(code.string, "fr-CA", count) == 0)
    {
        return KeyboardLayout_FrenchCa;
    }

    if (::nn::util::Strncmp(code.string, "es-419", count) == 0)
    {
        return KeyboardLayout_SpanishLatin;
    }

    if (::nn::util::Strncmp(code.string, "zh-Hans", count) == 0)
    {
        return KeyboardLayout_ChineseSimplified;
    }

    if (::nn::util::Strncmp(code.string, "zh-Hant", count) == 0)
    {
        return KeyboardLayout_ChineseTraditional;
    }

    return KeyboardLayout_EnglishUs;
}

} // namespace

SettingsServer::SettingsServer() NN_NOEXCEPT
{
    //何もしない
}

::nn::Result SettingsServer::GetLanguageCode(
    ::nn::sf::Out<::nn::settings::LanguageCode> outValue) NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetSystemSettingsLanguageCode(outValue.GetPointer()));
    NN_RESULT_SUCCESS;
}

::nn::Result SettingsServer::GetAvailableLanguageCodes(
    ::nn::sf::Out<int32_t> outValue,
    const ::nn::sf::OutArray<::nn::settings::LanguageCode>& outCodes
    ) NN_NOEXCEPT
{
    const int32_t count = ::std::min(
        static_cast<int32_t>(outCodes.GetLength()),
        CompatibleDefinition<4, 0, 0>::GetAvailableLanguageCodeCount());
    NN_RESULT_DO(
        GetSystemSettingsAvailableLanguageCodes(
            outValue.GetPointer(), outCodes.GetData(), count));
    NN_RESULT_SUCCESS;
}

::nn::Result SettingsServer::MakeLanguageCode(
    ::nn::sf::Out<::nn::settings::LanguageCode> outValue,
    int32_t value) NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetSystemSettingsAvailableLanguageCode(outValue.GetPointer(), value));
    NN_RESULT_SUCCESS;
}

::nn::Result SettingsServer::GetAvailableLanguageCodeCount(
    ::nn::sf::Out<int32_t> outValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        outValue.GetPointer() != nullptr, ResultNullLanguageCodeCountBuffer());
    outValue.Set(
        CompatibleDefinition<4, 0, 0>::GetAvailableLanguageCodeCount());
    NN_RESULT_SUCCESS;
}

::nn::Result SettingsServer::GetRegionCode(
    ::nn::sf::Out<int32_t> outValue) NN_NOEXCEPT
{
    NN_RESULT_DO(GetSystemSettingsRegionCode(outValue.GetPointer()));
    NN_RESULT_SUCCESS;
}

::nn::Result SettingsServer::GetAvailableLanguageCodes2(
    ::nn::sf::Out<int32_t> outValue,
    const ::nn::sf::OutArray<::nn::settings::LanguageCode>& outCodes
    ) NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetSystemSettingsAvailableLanguageCodes(
            outValue.GetPointer(),
            outCodes.GetData(),
            static_cast<int32_t>(outCodes.GetLength())));
    NN_RESULT_SUCCESS;
}

::nn::Result SettingsServer::GetAvailableLanguageCodeCount2(
    ::nn::sf::Out<int32_t> outValue) NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetSystemSettingsAvailableLanguageCodeCount(outValue.GetPointer()));
    NN_RESULT_SUCCESS;
}

::nn::Result SettingsServer::GetKeyCodeMap(
    ::nn::sf::Out<::nn::kpr::KeyCodeMap> outValue) NN_NOEXCEPT
{
    LanguageCode code = {};
    NN_RESULT_DO(
        GetSystemSettingsLanguageCode(&code));
    NN_RESULT_DO(
        ::nn::settings::detail::GetKeyCodeMap(
            outValue.GetPointer(), GetKeyboardLayout(code), 1));
    NN_RESULT_SUCCESS;
}

::nn::Result SettingsServer::GetQuestFlag(
    ::nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    NN_RESULT_DO(GetSystemSettingsQuestFlag(outValue.GetPointer()));
    NN_RESULT_SUCCESS;
}

::nn::Result SettingsServer::GetKeyCodeMap2(
    ::nn::sf::Out<::nn::kpr::KeyCodeMap> outValue) NN_NOEXCEPT
{
    LanguageCode code = {};
    NN_RESULT_DO(
        GetSystemSettingsLanguageCode(&code));
    NN_RESULT_DO(
        ::nn::settings::detail::GetKeyCodeMap(
            outValue.GetPointer(), GetKeyboardLayout(code), 2));
    NN_RESULT_SUCCESS;
}

::nn::Result CreateSettingsServerProxy(
    ::nn::sf::SharedPointer<ISettingsServer>* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    *pOutValue = StaticObject<
        ::nn::sf::UnmanagedServiceObject<ISettingsServer, SettingsServer>
        >::Get().GetShared();
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::settings::detail
