﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/settings/settings_ISettingsServer.sfdl.h>
#include <nn/settings/settings_ISystemSettingsServer.sfdl.h>
#include <nn/settings/settings_Language.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_Out.h>
#include <nn/TargetConfigs/build_Base.h>

#include "settings_LanguageImpl.h"
#include "settings_LockableMutexType.h"
#include "settings_SettingsServer.h"
#include "settings_SystemSettingsServer.h"

namespace nn { namespace settings { namespace detail {

::nn::Result MakeLanguageCode(
    LanguageCode* pOutValue, Language value) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullLanguageCodeBuffer());

    static int32_t s_Count = 0;

    static LanguageCode s_LanguageCodes[64] = {};

    NN_FUNCTION_LOCAL_STATIC(::std::atomic<bool>, s_IsInitialized, (false));

    if (!s_IsInitialized)
    {
        static LockableMutexType s_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

        ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

        if (!s_IsInitialized)
        {
            NN_RESULT_DO(
                GetAvailableLanguageCodes(
                    &s_Count, s_LanguageCodes, NN_ARRAY_SIZE(s_LanguageCodes)));

            s_IsInitialized = true;
        }
    }

    const auto index = static_cast<int>(value);

    if (0 <= index && index < s_Count)
    {
        *pOutValue = s_LanguageCodes[index];
    }
    else
    {
        ::nn::sf::SharedPointer<ISettingsServer> pProxy;
        NN_RESULT_DO(CreateSettingsServerProxy(&pProxy));
        NN_RESULT_DO(pProxy->MakeLanguageCode(pOutValue, index));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result GetLanguageCode(LanguageCode* pOutValue) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<ISettingsServer> pProxy;
    NN_RESULT_DO(CreateSettingsServerProxy(&pProxy));
    NN_RESULT_DO(pProxy->GetLanguageCode(pOutValue));
    NN_RESULT_SUCCESS;
}

::nn::Result SetLanguageCode(const LanguageCode& value) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<ISystemSettingsServer> pProxy;
    NN_RESULT_DO(CreateSystemSettingsServerProxy(&pProxy));
    NN_RESULT_DO(pProxy->SetLanguageCode(value));
    NN_RESULT_SUCCESS;
}

::nn::Result GetAvailableLanguageCodes(
    int32_t* pOutCount, LanguageCode* pOutValues, int count) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<ISettingsServer> pProxy;
    NN_RESULT_DO(CreateSettingsServerProxy(&pProxy));
    NN_RESULT_DO(
        pProxy->GetAvailableLanguageCodes2(
            pOutCount,
            ::nn::sf::OutArray<LanguageCode>(
                pOutValues,
                count <= 0 ? 0 : static_cast<size_t>(count))));
    NN_RESULT_SUCCESS;
}

::nn::Result GetAvailableLanguageCodeCount(int32_t* pOutValue) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<ISettingsServer> pProxy;
    NN_RESULT_DO(CreateSettingsServerProxy(&pProxy));
    NN_RESULT_DO(pProxy->GetAvailableLanguageCodeCount2(pOutValue));
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::settings::detail
