﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/settings/settings_ServiceTypes.h>

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include "settings_SystemSaveData-os.horizon.h"
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
#include "settings_SystemSaveData-os.win.h"
#endif

namespace nn { namespace settings { namespace detail {

//!< キーバリューストアのキーイテレータを表す構造体です。
struct KeyValueStoreKeyIterator final
{
    size_t headerSize;  //!< マップのキーのヘッダサイズ
    size_t entireSize;  //!< マップのキーの全体サイズ (NULL 終端文字込み)
    char* mapKey;       //!< マップのキー
};

//!< キーバリューストアの項目を表す構造体です。（デバッグ用）
struct KeyValueStoreItemForDebug final
{
    const char* mapKey;         //!< マップのキー
    uint8_t type;               //!< マップの値の種別
    ::nn::Bit8 _padding[7];
    size_t currentSize;         //!< マップの現在値のサイズ
    size_t defaultSize;         //!< マップの既定値のサイズ
    const void* currentValue;   //!< マップの現在値
    const void* defaultValue;   //!< マップの既定値
};

//!< キーバリューストアを扱うクラスです。
class KeyValueStore final
{
    NN_DISALLOW_COPY(KeyValueStore);
    NN_DISALLOW_MOVE(KeyValueStore);

private:
    const SettingsName& m_Name; //!< 設定の名前

public:
    explicit KeyValueStore(const SettingsName& name) NN_NOEXCEPT;

    //!< キーバリューストアから値のサイズを取得します。
    ::nn::Result GetValueSize(
        uint64_t* pOutCount, const SettingsItemKey& key) NN_NOEXCEPT;

    //!< キーバリューストアから値を取得します。
    ::nn::Result GetValue(
        uint64_t* pOutCount, char* outBuffer, size_t count,
        const SettingsItemKey& key) NN_NOEXCEPT;

    //!< キーバリューストアへ値を設定します。
    ::nn::Result SetValue(
        const SettingsItemKey& key, const void* buffer, size_t count
        ) NN_NOEXCEPT;

    //!< キーバリューストアの値をリセットします。
    ::nn::Result ResetValue(const SettingsItemKey& key) NN_NOEXCEPT;

    //!< キーイテレータを作成します。
    ::nn::Result CreateKeyIterator(
        KeyValueStoreKeyIterator* pOutValue) NN_NOEXCEPT;
};

//!< キーバリューストアのキーイテレータを破棄します。
void DestroyKeyValueStoreKeyIterator(
    KeyValueStoreKeyIterator* pOutValue) NN_NOEXCEPT;

//!< キーバリューストアのキーイテレータを進めます。
::nn::Result AdvanceKeyValueStoreKeyIterator(
    KeyValueStoreKeyIterator* pOutValue) NN_NOEXCEPT;

//!< キーバリューストアのキーイテレータを進めます。
::nn::Result GetKeyValueStoreKeyIteratorKeySize(
    uint64_t* pOutCount,
    const KeyValueStoreKeyIterator& iterator) NN_NOEXCEPT;

//!< キーバリューストアのキーイテレータを進めます。
::nn::Result GetKeyValueStoreKeyIteratorKey(
    uint64_t* pOutCount, char* outBuffer, size_t count,
    const KeyValueStoreKeyIterator& iterator) NN_NOEXCEPT;

//!< キーバリューストアの項目の数を取得します。（デバッグ用）
::nn::Result GetKeyValueStoreItemCountForDebug(uint64_t* pOutCount) NN_NOEXCEPT;

//!< キーバリューストアの項目を取得します。（デバッグ用）
::nn::Result GetKeyValueStoreItemForDebug(
    uint64_t* pOutCount,
    KeyValueStoreItemForDebug outItems[], size_t count) NN_NOEXCEPT;

//!< キーバリューストアの項目を追加します。（デバッグ用）
::nn::Result AddKeyValueStoreItemForDebug(
    const KeyValueStoreItemForDebug items[], size_t count) NN_NOEXCEPT;

//!< キーバリューストアの全情報をセーブします。（デバッグ用）
::nn::Result SaveKeyValueStoreAllForDebug(SystemSaveData* pData) NN_NOEXCEPT;

//!< キーバリューストアの再読み込みを行います。（デバッグ用）
::nn::Result ReloadKeyValueStoreForDebug() NN_NOEXCEPT;

//!< キーバリューストアの再読み込みを行います。（デバッグ用）
::nn::Result ReloadKeyValueStoreForDebug(
    SystemSaveData* pSystemSaveData,
    SystemSaveData* pFwdbgSystemData,
    SystemSaveData* pPfCfgSystemData) NN_NOEXCEPT;

//!< キーバリューストアのセーブデータを読み出します。
::nn::Result ReadKeyValueStoreSaveData(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT;

//!< キーバリューストアのファームウェアデバッグ設定を読み出します。
::nn::Result ReadKeyValueStoreFirmwareDebug(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT;

//!< キーバリューストアのプラットフォーム構成情報を読み出します。
::nn::Result ReadKeyValueStorePlatformConfigration(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT;

//!< キーバリューストアのセーブデータをリセットします。
::nn::Result ResetKeyValueStoreSaveData() NN_NOEXCEPT;

}}} // namespace nn::settings::detail
