﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cctype>
#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>

#include "settings_Bsp0-os.horizon.h"
#include "settings_StringBuilder.h"

namespace nn { namespace settings { namespace detail {

namespace {

//!< bsp0 形式の設定の名前
const char* const Bsp0SettingsName = "bsp0";

//!< bsp0 形式の設定ファイルの名前
const char* const Bsp0SettingsFileName = "bsp0.conf";

//!< bsp0 形式の設定のヘッダ行を表すシンボル
const char Bsp0SettingsHeader[] = "BSP0_CONFIG_START";

//!< bsp0 形式の設定のフッタ行を表すシンボル
const char Bsp0SettingsFooter[] = "BSP0_CONFIG_END";

//!< bsp0 の設定項目を表す構造体です。
struct Bsp0SettingsItem final
{
    const char* key;    //!< bsp0 の設定項目のキー
    const char* value;  //!< bsp0 の設定項目の値
};

//!< bsp0 の組み込み設定です。
const Bsp0SettingsItem BuiltInBsp0SettingsItems[] =
{
    { "dmnt0_use_htclow0", "TRUE" },
    { "fs0_use_htclow0", "TRUE" },
    { "htclow0_use_usb", "TRUE" }
};

//!< ワードを構成する文字であるか否かを表す値を返します。
bool IsCharacterOfWord(char c) NN_NOEXCEPT;

//!< 改行位置を返します。
const char* SearchLineEnd(const char* line, const char* eol) NN_NOEXCEPT;

//!< 非スペース位置を返します。
const char* SearchNonSpace(const char* line, const char* eol) NN_NOEXCEPT;

//!< 非ワード位置を返します。
const char* SearchNonWord(const char* line, const char* eol) NN_NOEXCEPT;

//!< 等号位置を返します。
const char* SearchEqual(const char* line, const char* eol) NN_NOEXCEPT;

//!< シンボル行か否かを表す値を返します。
bool IsSymbolLine(const char* line,
                  const char* eol,
                  const char* symbol,
                  size_t size) NN_NOEXCEPT;

//!< ヘッダ行か否かを表す値を返します。
bool IsHeaderLine(const char* line, const char* eol) NN_NOEXCEPT;

//!< フッタ行か否かを表す値を返します。
bool IsFooterLine(const char* line, const char* eol) NN_NOEXCEPT;

//!< 行をパースします。
bool ParseLine(StringBuilder<Bsp0SettingsFile::KeySizeMax>* pOutKey,
               StringBuilder<Bsp0SettingsFile::ValueSizeMax>* pOutValue,
               const char* line,
               const char* eol) NN_NOEXCEPT;

} // namespace

Bsp0SettingsFile::Bsp0SettingsFile(const char* buffer, size_t size) NN_NOEXCEPT
    : m_IsHeaderFound(false)
    , m_Ptr(buffer)
    , m_Eof(buffer + size)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
}

const char* Bsp0SettingsFile::GetName() NN_NOEXCEPT
{
    return Bsp0SettingsFileName;
}

bool Bsp0SettingsFile::ParseItem(
    StringBuilder<Bsp0SettingsFile::KeySizeMax>* pOutKey,
    StringBuilder<Bsp0SettingsFile::ValueSizeMax>* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutKey);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    while (m_Ptr < m_Eof)
    {
        const char* eol = SearchLineEnd(m_Ptr, m_Eof);

        if (!m_IsHeaderFound)
        {
            if (IsHeaderLine(m_Ptr, eol))
            {
                m_IsHeaderFound = true;
            }

            m_Ptr = eol + 1;
        }
        else if (IsFooterLine(m_Ptr, eol))
        {
            m_Ptr = m_Eof;

            break;
        }
        else
        {
            ParseLine(pOutKey, pOutValue, m_Ptr, eol);

            m_Ptr = eol + 1;

            if (pOutKey->GetLength() > 0)
            {
                return true;
            }
        }
    }

    return false;
}

const char* GetBsp0SettingsName() NN_NOEXCEPT
{
    return Bsp0SettingsName;
}

bool GetBuiltInBsp0SettingsItemValueSize(uint64_t* pOutValue,
                                         const char* name,
                                         const char* key) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(name);
    NN_SDK_REQUIRES_NOT_NULL(key);

    if (::std::strcmp(name, Bsp0SettingsName) == 0)
    {
        for (const Bsp0SettingsItem& item : BuiltInBsp0SettingsItems)
        {
            if (::std::strcmp(key, item.key) == 0)
            {
                *pOutValue = ::std::strlen(item.value) + 1;

                return true;
            }
        }
    }

    return false;
}

bool GetBuiltInBsp0SettingsItemValue(uint64_t* pOutValue,
                                     char* outBuffer,
                                     size_t outBufferSize,
                                     const char* name,
                                     const char* key) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES_NOT_NULL(name);
    NN_SDK_REQUIRES_NOT_NULL(key);
    NN_UNUSED(outBufferSize);

    if (::std::strcmp(name, Bsp0SettingsName) == 0)
    {
        for (const Bsp0SettingsItem& item : BuiltInBsp0SettingsItems)
        {
            if (::std::strcmp(key, item.key) == 0)
            {
                size_t length = ::std::strlen(item.value) + 1;

                *pOutValue = length;

                ::std::strncpy(outBuffer, item.value, length);

                return true;
            }
        }
    }

    return false;
}

namespace {

bool IsCharacterOfWord(char c) NN_NOEXCEPT
{
    return (!::std::isspace(c) && c != '=');
}

const char* SearchLineEnd(const char* line, const char* eol) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(line);
    NN_SDK_REQUIRES_NOT_NULL(eol);
    NN_SDK_REQUIRES(line <= eol);
    return ::std::find_if(line,
                          eol,
                          [](char c) { return (c == '\r' || c == '\n'); });

}

const char* SearchNonSpace(const char* line, const char* eol) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(line);
    NN_SDK_REQUIRES_NOT_NULL(eol);
    NN_SDK_REQUIRES(line <= eol);
    return ::std::find_if(line, eol, [](char c) { return !::std::isspace(c); });
}

const char* SearchNonWord(const char* line, const char* eol) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(line);
    NN_SDK_REQUIRES_NOT_NULL(eol);
    NN_SDK_REQUIRES(line <= eol);
    return ::std::find_if(line,
                          eol,
                          [](char c) { return !IsCharacterOfWord(c); });
}

const char* SearchEqual(const char* line, const char* eol) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(line);
    NN_SDK_REQUIRES_NOT_NULL(eol);
    NN_SDK_REQUIRES(line <= eol);
    return ::std::find(line, eol, '=');
}

bool IsSymbolLine(const char* line,
                  const char* eol,
                  const char* symbol,
                  size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(line);
    NN_SDK_REQUIRES_NOT_NULL(eol);
    NN_SDK_REQUIRES_NOT_NULL(symbol);
    NN_SDK_REQUIRES(line <= eol);

    line = SearchNonSpace(line, eol);

    if (static_cast<size_t>(eol - line) < size)
    {
        return false;
    }

    if (::std::strncmp(symbol, line, size) != 0)
    {
        return false;
    }

    line += size;

    line = SearchNonSpace(line, eol);

    return (line == eol);
}

bool IsHeaderLine(const char* line, const char* eol) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(line);
    NN_SDK_REQUIRES_NOT_NULL(eol);
    NN_SDK_REQUIRES(line <= eol);
    return IsSymbolLine(line,
                        eol,
                        Bsp0SettingsHeader,
                        sizeof(Bsp0SettingsHeader) - 1);
}

bool IsFooterLine(const char* line, const char* eol) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(line);
    NN_SDK_REQUIRES_NOT_NULL(eol);
    NN_SDK_REQUIRES(line <= eol);
    return IsSymbolLine(line,
                        eol,
                        Bsp0SettingsFooter,
                        sizeof(Bsp0SettingsFooter) - 1);
}

bool ParseLine(StringBuilder<Bsp0SettingsFile::KeySizeMax>* pOutKey,
               StringBuilder<Bsp0SettingsFile::ValueSizeMax>* pOutValue,
               const char* line,
               const char* eol) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutKey);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(line);
    NN_SDK_REQUIRES_NOT_NULL(eol);
    NN_SDK_REQUIRES(line <= eol);

    pOutKey->Clear();
    pOutValue->Clear();

    // 行頭の空白をスキップ
    line = SearchNonSpace(line, eol);
    if (line == eol)
    {
        // 空行ならばスキップ
        return true;
    }

    // キーの開始位置であることを確認
    if (!IsCharacterOfWord(*line))
    {
        return false;
    }

    const char* key = line;

    // キーの終了位置を探索
    line = SearchNonWord(line, eol);
    if (line == eol)
    {
        return false;
    }

    const char* eok = line;

    // セパレータをスキップ
    line = SearchEqual(line, eol);
    if (line == eol)
    {
        return false;
    }

    line++;

    line = SearchNonSpace(line, eol);
    if (line == eol)
    {
        return false;
    }

    // 値の開始位置であることを確認
    if (!IsCharacterOfWord(*line))
    {
        return false;
    }

    const char* value = line;

    // 値の終了位置を探索
    line = SearchNonWord(line, eol);
    const char* eov = line;

    // 行末の空白をスキップ
    line = SearchNonSpace(line, eol);
    if (line != eol)
    {
        return false;
    }

    while (key < eok)
    {
        const char c = *(key++);
        pOutKey->Append((c < 'A' || 'Z' < c) ? c : c - 'A' + 'a');
    }

    while (value < eov)
    {
        pOutValue->Append(*(value++));
    }

    return true;
}

} // namespace

}}} // namespace nn::settings::detail
