﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/os.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/sasbus/driver/sasbus.h>

#include <nn/sasbus/sasbus_IManager.sfdl.h>
#include <nn/sasbus/sasbus_ISession.sfdl.h>
#include <nn/sasbus/sasbus_ServiceType.h>
#include <nn/sasbus/sasbus_PeriodicReceiveModeType.h>

#include <nn/sf/impl/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/sasbus/server/sasbus_ManagerImpl.h>
#include <nn/sasbus/detail/sasbus_RingLifo.h>


namespace nn { namespace sasbus { namespace server {

// TransferMemory の Utility クラス
// Map 済みならデストラクタで自動的に Unmap する。
class SasbusTransferMemory
{
public:
    explicit SasbusTransferMemory() NN_NOEXCEPT
        :  m_MappedAddress(nullptr)
    {}

    ~SasbusTransferMemory() NN_NOEXCEPT
    {
        if(m_MappedAddress != nullptr)
        {
            m_TransferMemory.Unmap();
        }
    }

    //--------------------------------------------------------------------------
    /**
     * @brief   既存の Transfer memory ハンドルを関連付け、同時に Map もします。
     *
     * @param[in]  ownerPermission Transfer memory を生成する際に満たすべきアクセス権
     * @param[in]  size            Transfer memory のサイズ
     * @param[in]  handle          Transfer memory のハンドル
     * @param[in]  managed         Transfer memory のハンドル管理フラグ
     *
     * @retresult
     *   @handleresult{nn::os::ResultInvalidHandle}
     *   @handleresult{nn::os::ResultInvalidTransferMemoryState}
     *   @handleresult{nn::os::ResultInvalidTransferMemorySize}
     *   @handleresult{nn::os::ResultOutOfAddressSpace}
     * @endretresult
     *
     * @details
     *  指定されたハンドルを使って Transfer memory オブジェクトを初期化します。@n
     *  詳細は @ref nn::os::AttachTransferMemory 、@ref nn::os::MapTransferMemory を参照して下さい。
     */
    nn::Result AttachAndMap(nn::os::MemoryPermission ownerPermission, size_t size, nn::os::NativeHandle handle, bool managed) NN_NOEXCEPT
    {
        void* mappedAddress;

        m_TransferMemory.Attach(size, handle, managed);
        NN_RESULT_DO(m_TransferMemory.Map(&mappedAddress, ownerPermission));
        m_MappedAddress = mappedAddress;

        return nn::ResultSuccess();
    }

    void UnmapAndDestroy()
    {
        if (m_MappedAddress != nullptr)
        {
            m_TransferMemory.Unmap();
            m_TransferMemory.~TransferMemory();

        }
    }

    void* GetMappedAddress() const NN_NOEXCEPT
    {
        return m_MappedAddress;
    }

private:
    void*                   m_MappedAddress;
    nn::os::TransferMemory  m_TransferMemory;
};

// ISession 実装クラス
class SessionImpl
{
public:
    // コンストラクタで親の SharedPointer を設定し、セッションに必要な情報を渡す
    SessionImpl(ManagerImpl* pParent, std::int32_t device) NN_NOEXCEPT
        : m_Parent(pParent, true)
    {
        nn::sasbus::driver::OpenSession(&m_InternalSession, static_cast<nn::sasbus::SasbusDevice>(device));
    }

    // Session のデストラクタで Session のクローズを行う
    ~SessionImpl()
    {
        // すでにスレッドが止まっていた場合はそのまま先へ進む

        StopPeriodicReceiveMode();
        nn::sasbus::driver::CloseSession(&m_InternalSession);
    }

    nn::Result Send(const nn::sf::InBuffer& inData, char reg) NN_NOEXCEPT;
    nn::Result Receive(const nn::sf::OutBuffer& outData, char reg) NN_NOEXCEPT;
    nn::Result StartPeriodicReceiveMode(nn::sf::NativeHandle&& receiveBufferHandle, std::uint64_t receiveBufferSize, char reg, std::int32_t size, std::int64_t interval) NN_NOEXCEPT;
    nn::Result StopPeriodicReceiveMode() NN_NOEXCEPT;

private:

    // 親への SharedPointer
    nn::sf::SharedPointer<ManagerImpl> m_Parent;

    // SASBUS ドライバライブラリが持つ Session 構造体
    nn::sasbus::driver::Session m_InternalSession;

    // PeriodicReceiveMode 用の共有メモリ
    SasbusTransferMemory        m_ReceiveTransferMemory;
};

// ここから各 SASBUS API の内部実装
nn::Result SessionImpl::Send(const nn::sf::InBuffer& inData, char reg) NN_NOEXCEPT
{
    nn::sasbus::driver::Send(&m_InternalSession,
                         inData.GetPointerUnsafe(),
                         inData.GetSize(),
                         reg);

    return nn::ResultSuccess();
}

nn::Result SessionImpl::Receive(const nn::sf::OutBuffer& outData, char reg) NN_NOEXCEPT
{
    nn::sasbus::driver::Receive(outData.GetPointerUnsafe(),
                            &m_InternalSession,
                            outData.GetSize(),
                            reg);

    return nn::ResultSuccess();
}
nn::Result SessionImpl::StartPeriodicReceiveMode(nn::sf::NativeHandle&& receiveBufferHandle,
                                             std::uint64_t receiveBufferSize,
                                             char reg, std::int32_t size,
                                             std::int64_t microSecondsInterval) NN_NOEXCEPT
{
    NN_RESULT_DO(m_ReceiveTransferMemory.AttachAndMap(nn::os::MemoryPermission_ReadWrite,
        static_cast<size_t>(receiveBufferSize), receiveBufferHandle.GetOsHandle(), receiveBufferHandle.IsManaged()));
    receiveBufferHandle.Detach();

    NN_RESULT_DO(nn::sasbus::driver::StartPeriodicReceiveMode(&m_InternalSession, size, reg, nn::TimeSpan::FromMicroSeconds(microSecondsInterval), static_cast<char*>(m_ReceiveTransferMemory.GetMappedAddress()), receiveBufferSize));

    return nn::ResultSuccess();
}

nn::Result SessionImpl::StopPeriodicReceiveMode() NN_NOEXCEPT
{
    nn::sasbus::driver::StopPeriodicReceiveMode(&m_InternalSession);

    // 再度使われるときに初期化済みのハンドルのアタッチからやり直すため、ここでは破棄が必要
    m_ReceiveTransferMemory.UnmapAndDestroy();

    return nn::ResultSuccess();
}


// マネージャーの実装
ManagerImpl::ManagerImpl() NN_NOEXCEPT
{
    // コンストラクタ内で拡張ヒープを初期化
    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_NoOption);
    // アロケータにヒープハンドルをアタッチ
    m_Allocator.Attach(m_HeapHandle);
}

ManagerImpl::~ManagerImpl() NN_NOEXCEPT
{
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

nn::Result ManagerImpl::OpenSession(nn::sf::Out<nn::sf::SharedPointer<nn::sasbus::ISession>> outSession, std::int32_t device) NN_NOEXCEPT
{
    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy>  Factory;
    auto p = Factory::CreateSharedEmplaced<ISession, SessionImpl>(&m_Allocator, this, device);

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    return nn::ResultSuccess();
}

}}}
