﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pwm/detail/pwm_Log.h>

#include "pwm_DriverImplBlinkLed-soc.tegra-x1.h"
#include "pwm_PmcUtil.h"

namespace nn {
namespace pwm {
namespace driver {
namespace detail {

void DriverImplBlinkLed::UpdateRegister()
{
    NN_SDK_ASSERT(0 <= m_Duty && m_Duty <= MaxDuty);

    // Blink を無効化
    SetBitOff32(PmcCntrlPhysicalAddress, PmcRegister::Cntrl_BlinkEn::Mask);
    DummyRead(PmcCntrlPhysicalAddress);

    // Wait for minimum 2 32K cycles (See NSBG-6438)
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(62));

    // Disable の場合は Blink　を無効化したまま終了
    if (!m_Enabled)
    {
        return;
    }

    // Period (秒) を表す浮動小数点数に変換
    float period = (float) m_Period.GetMicroSeconds() / (1000 * 1000);

    // on / off 時間: (DataOn / DataOff のレジスタ値) * (1 / 2048) 秒
    uint32_t dataOn = static_cast<uint32_t>(period * 2048 * m_Duty / MaxDuty);
    uint32_t dataOff = static_cast<uint32_t>(period * 2048 * (MaxDuty - m_Duty) / MaxDuty);

    // dataOn を 0 に設定しても実際は 1/32k 秒 Hi が出力されるので、
    // on 時間を 0 にしたいときは Disable にする
    if (dataOn == 0)
    {
        return;
    }

    nn::util::BitPack32 reg;
    reg.Clear();
    reg.Set<PmcRegister::BlinkTimer_ForceBlink>(true);
    reg.Set<PmcRegister::BlinkTimer_DataOn>(dataOn);
    reg.Set<PmcRegister::BlinkTimer_DataOff>(dataOff);

    Write32(PmcBlinkTimerPhysicalAddress,  reg.GetMaskedBits(0xffffffff));
    DummyRead(PmcBlinkTimerPhysicalAddress);

    // Wait for minimum 2 32K cycles (See NSBG-6438)
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(62));

    // Blink　を有効化
    SetBitOn32(PmcCntrlPhysicalAddress, PmcRegister::Cntrl_BlinkEn::Mask);
    DummyRead(PmcCntrlPhysicalAddress);
}

void DriverImplBlinkLed::Open(ChannelSessionImpl& session) NN_NOEXCEPT
{
    NN_UNUSED(session);

    m_Period = nn::TimeSpan::FromSeconds(0);
    m_Duty = 0;
    m_Enabled = false;

    m_PreviousDpdPadsOrideBlink = GetBit32(PmcDpdPadsOridePhysicalAddress, PmcRegister::DpdPadsOride_Blink::Mask);
    SetBitOn32(PmcDpdPadsOridePhysicalAddress, PmcRegister::DpdPadsOride_Blink::Mask);

    UpdateRegister();
    m_Initialized = true;
}

void DriverImplBlinkLed::Close(ChannelSessionImpl& session) NN_NOEXCEPT
{
    SetEnabled(session, false);

    // DpdPadsOride_Blink を元の値に戻す
    if (!m_PreviousDpdPadsOrideBlink)
    {
        SetBitOff32(PmcDpdPadsOridePhysicalAddress, PmcRegister::DpdPadsOride_Blink::Mask);
    }

    m_Initialized = false;
}

void DriverImplBlinkLed::SetPeriod(ChannelSessionImpl& session, nn::TimeSpan period) NN_NOEXCEPT
{
    NN_UNUSED(session);

    m_Period = period;
    UpdateRegister();
}

nn::TimeSpan DriverImplBlinkLed::GetPeriod(ChannelSessionImpl& session) NN_NOEXCEPT
{
    NN_UNUSED(session);

    return m_Period;
}

void DriverImplBlinkLed::SetDuty(ChannelSessionImpl& session, int duty) NN_NOEXCEPT
{
    NN_UNUSED(session);

    if (duty < 0 || MaxDuty < duty)
    {
        NN_DETAIL_PWM_ERROR("Duty == %d is out of range.\n", duty);
        return;
    }
    m_Duty = duty;
    UpdateRegister();
}

int DriverImplBlinkLed::GetDuty(ChannelSessionImpl& session) NN_NOEXCEPT
{
    NN_UNUSED(session);

    return m_Duty;
}

void DriverImplBlinkLed::SetEnabled(ChannelSessionImpl& session, bool enable) NN_NOEXCEPT
{
    NN_UNUSED(session);

    m_Enabled = enable;
    UpdateRegister();
}

bool DriverImplBlinkLed::GetEnabled(ChannelSessionImpl& session) NN_NOEXCEPT
{
    NN_UNUSED(session);

    return m_Enabled;
}

} // detail
} // driver
} // pwm
} // nn
