﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/psm/psm_Api.h>
#include <nn/psm/psm_Internal.h>
#include <nn/psm/psm_IPsmServer.sfdl.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_SystemProcess.h>
#include <nn/sf/sf_ISharedObject.h>

#include "psm_PsmServerByHipc.h"

namespace nn { namespace psm {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
// DFC と HIPC とで共通でよい。
::nn::sf::SharedPointer<IPsmServer> g_Server;

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
struct StaticMutex
{
    ::nn::os::MutexType mutex;

    void lock() NN_NOEXCEPT
    {
        ::nn::os::LockMutex(&mutex);
    }

    void unlock() NN_NOEXCEPT
    {
        ::nn::os::UnlockMutex(&mutex);
    }
};

StaticMutex g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

::nn::sf::SharedPointer<IPsmSession> g_Session;
::nn::sf::NativeHandle g_Handle;
bool g_IsSessionInitialized = false;

} // namespace

#if defined(PSM_DETAIL_MANUFACTURE_BUILD)
void InitializeForManufacture();
void FinalizeForManufacture();
#endif

void Initialize() NN_NOEXCEPT
{
    {
        ::std::lock_guard<StaticMutex> lock(g_Mutex);

        if ( g_InitializeCount == 0 )
        {
            NN_SDK_ASSERT(!g_Server);

            g_Server = GetPsmServerByHipc();
        }

        g_InitializeCount++;
    }

#if defined(PSM_DETAIL_MANUFACTURE_BUILD)
    InitializeForManufacture();
#endif
}

void InitializeWith(
    ::nn::sf::SharedPointer<nn::psm::IPsmServer> server) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_SDK_ASSERT(g_InitializeCount == 0);

    g_Server = server;

    g_InitializeCount++;
}

nn::sf::SharedPointer<IPsmServer> GetInternalPsmServer() NN_NOEXCEPT
{
    return g_Server;
}

void Finalize() NN_NOEXCEPT
{
#if defined(PSM_DETAIL_MANUFACTURE_BUILD)
    FinalizeForManufacture();
#endif

    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    if(g_InitializeCount == 0)
    {
        NN_SDK_ASSERT(g_Server);
        // 共有ポインタへの nullptr の代入で解放できる。
        g_Session = nullptr;
        g_Server = nullptr;
        g_IsSessionInitialized = false;
    }
}

int GetBatteryChargePercentage() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    int batteryCharge;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_Server->GetBatteryChargePercentage(&batteryCharge));

    return batteryCharge;
}

double GetRawBatteryChargePercentage() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    double rawBatteryCharge;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_Server->GetRawBatteryChargePercentage(&rawBatteryCharge));

    return rawBatteryCharge;
}

double GetBatteryAgePercentage() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    double age;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_Server->GetBatteryAgePercentage(&age));

    return age;
}

ChargerType GetChargerType() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    int chargerType;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_Server->GetChargerType(&chargerType));

    return static_cast<ChargerType>(chargerType);
}

BatteryVoltageState GetBatteryVoltageState() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    int batteryVoltageState;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_Server->GetBatteryVoltageState(&batteryVoltageState));

    return static_cast<BatteryVoltageState>(batteryVoltageState);
}

void EnableBatteryCharging() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Server->EnableBatteryCharging());
}

void DisableBatteryCharging() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Server->DisableBatteryCharging());
}

bool IsBatteryChargingEnabled() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    bool isEnabled;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Server->IsBatteryChargingEnabled(&isEnabled));

    return isEnabled;
}

void AcquireControllerPowerSupply() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Server->AcquireControllerPowerSupply());
}

void ReleaseControllerPowerSupply() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Server->ReleaseControllerPowerSupply());
}

void OpenSession(Session* pOutSession) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutSession);
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    ::nn::sf::SharedPointer<IPsmSession> handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Server->OpenSession(&handle));

    pOutSession->_pHandle = handle.Detach();
    pOutSession->_pEvent = nullptr;
}

void CloseSession(Session* pSession) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pSession);

    if (pSession->_pEvent != nullptr)
    {
        UnbindStateChangeEvent(pSession);
    }

    ::nn::sf::ReleaseSharedObject(pSession->_pHandle);
}

void BindStateChangeEvent(
    nn::os::SystemEventType* pEvent,
    Session* pSession) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    ::nn::sf::NativeHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pSession->_pHandle->BindStateChangeEvent(&handle));

    ::nn::os::AttachReadableHandleToSystemEvent(
        pEvent,
        handle.GetOsHandle(),
        handle.IsManaged(),
        ::nn::os::EventClearMode_ManualClear);
    handle.Detach();

    pSession->_pEvent = pEvent;
}

void SetChargerTypeChangeEventEnabled(Session* pSession, bool isEnabled) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pSession);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pSession->_pHandle->SetChargerTypeChangeEventEnabled(isEnabled));
}

void SetPowerSupplyChangeEventEnabled(Session* pSession, bool isEnabled) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pSession);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pSession->_pHandle->SetPowerSupplyChangeEventEnabled(isEnabled));
}

void SetBatteryVoltageStateChangeEventEnabled(Session* pSession, bool isEnabled) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pSession);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pSession->_pHandle->SetBatteryVoltageStateChangeEventEnabled(isEnabled));
}

void UnbindStateChangeEvent(Session* pSession) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pSession->_pEvent);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pSession->_pHandle->UnbindStateChangeEvent());

    ::nn::os::DestroySystemEvent(pSession->_pEvent);

    pSession->_pEvent = nullptr;
}

void EnableEnoughPowerChargeEmulation() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_Server->EnableEnoughPowerChargeEmulation());
}

void DisableEnoughPowerChargeEmulation() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_Server->DisableEnoughPowerChargeEmulation());
}

void EnableFastBatteryCharging() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_Server->EnableFastBatteryCharging());
}

void DisableFastBatteryCharging() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_Server->DisableFastBatteryCharging());
}

void GetChargerTypeChangeEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    if (!g_IsSessionInitialized)
    {
        g_Server->OpenSession(&g_Session);
        g_Session->SetChargerTypeChangeEventEnabled(true);
        g_Session->SetPowerSupplyChangeEventEnabled(false);
        g_Session->SetBatteryVoltageStateChangeEventEnabled(false);
        g_Session->BindStateChangeEvent(&g_Handle);
        g_IsSessionInitialized = true;
    }

    ::nn::os::AttachReadableHandleToSystemEvent(
        pEvent,
        g_Handle.GetOsHandle(),
        g_Handle.IsManaged(),
        ::nn::os::EventClearMode_ManualClear);
}

bool IsEnoughPowerSupplied() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    bool isEnoughPowerSupplied;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_Server->IsEnoughPowerSupplied(&isEnoughPowerSupplied));

    return isEnoughPowerSupplied;
}

void GetBatteryChargeInfoEvent(nn::os::SystemEventType* pOutSystemEvent, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    ::nn::sf::NativeHandle handle;

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Server->GetBatteryChargeInfoEvent(&handle));

    ::nn::os::AttachReadableHandleToSystemEvent(pOutSystemEvent, handle.GetOsHandle(), handle.IsManaged(), eventClearMode);

    handle.Detach();
}

void GetBatteryChargeInfoFields(BatteryChargeInfoFields* pOutFields) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Server->GetBatteryChargeInfoFields(pOutFields));
}

void GetBatteryChargeCalibratedEvent(nn::os::SystemEventType* pOutSystemEvent, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_Server);

    ::nn::sf::NativeHandle handle;

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Server->GetBatteryChargeCalibratedEvent(&handle));

    ::nn::os::AttachReadableHandleToSystemEvent(pOutSystemEvent, handle.GetOsHandle(), handle.IsManaged(), eventClearMode);

    handle.Detach();
}

}} // namespace nn::psm
