﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/psm/driver/psm_Api.h>
#include <nn/result/result_HandlingUtility.h>

#include "detail/psm_PowerSupplyManager.h"
#include "detail/psm_ErrorReporter.h"

namespace nn { namespace psm { namespace driver {

void Initialize() NN_NOEXCEPT
{
    detail::GetPowerSupplyManager().Initialize();
}

void Finalize() NN_NOEXCEPT
{
    detail::GetPowerSupplyManager().Finalize();
}

::nn::Result GetBatteryChargePercentage(int* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 GetBatteryChargePercentage(pOutValue));

    NN_RESULT_SUCCESS;
}

::nn::Result GetRawBatteryChargePercentage(double* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 GetRawBatteryChargePercentage(pOutValue));

    NN_RESULT_SUCCESS;
}

::nn::Result GetBatteryAgePercentage(double* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 GetBatteryAgePercentage(pOutValue));

    NN_RESULT_SUCCESS;
}

::nn::Result GetChargerType(ChargerType* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 GetChargerType(pOutValue));

    NN_RESULT_SUCCESS;
}

::nn::Result GetBatteryVoltageState(BatteryVoltageState* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 GetBatteryVoltageState(pOutValue));

    NN_RESULT_SUCCESS;
}

::nn::Result EnableBatteryCharging() NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().EnableBatteryCharging());

    NN_RESULT_SUCCESS;
}

::nn::Result DisableBatteryCharging() NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().DisableBatteryCharging());

    NN_RESULT_SUCCESS;
}

::nn::Result IsBatteryChargingEnabled(bool* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 IsBatteryChargingEnabled(pOutValue));

    NN_RESULT_SUCCESS;
}

::nn::Result AcquireControllerPowerSupply() NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 AcquireControllerPowerSupply());

    NN_RESULT_SUCCESS;
}

::nn::Result ReleaseControllerPowerSupply() NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 ReleaseControllerPowerSupply());

    NN_RESULT_SUCCESS;
}

::nn::Result OpenSession(Session* pOutSession) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 OpenSession(pOutSession));

    NN_RESULT_SUCCESS;
}

::nn::Result CloseSession(const Session& session) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 CloseSession(session));

    NN_RESULT_SUCCESS;
}

::nn::Result BindStateChangeEvent(
    ::nn::os::SystemEventType* pEvent,
    const Session& session) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 BindStateChangeEvent(pEvent, session));

    NN_RESULT_SUCCESS;
}

::nn::Result UnbindStateChangeEvent(const Session& session) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 UnbindStateChangeEvent(session));

    NN_RESULT_SUCCESS;
}

::nn::Result SetChargerTypeChangeEventEnabled(
    const Session& session,
    bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 SetChargerTypeChangeEventEnabled(session, isEnabled));

    NN_RESULT_SUCCESS;
}

::nn::Result SetPowerSupplyChangeEventEnabled(
    const Session& session,
    bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 SetPowerSupplyChangeEventEnabled(session, isEnabled));

    NN_RESULT_SUCCESS;
}

::nn::Result SetBatteryVoltageStateChangeEventEnabled(
    const Session& session,
    bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 SetBatteryVoltageStateChangeEventEnabled(session, isEnabled));

    NN_RESULT_SUCCESS;
}

::nn::Result EnableEnoughPowerChargeEmulation() NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 EnableEnoughPowerChargeEmulation());

    NN_RESULT_SUCCESS;
}

::nn::Result DisableEnoughPowerChargeEmulation() NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 DisableEnoughPowerChargeEmulation());

    NN_RESULT_SUCCESS;
}

::nn::Result EnableFastBatteryCharging() NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 EnableFastBatteryCharging());

    NN_RESULT_SUCCESS;
}

::nn::Result DisableFastBatteryCharging() NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 DisableFastBatteryCharging());

    NN_RESULT_SUCCESS;
}

::nn::Result IsEnoughPowerSupplied(bool* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_RESULT_DO(detail::GetPowerSupplyManager().
                 IsEnoughPowerSupplied(pOutValue));

    NN_RESULT_SUCCESS;
}

::nn::Result GetPmModuleForTest(::nn::psc::PmModule** pOutPmModulePointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutPmModulePointer);

    NN_RESULT_DO(detail::GetPowerSupplyManager().GetPmModuleForTest(pOutPmModulePointer));

    NN_RESULT_SUCCESS;
}

::nn::Result GetPscEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(detail::GetPowerSupplyManager().GetPscEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result GetFuelGaugeEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(detail::GetPowerSupplyManager().GetFuelGaugeEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result GetFuelGaugeTimerEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(detail::GetPowerSupplyManager().GetFuelGaugeTimerEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result GetUsbPdEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(detail::GetPowerSupplyManager().GetUsbPdEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result GetUsbPmEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(detail::GetPowerSupplyManager().GetUsbPmEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result GetEnoughPowerChargeEmulationEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(detail::GetPowerSupplyManager().GetEnoughPowerChargeEmulationEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result GetBatteryChargeInfoEvent(::nn::os::SystemEventType** pOutSystemEventPointer) NN_NOEXCEPT
{
    detail::GetErrorReporter().GetBatteryChargeInfoEvent(pOutSystemEventPointer);

    NN_RESULT_SUCCESS;
}

::nn::Result GetBatteryChargeInfoFields(BatteryChargeInfoFields* pOutFields) NN_NOEXCEPT
{
    detail::GetErrorReporter().GetBatteryChargeInfoFields(pOutFields);

    NN_RESULT_SUCCESS;
}

::nn::Result GetBatteryChargeCalibratedEvent(::nn::os::SystemEventType** pOutSystemEventPointer) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetPowerSupplyManager().GetBatteryChargeCalibratedEvent(pOutSystemEventPointer));

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::psm::driver
