﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>

#include <nn/psm/detail/psm_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/factory/settings_Usb.h>

#include "psm_SettingsHolder-spec.nx.h"

namespace nn { namespace psm { namespace driver { namespace detail {

SettingsHolder::SettingsHolder() NN_NOEXCEPT
    : m_EvaluationLogEnabled(false)
    , m_ChargeCurrentLimitMilliAmpere(2048)
    , m_UsbPowerManagementSupport(true)
    , m_HasBattery(true)
    , m_EnableChargerDriver(true)
    , m_EnableFuelGaugeDriver(true)
    , m_EnableSupplyRouteDriver(true)
    , m_EnableVdd50B(true)
    , m_UsbTypeCPowerSourceCircuitVersion(0)
{
    // 何も処理をしない。
}

void SettingsHolder::LoadSettings() NN_NOEXCEPT
{
    m_FirmwareDebugSettingsAccessor.ReadSetting(&m_EvaluationLogEnabled, "evaluation_log_enabled");
    m_FirmwareDebugSettingsAccessor.ReadSetting(&m_ChargeCurrentLimitMilliAmpere, "charge_current_limit_milli_ampere");
    m_FirmwareDebugSettingsAccessor.ReadSetting(&m_UsbPowerManagementSupport, "usb_power_management_support");
    m_FirmwareDebugSettingsAccessor.ReadSetting(&m_HasBattery, "has_battery", "platformconfig");
    m_FirmwareDebugSettingsAccessor.ReadSetting(&m_EnableChargerDriver, "enable_charger_driver");
    m_FirmwareDebugSettingsAccessor.ReadSetting(&m_EnableFuelGaugeDriver, "enable_fuel_gauge_driver");
    m_FirmwareDebugSettingsAccessor.ReadSetting(&m_EnableSupplyRouteDriver, "enable_supply_route_driver");
    m_FirmwareDebugSettingsAccessor.ReadSetting(&m_EnableVdd50B, "enable_vdd50b");

    int maximumUsbTypeCPowerSourceCircuitVersion = 0;
    m_FirmwareDebugSettingsAccessor.ReadSetting(&maximumUsbTypeCPowerSourceCircuitVersion, "maximum_usb_type_c_power_source_circuit_version");

    NN_DETAIL_PSM_TRACE("evaluation_log_enabled : %s\n", m_EvaluationLogEnabled ? "true" : "false");
    NN_DETAIL_PSM_TRACE("charge_current_limit_milli_ampere : %d\n", m_ChargeCurrentLimitMilliAmpere);
    NN_DETAIL_PSM_TRACE("usb_power_management_support : %s\n", m_UsbPowerManagementSupport ? "true" : "false");
    NN_DETAIL_PSM_TRACE("has_battery : %s\n", m_HasBattery ? "true" : "false");
    NN_DETAIL_PSM_TRACE("enable_charger_driver : %s\n", m_EnableChargerDriver ? "true" : "false");
    NN_DETAIL_PSM_TRACE("enable_fuel_gauge_driver : %s\n", m_EnableFuelGaugeDriver ? "true" : "false");
    NN_DETAIL_PSM_TRACE("enable_supply_route_driver : %s\n", m_EnableSupplyRouteDriver ? "true" : "false");
    NN_DETAIL_PSM_TRACE("enable_vdd50b : %s\n", m_EnableVdd50B ? "true" : "false");
    NN_DETAIL_PSM_TRACE("maximum_usb_type_c_power_source_circuit_version : %d\n", maximumUsbTypeCPowerSourceCircuitVersion);

    ::nn::Bit8 usbTypeCPowerSourceCircuitVersion;
    auto getterResult = ::nn::settings::factory::GetUsbTypeCPowerSourceCircuitVersion(&usbTypeCPowerSourceCircuitVersion);
    if ( getterResult.IsFailure() && !::nn::settings::factory::ResultCalibrationDataCrcError::Includes(getterResult) )
    {
#if (defined NN_BUILD_CONFIG_HARDWARE_NX)
        // HARDWARE_NX の場合 CRC エラー以外であればここで ABORT する
        NN_ABORT_UNLESS_RESULT_SUCCESS(getterResult);
#else
        // HARDWARE_NX ではない Windows テスト環境などの場合警告のみを表示する
        NN_DETAIL_PSM_WARN_UNLESS_RESULT_SUCCESS(getterResult);
#endif
    }

    // CRC エラーの場合、当該フィールドが存在しないフォーマットの較正情報とみなし 0 とする
    m_UsbTypeCPowerSourceCircuitVersion = getterResult.IsFailure() ? 0 : usbTypeCPowerSourceCircuitVersion;

    NN_ABORT_UNLESS(m_UsbTypeCPowerSourceCircuitVersion <= maximumUsbTypeCPowerSourceCircuitVersion);
    NN_ABORT_UNLESS(m_UsbTypeCPowerSourceCircuitVersion >= 0);

    NN_DETAIL_PSM_TRACE("usb_type_c_power_source_circuit_version : %d\n", m_UsbTypeCPowerSourceCircuitVersion);
}

bool SettingsHolder::IsEvaluationLogEnabled() const NN_NOEXCEPT
{
    return m_EvaluationLogEnabled;
}

int SettingsHolder::GetChargeCurrentLimitMilliAmpere() const NN_NOEXCEPT
{
    return m_ChargeCurrentLimitMilliAmpere;
}

bool SettingsHolder::IsUsbPowerManagementSupported() const NN_NOEXCEPT
{
    return m_UsbPowerManagementSupport;
}

bool SettingsHolder::HasBattery() const NN_NOEXCEPT
{
    return m_HasBattery;
}

bool SettingsHolder::IsChargerDriverEnabled() const NN_NOEXCEPT
{
    return m_EnableChargerDriver;
}

bool SettingsHolder::IsFuelGaugeDriverEnabled() const NN_NOEXCEPT
{
    return m_EnableFuelGaugeDriver;
}

bool SettingsHolder::IsSupplyRouteDriverEnabled() const NN_NOEXCEPT
{
    return m_EnableSupplyRouteDriver;
}

bool SettingsHolder::IsVdd50BEnabled() const NN_NOEXCEPT
{
    return m_EnableVdd50B;
}

int SettingsHolder::GetUsbTypeCPowerSourceCircuitVersion() const NN_NOEXCEPT
{
    return m_UsbTypeCPowerSourceCircuitVersion;
}

}}}} // namespace nn::psm::driver::detail
