﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/psm/driver/psm_Types.h>
#include <nn/util/util_IntrusiveList.h>

#include "psm_EventHolder.h"

namespace nn { namespace psm { namespace driver { namespace detail {

class SessionManager final
{
    NN_DISALLOW_COPY(SessionManager);
    NN_DISALLOW_MOVE(SessionManager);

public:
    // セッション数を増やすためには PsmServer::m_HeapBuffer のサイズ変更が必要
    static const int SessionCountMax = 7;

    enum EventType
    {
        EventType_ChargerTypeChange,
        EventType_PowerSupplyChange,
        EventType_BatteryVoltageStateChange,
        EventType_Count,
    };

public:
    SessionManager() NN_NOEXCEPT
        : m_Mutex(false)
    {
        for (auto i = 0; i < SessionCountMax; ++i)
        {
            m_IsSessionOpen[i] = false;
        }
    }

    ~SessionManager() NN_NOEXCEPT;

    //! 初期化します。
    void Initialize() NN_NOEXCEPT;

    //! 終了処理をします。
    void Finalize() NN_NOEXCEPT;

    //! セッションをオープンします。
    ::nn::Result OpenSession(Session* pOutSession) NN_NOEXCEPT;

    //! セッションをクローズします。
    ::nn::Result CloseSession(const Session& session) NN_NOEXCEPT;

    //! 充電器の変化によるイベントのシグナルを有効化するかどうかを設定します。
    ::nn::Result SetChargerTypeChangeEventEnabled(
         const Session& session,
         bool isEnabled) NN_NOEXCEPT;

    //! 給電能力の変化によるイベントのシグナルを有効化するかどうかを設定します。
    ::nn::Result SetPowerSupplyChangeEventEnabled(
         const Session& session,
         bool isEnabled) NN_NOEXCEPT;

    //! 電池電圧の状態の変化によるイベントのシグナルを有効化するかどうかを設定します。
    ::nn::Result SetBatteryVoltageStateChangeEventEnabled(
         const Session& session,
         bool isEnabled) NN_NOEXCEPT;

    //! 本体に接続された充電器や電池電圧レベルの状態の変化とイベントを紐付けます。
    ::nn::Result BindStateChangeEvent(
        ::nn::os::SystemEventType* pEvent,
        const Session& session) NN_NOEXCEPT;

    //! 本体に接続された充電器や電池電圧レベルの状態の変化とイベントの紐付けを解除します。
    ::nn::Result UnbindStateChangeEvent(
        const Session& session) NN_NOEXCEPT;

    //! すべての紐付けされたイベントをシグナルします。
    void SignalAllStateChangeEvents() NN_NOEXCEPT;

    //! 指定のイベントを有効化しているセッションに対応するイベントをシグナルします。
    void SignalStateChangeEvents(EventType event) NN_NOEXCEPT;

private:
    typedef nn::util::IntrusiveList<
        EventHolder,
        nn::util::IntrusiveListMemberNodeTraits<
            EventHolder,
            &EventHolder::m_Node>> BoundEventHolderList;

private:
    //! セッションがオープン状態かどうか
    bool m_IsSessionOpen[SessionCountMax];

    //! イベントをシグナルするかどうか管理するテーブル
    bool m_IsEventEnabled[SessionCountMax][EventType_Count];

    //! 排他制御
    nn::os::Mutex m_Mutex;

    //! 紐付け済みセッションを管理するリスト
    BoundEventHolderList m_StateChangeEventList;

    //! イベントホルダー
    EventHolder m_EventHolders[SessionCountMax];
};

}}}} // namespace nn::psm::driver::detail
