﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/spl/spl_Api.h>

#include "psm_PowerSupplyManager.h"
#include "psm_ChargeArbiter.h"
#include "psm_IChargerDriver.h"
#include "psm_IFuelGaugeDriver.h"
#include "psm_ISupplyRouteDriver.h"
#include "psm_ModuleHolder.h"

namespace nn { namespace psm { namespace driver { namespace detail {

PowerSupplyManager::PowerSupplyManager() NN_NOEXCEPT
    : m_ChargerDriverStorage()
    , m_pChargerDriver(nullptr)
    , m_FuelGaugeDriverStorage()
    , m_pFuelGaugeDriver(nullptr)
    , m_SupplyRouteDriverStorage()
    , m_pSupplyRouteDriver(nullptr)
    , m_ChargeArbiter()
    , m_EventMonitor()
    , m_EventMonitorForUsbPowerDelivery()
    , m_ModuleHolder()
    , m_SettingsHolder()
{
    // 何もしない
}

PowerSupplyManager::~PowerSupplyManager() NN_NOEXCEPT
{
    // 何もしない
}

void PowerSupplyManager::Initialize() NN_NOEXCEPT
{
    // HiZ モード通知用
    ::nn::spl::Initialize();

    m_SettingsHolder.LoadSettings();

    if ( m_SettingsHolder.IsChargerDriverEnabled() )
    {
        m_pChargerDriver = new (&(m_ChargerDriverStorage.driver)) ChargerDriver();
    }
    else
    {
        m_pChargerDriver = new (&(m_ChargerDriverStorage.stub)) ChargerDriverStub();
    }

    if ( m_SettingsHolder.IsFuelGaugeDriverEnabled() )
    {
        m_pFuelGaugeDriver = new (&(m_FuelGaugeDriverStorage.driver)) FuelGaugeDriver();
    }
    else
    {
        m_pFuelGaugeDriver = new (&(m_FuelGaugeDriverStorage.stub)) FuelGaugeDriverStub();
    }

    if ( m_SettingsHolder.IsSupplyRouteDriverEnabled() )
    {
        m_pSupplyRouteDriver = new (&(m_SupplyRouteDriverStorage.driver)) SupplyRouteDriver();
    }
    else
    {
        m_pSupplyRouteDriver = new (&(m_SupplyRouteDriverStorage.stub)) SupplyRouteDriverStub();
    }

    m_pChargerDriver->Initialize();
    m_pFuelGaugeDriver->Initialize();
    m_ChargeArbiter.Initialize(m_pChargerDriver, &m_SettingsHolder);

    m_EventMonitor.Initialize(
        NN_SYSTEM_THREAD_PRIORITY(psm, BatteryEventMonitor),
        NN_SYSTEM_THREAD_NAME(psm, BatteryEventMonitor),
        s_EventMonitorStack, sizeof(s_EventMonitorStack));

    m_EventMonitorForUsbPowerDelivery.Initialize(
        NN_SYSTEM_THREAD_PRIORITY(psm, EventMonitorForUsbPowerDelivery),
        NN_SYSTEM_THREAD_NAME(psm, EventMonitorForUsbPowerDelivery),
        s_EventMonitorForUsbPowerDeliveryStack, sizeof(s_EventMonitorForUsbPowerDeliveryStack));

    m_ModuleHolder.Initialize(
        m_pChargerDriver,
        m_pFuelGaugeDriver,
        m_pSupplyRouteDriver,
        &m_ChargeArbiter,
        &m_SessionManager,
        &m_EventMonitor,
        &m_EventMonitorForUsbPowerDelivery,
        &m_SettingsHolder);
}

void PowerSupplyManager::Finalize() NN_NOEXCEPT
{
    m_ModuleHolder.Finalize();

    m_EventMonitorForUsbPowerDelivery.Finalize();

    m_EventMonitor.Finalize();

    m_ChargeArbiter.Finalize();
    m_pFuelGaugeDriver->Finalize();
    m_pChargerDriver->Finalize();

    m_pSupplyRouteDriver->~ISupplyRouteDriver();
    m_pSupplyRouteDriver = nullptr;

    m_pFuelGaugeDriver->~IFuelGaugeDriver();
    m_pFuelGaugeDriver = nullptr;

    m_pChargerDriver->~IChargerDriver();
    m_pChargerDriver = nullptr;

    ::nn::spl::Finalize();
}

::nn::Result PowerSupplyManager::GetBatteryChargePercentage(
    int* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_RESULT_DO(m_ModuleHolder.GetBatteryChargePercentage(pOutValue));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::GetRawBatteryChargePercentage(
    double* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_RESULT_DO(m_ModuleHolder.GetRawBatteryChargePercentage(pOutValue));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::GetBatteryAgePercentage(
    double* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_RESULT_DO(m_ModuleHolder.GetBatteryAgePercentage(pOutValue));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::GetChargerType(
    ChargerType* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_RESULT_DO(m_ModuleHolder.GetChargerType(pOutValue));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::GetBatteryVoltageState(
    BatteryVoltageState* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_RESULT_DO(m_ModuleHolder.GetBatteryVoltageState(pOutValue));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::EnableBatteryCharging() NN_NOEXCEPT
{
    NN_RESULT_DO(m_ChargeArbiter.StartChargeRequest());

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::DisableBatteryCharging() NN_NOEXCEPT
{
    NN_RESULT_DO(m_ChargeArbiter.StopChargeRequest());

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::IsBatteryChargingEnabled(
    bool* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    *pOutValue = m_ChargeArbiter.IsChargeRequested();

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::AcquireControllerPowerSupply(
    ) NN_NOEXCEPT
{
    NN_RESULT_DO(m_ModuleHolder.AcquireControllerPowerSupply());

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::ReleaseControllerPowerSupply(
    ) NN_NOEXCEPT
{
    NN_RESULT_DO(m_ModuleHolder.ReleaseControllerPowerSupply());

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::OpenSession(
    Session* pOutSession) NN_NOEXCEPT
{
    NN_RESULT_DO(m_SessionManager.OpenSession(pOutSession));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::CloseSession(
    const Session& session) NN_NOEXCEPT
{
    NN_RESULT_DO(m_SessionManager.CloseSession(session));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::BindStateChangeEvent(
    ::nn::os::SystemEventType* pEvent,
    const Session& session) NN_NOEXCEPT
{
    NN_RESULT_DO(m_SessionManager.BindStateChangeEvent(pEvent, session));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::UnbindStateChangeEvent(
    const Session& session) NN_NOEXCEPT
{
    NN_RESULT_DO(m_SessionManager.UnbindStateChangeEvent(session));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::SetChargerTypeChangeEventEnabled(
    const Session& session,
    bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(m_SessionManager.SetChargerTypeChangeEventEnabled(session, isEnabled));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::SetPowerSupplyChangeEventEnabled(
    const Session& session,
    bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(m_SessionManager.SetPowerSupplyChangeEventEnabled(session, isEnabled));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::SetBatteryVoltageStateChangeEventEnabled(
    const Session& session,
    bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(m_SessionManager.SetBatteryVoltageStateChangeEventEnabled(session, isEnabled));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::EnableEnoughPowerChargeEmulation(
    ) NN_NOEXCEPT
{
    NN_RESULT_DO(m_ModuleHolder.EnableEnoughPowerChargeEmulation());

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::DisableEnoughPowerChargeEmulation(
    ) NN_NOEXCEPT
{
    NN_RESULT_DO(m_ModuleHolder.DisableEnoughPowerChargeEmulation());

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::EnableFastBatteryCharging() NN_NOEXCEPT
{
    NN_RESULT_DO(m_ChargeArbiter.EnableFastBatteryCharging());

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::DisableFastBatteryCharging() NN_NOEXCEPT
{
    NN_RESULT_DO(m_ChargeArbiter.DisableFastBatteryCharging());

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::IsEnoughPowerSupplied(
    bool* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_RESULT_DO(m_ModuleHolder.IsEnoughPowerSupplied(pOutValue));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::GetPmModuleForTest(::nn::psc::PmModule** pOutPmModulePointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutPmModulePointer);

    NN_RESULT_DO(m_ModuleHolder.GetPmModuleForTest(pOutPmModulePointer));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::GetBatteryChargeCalibratedEvent(::nn::os::SystemEventType** pOutSystemEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutSystemEventPointer);

    NN_RESULT_DO(m_ModuleHolder.GetBatteryChargeCalibratedEvent(pOutSystemEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::GetPscEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(m_ModuleHolder.GetPscEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::GetFuelGaugeEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(m_ModuleHolder.GetFuelGaugeEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::GetFuelGaugeTimerEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(m_ModuleHolder.GetFuelGaugeTimerEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::GetUsbPdEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(m_ModuleHolder.GetUsbPdEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::GetUsbPmEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(m_ModuleHolder.GetUsbPmEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result PowerSupplyManager::GetEnoughPowerChargeEmulationEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(m_ModuleHolder.GetEnoughPowerChargeEmulationEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

NN_OS_ALIGNAS_THREAD_STACK uint8_t PowerSupplyManager::s_EventMonitorStack[EventMonitorStackSize];

NN_OS_ALIGNAS_THREAD_STACK uint8_t PowerSupplyManager::s_EventMonitorForUsbPowerDeliveryStack[EventMonitorStackSize];

PowerSupplyManager& GetPowerSupplyManager() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(PowerSupplyManager, s_PowerSupplyManager);

    return s_PowerSupplyManager;
}

}}}} // namespace nn::psm::driver::detail
