﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>

#include <nn/ovln/format/ovln_PowerManagementMessage.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/psm/detail/psm_Log.h>
#include <nn/psm/psm_System.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "psm_OverlayNotificationSender.h"

namespace nn { namespace psm { namespace driver { namespace detail {

OverlayNotificationSender::OverlayNotificationSender() NN_NOEXCEPT
    : m_pBatteryState()
    , m_OverlaySenderForChagerStateChange()
    , m_OverlaySenderForLowBattery()
{
    // 何もしない
}

OverlayNotificationSender::~OverlayNotificationSender() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result OverlayNotificationSender::Initialize(
    IBatteryState* pBatteryState) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pBatteryState);

    m_pBatteryState = pBatteryState;

    auto needsRollback = true;

    NN_RESULT_DO(
        ::nn::ovln::InitializeSenderLibraryForOverlay());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            ::nn::ovln::FinalizeSenderLibraryForOverlay();
        }
    };

    NN_RESULT_DO(::nn::ovln::InitializeSenderForValue(&m_OverlaySenderForChagerStateChange));
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            ::nn::ovln::FinalizeSender(&m_OverlaySenderForChagerStateChange);
        }
    };

    NN_RESULT_DO(::nn::ovln::InitializeSenderForValue(&m_OverlaySenderForLowBattery));
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            ::nn::ovln::FinalizeSender(&m_OverlaySenderForLowBattery);
        }
    };

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

void OverlayNotificationSender::Finalize() NN_NOEXCEPT
{
    ::nn::ovln::FinalizeSender(&m_OverlaySenderForChagerStateChange);
    ::nn::ovln::FinalizeSender(&m_OverlaySenderForLowBattery);
    ::nn::ovln::FinalizeSenderLibraryForOverlay();
}

::nn::Result OverlayNotificationSender::NotifyChargerStateChange() NN_NOEXCEPT
{
    NN_RESULT_DO(Notify(&m_OverlaySenderForChagerStateChange, ::nn::ovln::format::PowerManagementChargerStateChangeDataTag));

    NN_RESULT_SUCCESS;
}

::nn::Result OverlayNotificationSender::NotifyLowBattery() NN_NOEXCEPT
{
    NN_RESULT_DO(Notify(&m_OverlaySenderForLowBattery, ::nn::ovln::format::PowerManagementLowBatteryDataTag));

    NN_RESULT_SUCCESS;
}

::nn::Result OverlayNotificationSender::Notify(::nn::ovln::SenderForOverlayType* pSender, uint32_t tag) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBatteryState);

    auto batteryCharge = m_pBatteryState->GetBatteryChargePercentage();
    auto psmChargerType = m_pBatteryState->GetChargerType();

    ::nn::ovln::format::BatteryStateChangeData data = {};

    data.batteryChargePercentage = static_cast<int32_t>(batteryCharge);

    auto chargerType = ::nn::ovln::format::ChargerType_Unconnected;
    switch (psmChargerType)
    {
    case ChargerType_Unconnected:
        chargerType = ::nn::ovln::format::ChargerType_Unconnected;
        break;
    case ChargerType_EnoughPower:
        chargerType = ::nn::ovln::format::ChargerType_EnoughPower;
        break;
    case ChargerType_LowPower:
        chargerType = ::nn::ovln::format::ChargerType_LowPower;
        break;
    case ChargerType_NotSupported:
        chargerType = ::nn::ovln::format::ChargerType_NotSupported;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    data.chargerType = static_cast<int32_t>(chargerType);

    ::nn::ovln::Message message = {};
    message.tag = tag;
    message.dataSize = sizeof(data);
    ::std::memcpy(&message.data, &data, sizeof(data));
    ::nn::ovln::Send(pSender, message);

    NN_DETAIL_PSM_TRACE("Send message to overlay.\n");

    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::psm::driver::detail
