﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>

#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os.h>

namespace nn { namespace psm { namespace driver { namespace detail {

class IChargerDriver
{
public:
    enum class FastChargeCurrentLimitOption
    {
        None,
        Current20Percent,
    };

    enum class Status
    {
        NotCharging,
        PreCharge,
        FastCharge,
        ChargeTermination,
    };

    struct FaultStatus
    {
        bool isBoostFaultGenerated;

        void Clear() NN_NOEXCEPT
        {
            isBoostFaultGenerated = false;
        }
    };

public:
    IChargerDriver() NN_NOEXCEPT {}

    virtual ~IChargerDriver() NN_NOEXCEPT {}

    //! チャージャを初期化します。
    virtual void Initialize() NN_NOEXCEPT = 0;

    //! チャージャの終了処理をします。
    virtual void Finalize() NN_NOEXCEPT = 0;

    //! 充電を有効にします。
    virtual ::nn::Result EnableBatteryCharging() NN_NOEXCEPT = 0;

    //! OTG モードを有効にします。
    virtual ::nn::Result EnableOnTheGo() NN_NOEXCEPT = 0;

    //! HiZ モードを開始します。
    virtual ::nn::Result EnterHiZMode() NN_NOEXCEPT = 0;

    //! HiZ モードを終了します。
    virtual ::nn::Result ExitHiZMode() NN_NOEXCEPT = 0;

    //! 入力電圧の下限を設定します。
    virtual ::nn::Result SetInputVoltageLimitMilliVolt(int inputVoltageLimitMilliVolt) NN_NOEXCEPT = 0;

    //! 入力電流を制限します。
    virtual ::nn::Result SetInputCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT = 0;

    //! 高速充電の入力電流を制限します。
    virtual ::nn::Result SetFastChargeCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT = 0;

    //! 高速充電の入力電流を制限します。
    virtual ::nn::Result SetFastChargeCurrentLimit(int limitMilliAmpere, FastChargeCurrentLimitOption fastChargeCurrentLimitOption) NN_NOEXCEPT = 0;

    //! ブーストモード時の出力電流を制限します。
    virtual ::nn::Result SetBoostModeCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT = 0;

    //! 充電状況を取得します。
    virtual ::nn::Result GetStatus(Status* pStatus) NN_NOEXCEPT = 0;

    //! 充電電圧を制限します。
    virtual ::nn::Result SetChargeVoltageLimit(int limitMilliVolt) NN_NOEXCEPT = 0;

    //! フォルトステータスを取得します。
    virtual ::nn::Result GetFaultStatus(FaultStatus* pFaultStatus) NN_NOEXCEPT = 0;

    //! ウォッチドッグの満了期間を設定します。
    virtual ::nn::Result SetWatchdogTimerPeriod(::nn::TimeSpan period) NN_NOEXCEPT = 0;

    //! ウォッチドッグを有効にします。
    virtual ::nn::Result EnableWatchdogTimer() NN_NOEXCEPT = 0;

    //! ウォッチドッグを無効にします。
    virtual ::nn::Result DisableWatchdogTimer() NN_NOEXCEPT = 0;

    //! ウォッチドッグをリセットします。
    virtual ::nn::Result ResetWatchdogTimer() NN_NOEXCEPT = 0;

    //! ウォッチドッグが有効かどうか取得します。
    virtual bool IsWatchdogTimerEnabled() NN_NOEXCEPT = 0;

    //! 割込みイベントを取得します。
    virtual ::nn::Result BindInterrupt(::nn::os::SystemEventType* pOutInterruptEvent) NN_NOEXCEPT = 0;

    //! 割込みイベントをクリアします。
    virtual ::nn::Result ClearInterrupt(::nn::os::SystemEventType* pInterruptEvent) NN_NOEXCEPT = 0;

    //! ハードウェア上での充電可否を設定します。
    virtual ::nn::Result SetChargeEnable(bool enabled) NN_NOEXCEPT = 0;

    //! 給電能力が十分な状態であるように振舞うかどうかを設定します。
    virtual ::nn::Result SetEnoughPowerChargeEmulation(bool isEnabled) NN_NOEXCEPT = 0;
};

}}}} // namespace nn::psm::driver::detail
